#ifndef KID_LED_EFFECTS_H
#define KID_LED_EFFECTS_H

template <typename T>
inline T clamp(T value, T low, T high) {
    return (value < low) ? low : ((value > high) ? high : value);
}
#include <Adafruit_NeoPixel.h>
#include <Arduino.h>

/**
 * Базовый абстрактный класс для эффектов освещения.
 * Определяет интерфейс для всех эффектов (методы `update`, `setSpeed`, `setBrightness` и `reset`).
 */
class LedEffect {
  public:
    explicit LedEffect(Adafruit_NeoPixel& strip) : strip(strip) {}
    virtual ~LedEffect() = default;  // Виртуальный деструктор

    /**
     * Метод обновления эффекта. Должен вызываться в `loop()`.
     * @param value Дополнительный параметр управления эффектом.
     */
    virtual void update(float value) = 0;

    /**
     * Сброс эффекта: очистка всех пикселей и обновление ленты.
     */
    inline void reset() {
        strip.clear();
        strip.show();
    }

    /**
     * Устанавливает скорость эффекта в пределах допустимого диапазона.
     * @param speed Скорость эффекта (ограничивается от `SPEED_MIN` до `SPEED_MAX`).
     */
    inline void setSpeed(float speed) {
        this->speed = clamp(speed, SPEED_MIN, SPEED_MAX);
    }

    /**
     * Устанавливает яркость эффекта в пределах допустимого диапазона.
     * @param brightness Яркость эффекта (ограничивается от `BRIGHTNESS_MIN` до `BRIGHTNESS_MAX`).
     */
    inline void setBrightness(float brightness) {
        this->brightness = clamp(brightness, BRIGHTNESS_MIN, BRIGHTNESS_MAX);
    }

  protected:
    Adafruit_NeoPixel& strip; // Ссылка на объект ленты

    // Минимальное и максимальное значение скорости эффекта
    static constexpr float SPEED_MIN = 0.1f;
    static constexpr float SPEED_MAX = 5.0f;

    // Минимальное и максимальное значение яркости
    static constexpr float BRIGHTNESS_MIN = 0.0f;
    static constexpr float BRIGHTNESS_MAX = 1.0f;

    float speed = 1.0f;      // Текущая скорость эффекта (по умолчанию 1.0)
    float brightness = 1.0f; // Текущая яркость эффекта (по умолчанию 100%)
};

/**
 * Основной класс для управления эффектами на ленте Adafruit_NeoPixel.
 * Позволяет выбирать и переключать эффекты.
 */
class KidLedEffects {
  public:
    /**
     * Конструктор класса KidLedEffects.
     * @param strip Ссылка на объект `Adafruit_NeoPixel`.
     * @param numPixels Количество светодиодов в ленте.
     */
    explicit KidLedEffects(Adafruit_NeoPixel& strip, uint16_t numPixels);

    /**
     * Эффект "Огонь" (имитация пламени).
     */
    class FireEffect : public LedEffect {
      public:
        explicit FireEffect(Adafruit_NeoPixel& strip, uint16_t numPixels);
        void update(float value) override;

      private:
        uint16_t numPixels;
        unsigned long lastUpdate = 0;
    };

    /**
     * Эффект "Вода" (волновое движение, имитация воды).
     */
    class WaterEffect : public LedEffect {
      public:
        explicit WaterEffect(Adafruit_NeoPixel& strip, uint16_t numPixels);
        void update(float value) override;

      private:
        uint16_t numPixels;
        float offset = 0; // Смещение волны
    };

    /**
     * Эффект "Радуга" (плавное изменение цветов).
     */
    class RainbowEffect : public LedEffect {
      public:
        explicit RainbowEffect(Adafruit_NeoPixel& strip, uint16_t numPixels);
        void update(float value) override;

      private:
        uint16_t numPixels;
        float hue = 0; // Текущий угол оттенка (HSV)
    };

    /**
     * Эффект "Матрица" (имитация бегущих зеленых символов).
     */
    class MatrixEffect : public LedEffect {
      public:
        explicit MatrixEffect(Adafruit_NeoPixel& strip, uint16_t numPixels);
        void update(float value) override;

      private:
        uint16_t numPixels;
        unsigned long lastUpdate = 0;
        int currentPosition = 0; // Текущая позиция падающей точки
    };

    /**
     * Устанавливает текущий эффект.
     * @param effect Указатель на объект эффекта.
     */
    void setEffect(LedEffect* effect);

    /**
     * Вызывает обновление текущего эффекта.
     * @param value Дополнительный параметр (например, динамическое управление интенсивностью).
     */
    void process(float value);

    FireEffect fire;      // Объект эффекта "Огонь"
    WaterEffect water;    // Объект эффекта "Вода"
    RainbowEffect rainbow; // Объект эффекта "Радуга"
    MatrixEffect matrix;  // Объект эффекта "Матрица"

  private:
    Adafruit_NeoPixel& strip; // Ссылка на объект ленты
    uint16_t numPixels;       // Количество пикселей в ленте
    LedEffect* currentEffect = nullptr; // Указатель на текущий эффект
};

#endif // KID_LED_EFFECTS_H
