#include "kid_LED_Effects.h"

KidLedEffects::KidLedEffects(Adafruit_NeoPixel& strip, uint16_t numPixels)
    : strip(strip), numPixels(numPixels),
      fire(strip, numPixels),
      water(strip, numPixels),
      rainbow(strip, numPixels),
      matrix(strip, numPixels)
{
    currentEffect = &fire;
}

void KidLedEffects::setEffect(LedEffect* effect) {
    if (currentEffect != effect) {
        currentEffect = effect;
        currentEffect->reset();
    }
}

void KidLedEffects::process(float value) {
    if (currentEffect) {
        currentEffect->update(value);
    }
}

// === FireEffect ===
KidLedEffects::FireEffect::FireEffect(Adafruit_NeoPixel& strip, uint16_t numPixels)
    : LedEffect(strip), numPixels(numPixels) {}

void KidLedEffects::FireEffect::update(float value) {
    unsigned long now = millis();
    if (now - lastUpdate > 50 / speed) {  // Задержка обновления с учетом скорости

        for (int i = 0; i < numPixels; i++) {
            // Случайная яркость для огня с учетом яркости и изменения от значения
            uint8_t heat = static_cast<uint8_t>(random(180, 255) * value * brightness);

            // Используем красный цвет для пламени и желто-оранжевый оттенок для огня
            // Зеленый компонент небольшой, чтобы был настоящий цвет пламени
            uint8_t red = heat;
            uint8_t green = heat / 3; // Желтоватый цвет
            uint8_t blue = 0;  // Синий не используется в огне

            // Устанавливаем цвет пикселя
            strip.setPixelColor(i, strip.Color(red, green, blue));
        }

        // Показываем изменения на полосе
        strip.show();
        
        // Обновляем время для следующего обновления
        lastUpdate = now;
    }
}

// === WaterEffect ===
KidLedEffects::WaterEffect::WaterEffect(Adafruit_NeoPixel& strip, uint16_t numPixels)
    : LedEffect(strip), numPixels(numPixels) {}

void KidLedEffects::WaterEffect::update(float value) {
    // Регулируем скорость волны
    offset += 0.05 * speed * value;  // Уменьшена скорость для плавности

    for (int i = 0; i < numPixels; i++) {
        // Используем синус для волны, добавляем отражение
        float wave = 0.5 * (1 + sin(i * 0.3 + offset));  // Волна слева направо
        float reflectedWave = 0.5 * (1 + sin((numPixels - i) * 0.3 + offset));  // Волна справа налево

        // Смешиваем волны с отражением
        float finalWave = (wave + reflectedWave) / 2;

        // Устанавливаем минимальное значение для волны
        finalWave = max(finalWave, 0.1);  // Волна не гаснет полностью, минимум 10% от максимальной яркости

        // Расчет яркости для волны (синий цвет)
        uint8_t blue = static_cast<uint8_t>(255 * finalWave * brightness);

        // Устанавливаем синий цвет для эффекта воды
        strip.setPixelColor(i, strip.Color(0, 0, blue));
    }

    strip.show();
}

// === RainbowEffect ===
KidLedEffects::RainbowEffect::RainbowEffect(Adafruit_NeoPixel& strip, uint16_t numPixels)
    : LedEffect(strip), numPixels(numPixels) {}

void KidLedEffects::RainbowEffect::update(float value) {
    hue += 0.5 * speed * value;
    hue = fmod(hue, 360.0f);
    
    for (int i = 0; i < numPixels; i++) {
        float pixelHue = hue + (i * 360.0f / numPixels);
        uint16_t hue16 = static_cast<uint16_t>((pixelHue / 360.0f) * 65535);
        
        uint32_t color = strip.gamma32(strip.ColorHSV(hue16, 255, static_cast<uint8_t>(255 * brightness)));
        strip.setPixelColor(i, color);
    }
    strip.show();
}

// === MatrixEffect ===
KidLedEffects::MatrixEffect::MatrixEffect(Adafruit_NeoPixel& strip, uint16_t numPixels)
    : LedEffect(strip), numPixels(numPixels) {}

void KidLedEffects::MatrixEffect::update(float value) {
    unsigned long now = millis();
    if (now - lastUpdate > 100 / (speed * value)) {
        // Гасим предыдущие пиксели с эффектом затухания
        for (int i = 0; i < numPixels; i++) {
            uint32_t color = strip.getPixelColor(i);
            uint8_t green = (color >> 8) & 0xFF;
            green = green * 0.8; // Затухание
            strip.setPixelColor(i, strip.Color(0, green, 0));
        }

        // Добавляем новую "падающую" точку
        currentPosition = (currentPosition + 1) % numPixels;
        strip.setPixelColor(currentPosition, strip.Color(0, 255 * brightness, 0));
        
        strip.show();
        lastUpdate = now;
    }
}
