---
hide:
  - footer
---

<!--
SPDX-FileCopyrightText: 2025 KOINSLOT, Inc.

SPDX-License-Identifier: GPL-3.0-or-later
-->

## Generating images using the kywy image design tool


To be completed once the kywy image editing tool is completed

The image editing tool will allow you to generate hpp files which can be included in projects

## Including HPP Files in Your Kywy Project

Once you have generated your HPP file from the image tool, here's how to include and use it in your Kywy project:

### Step 1: Add the HPP File to Your Project

1. **Copy the generated .hpp file** to your project directory (same folder as your `.ino` file)
2. **Include it in your main sketch** at the top of your file:

```cpp
// Include Kywy library
#include "Kywy.hpp"

// Include your generated sprite files
#include "my_hero.hpp"
#include "forest_background.hpp"
#include "treasure_chest.hpp"

// Your existing includes
#include "screens.hpp"
```

### Step 2: Use the Generated Constants and Data

The HPP file will contain:
- **Data array**: `uint8_t sprite_name_data[]` - The actual image data
- **Width constant**: `#define SPRITE_NAME_WIDTH value`
- **Height constant**: `#define SPRITE_NAME_HEIGHT value`

Example of a generated HPP file:
```cpp
// Generated by Kywy HPP Generator
// Dimensions: 64x64
uint8_t my_hero_data[8192] = {
  0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
  // ... lots of hex numbers ...
};

// Bitmap Constants
#define MY_HERO_WIDTH 64
#define MY_HERO_HEIGHT 64
```

### Step 3: Draw Your Sprites in the Game

Use the `drawBitmap()` function to display your sprites:

```cpp
void loop() {
  // Clear the screen
  engine.display.clear();

  // Draw your custom hero sprite at position (50, 100)
  engine.display.drawBitmap(50, 100, MY_HERO_WIDTH, MY_HERO_HEIGHT, my_hero_data);

  // Draw other sprites
  engine.display.drawBitmap(0, 0, FOREST_BACKGROUND_WIDTH, FOREST_BACKGROUND_HEIGHT, forest_background_data);
  engine.display.drawBitmap(200, 150, TREASURE_CHEST_WIDTH, TREASURE_CHEST_HEIGHT, treasure_chest_data);

  // Update the display
  engine.display.update();

  // Your game logic here...
}
```

### Step 4: Sprite Positioning Tips

- **Screen dimensions**: Kywy display is 400x240 pixels
- **Center sprites**: Use `(KYWY_DISPLAY_WIDTH - SPRITE_WIDTH) / 2` for horizontal centering
- **Bottom alignment**: Use `KYWY_DISPLAY_HEIGHT - SPRITE_HEIGHT` for bottom positioning
- **Layering**: Draw background sprites first, then foreground sprites

```cpp
// Center a sprite horizontally
int centerX = (KYWY_DISPLAY_WIDTH - MY_HERO_WIDTH) / 2;
engine.display.drawBitmap(centerX, 100, MY_HERO_WIDTH, MY_HERO_HEIGHT, my_hero_data);

// Position at bottom of screen
int bottomY = KYWY_DISPLAY_HEIGHT - MY_HERO_HEIGHT;
engine.display.drawBitmap(50, bottomY, MY_HERO_WIDTH, MY_HERO_HEIGHT, my_hero_data);
```

### Step 5: Organizing Multiple Sprites

For games with many sprites, create helper functions:

```cpp
// Helper functions for drawing sprites
void drawHero(int x, int y) {
  engine.display.drawBitmap(x, y, MY_HERO_WIDTH, MY_HERO_HEIGHT, my_hero_data);
}

void drawTreasure(int x, int y) {
  engine.display.drawBitmap(x, y, TREASURE_CHEST_WIDTH, TREASURE_CHEST_HEIGHT, treasure_chest_data);
}

void drawBackground() {
  engine.display.drawBitmap(0, 0, FOREST_BACKGROUND_WIDTH, FOREST_BACKGROUND_HEIGHT, forest_background_data);
}

// Use in your game
void loop() {
  engine.display.clear();
  drawBackground();
  drawHero(50, 100);
  drawTreasure(200, 150);
  engine.display.update();
}
```

### Step 6: Memory Considerations

- **Smaller sprites** use less memory
- **Reuse sprites** when possible
- **Test on device** - Kywy has limited memory
- **Use appropriate sizes** - 32x32 or 64x64 pixels work well

### Troubleshooting

**Common Issues:**
- **"File not found" error**: Make sure the .hpp file is in the same directory as your .ino file
- **"Undefined reference" error**: Check that the sprite name in the HPP file matches what you're using in code
- **Sprite not displaying**: Verify the width/height constants are correct
- **Memory errors**: Try smaller sprite sizes or fewer sprites

**File Structure Example:**
```
your_project/
├── your_project.ino
├── my_hero.hpp
├── forest_background.hpp
├── treasure_chest.hpp
└── screens.hpp
```

This structure ensures all your sprite files are easily accessible and properly organized in your Kywy project.