/*
  ServoM - Advanced Servo Motor Control Library
  Created by Ivan Ivanov, December 2024
  Released under MIT License
*/

#ifndef SERVO_M_H
#define SERVO_M_H

#include "Arduino.h"

// Default constants
#define SERVOM_MIN_PULSE_WIDTH     544     // Minimum pulse width in microseconds
#define SERVOM_MAX_PULSE_WIDTH     2400    // Maximum pulse width in microseconds
#define SERVOM_DEFAULT_MIN_ANGLE   0       // Minimum angle in degrees
#define SERVOM_DEFAULT_MAX_ANGLE   180     // Maximum angle in degrees
#define SERVOM_REFRESH_INTERVAL    20      // Refresh interval in milliseconds

class ServoM {
public:
    // Constructors
    ServoM();
    
    // Basic control methods
    bool attach(uint8_t pin);
    bool attach(uint8_t pin, uint16_t minPulse, uint16_t maxPulse);
    void detach();
    
    // Movement control
    void write(uint8_t angle);
    void writeMicroseconds(uint16_t pulseWidth);
    uint8_t read() const;
    
    // Smooth movement
    void smoothMove(uint8_t targetAngle, uint16_t duration = 1000);
    void stop();
    void update();
    
    // Configuration
    void setMinPulse(uint16_t minPulse);
    void setMaxPulse(uint16_t maxPulse);
    void setLimits(uint8_t minAngle, uint8_t maxAngle);
    
    // Status
    bool isAttached() const;
    bool isMoving() const;
    uint8_t getCurrentAngle() const;
    
private:
    // Private methods
    uint16_t _angleToMicroseconds(uint8_t angle) const;
    bool _isAngleValid(uint8_t angle) const;
    void _sendPulse(uint16_t pulseWidth);
    
    // Member variables
    uint8_t _pin;
    uint8_t _currentAngle;
    uint8_t _targetAngle;
    uint8_t _startAngle;
    uint8_t _minAngle;
    uint8_t _maxAngle;
    uint16_t _minPulse;
    uint16_t _maxPulse;
    
    // Smooth movement control
    bool _isMoving;
    unsigned long _moveStartTime;
    uint16_t _moveDuration;
    
    // Internal flags
    bool _attached;
    static const uint8_t _INVALID_PIN = 255;
};

#endif // SERVO_M_H
