/*
  ServoM - Advanced Servo Motor Control Library
  Created by Ivan Ivanov, December 2024
  Released under MIT License
*/

#include "ServoM.h"

ServoM::ServoM() {
    _pin = _INVALID_PIN;
    _currentAngle = 90;
    _targetAngle = 90;
    _startAngle = 90;
    _minAngle = SERVOM_DEFAULT_MIN_ANGLE;
    _maxAngle = SERVOM_DEFAULT_MAX_ANGLE;
    _minPulse = SERVOM_MIN_PULSE_WIDTH;
    _maxPulse = SERVOM_MAX_PULSE_WIDTH;
    _isMoving = false;
    _attached = false;
    _moveDuration = 1000;
}

bool ServoM::attach(uint8_t pin) {
    return attach(pin, SERVOM_MIN_PULSE_WIDTH, SERVOM_MAX_PULSE_WIDTH);
}

bool ServoM::attach(uint8_t pin, uint16_t minPulse, uint16_t maxPulse) {
    if (_attached) {
        detach();
    }
    
    if (minPulse >= maxPulse) {
        return false;
    }
    
    _pin = pin;
    _minPulse = minPulse;
    _maxPulse = maxPulse;
    
    pinMode(_pin, OUTPUT);
    digitalWrite(_pin, LOW);
    
    _attached = true;
    _currentAngle = 90;
    _targetAngle = 90;
    _isMoving = false;
    
    // Set to initial position
    write(_currentAngle);
    
    return true;
}

void ServoM::detach() {
    if (_attached) {
        digitalWrite(_pin, LOW);
        _pin = _INVALID_PIN;
        _attached = false;
        _isMoving = false;
    }
}

void ServoM::write(uint8_t angle) {
    if (!_attached || !_isAngleValid(angle)) {
        return;
    }
    
    _currentAngle = angle;
    _isMoving = false;
    
    uint16_t pulseWidth = _angleToMicroseconds(angle);
    _sendPulse(pulseWidth);
}

void ServoM::writeMicroseconds(uint16_t pulseWidth) {
    if (!_attached) {
        return;
    }
    
    // Constrain pulse width
    if (pulseWidth < _minPulse) pulseWidth = _minPulse;
    if (pulseWidth > _maxPulse) pulseWidth = _maxPulse;
    
    // Convert pulse width to approximate angle
    _currentAngle = map(pulseWidth, _minPulse, _maxPulse, _minAngle, _maxAngle);
    _isMoving = false;
    
    _sendPulse(pulseWidth);
}

uint8_t ServoM::read() const {
    return _currentAngle;
}

void ServoM::smoothMove(uint8_t targetAngle, uint16_t duration) {
    if (!_attached || !_isAngleValid(targetAngle) || duration == 0) {
        return;
    }
    
    _startAngle = _currentAngle;
    _targetAngle = targetAngle;
    _moveStartTime = millis();
    _moveDuration = duration;
    _isMoving = true;
}

void ServoM::stop() {
    _isMoving = false;
}

void ServoM::update() {
    if (!_isMoving || !_attached) {
        return;
    }
    
    unsigned long elapsed = millis() - _moveStartTime;
    
    if (elapsed >= _moveDuration) {
        // Movement completed
        _currentAngle = _targetAngle;
        write(_currentAngle);
        _isMoving = false;
        return;
    }
    
    // Linear interpolation
    float progress = (float)elapsed / _moveDuration;
    uint8_t newAngle = _startAngle + (_targetAngle - _startAngle) * progress;
    
    if (newAngle != _currentAngle) {
        _currentAngle = newAngle;
        uint16_t pulseWidth = _angleToMicroseconds(newAngle);
        _sendPulse(pulseWidth);
    }
}

void ServoM::setMinPulse(uint16_t minPulse) {
    if (minPulse < _maxPulse) {
        _minPulse = minPulse;
    }
}

void ServoM::setMaxPulse(uint16_t maxPulse) {
    if (maxPulse > _minPulse) {
        _maxPulse = maxPulse;
    }
}

void ServoM::setLimits(uint8_t minAngle, uint8_t maxAngle) {
    if (minAngle < maxAngle && maxAngle <= 180) {
        _minAngle = minAngle;
        _maxAngle = maxAngle;
        
        // Adjust current angle if needed
        if (_currentAngle < minAngle) {
            write(minAngle);
        } else if (_currentAngle > maxAngle) {
            write(maxAngle);
        }
    }
}

bool ServoM::isAttached() const {
    return _attached;
}

bool ServoM::isMoving() const {
    return _isMoving;
}

uint8_t ServoM::getCurrentAngle() const {
    return _currentAngle;
}

// Private methods
uint16_t ServoM::_angleToMicroseconds(uint8_t angle) const {
    return map(angle, _minAngle, _maxAngle, _minPulse, _maxPulse);
}

bool ServoM::_isAngleValid(uint8_t angle) const {
    return (angle >= _minAngle && angle <= _maxAngle);
}

void ServoM::_sendPulse(uint16_t pulseWidth) {
    digitalWrite(_pin, HIGH);
    delayMicroseconds(pulseWidth);
    digitalWrite(_pin, LOW);
    delayMicroseconds(20000 - pulseWidth); // 50Hz refresh rate
}
