# Библиотека ServoM

Продвинутая библиотека для управления сервоприводами Arduino с плавными движениями и расширенными функциями.

## Особенности

- **Плавные движения**: Линейная интерполяция между углами
- **Неблокирующая работа**: Обновление сервоприводов без остановки основного кода
- **Ограничения углов**: Установка пользовательских диапазонов
- **Управление шириной импульса**: Точная настройка импульсов сервопривода
- **Несколько сервоприводов**: Одновременное управление несколькими сервоприводами
- **Защитные функции**: Проверка входных данных и граничных значений

## Установка

### Через Менеджер библиотек Arduino
1. Откройте Arduino IDE
2. Скетч → Подключить библиотеку → Управлять библиотеками
3. Найдите "ServoM"
4. Нажмите "Установить"

### Ручная установка
1. Скачайте библиотеку в формате ZIP
2. В Arduino IDE: Скетч → Подключить библиотеку → Добавить .ZIP библиотеку
3. Выберите скачанный файл

## Быстрый старт

```cpp
#include <ServoM.h>

ServoM myservo;

void setup() {
  myservo.attach(9);
  myservo.smoothMove(90, 1000); // Плавное движение до 90° за 1 секунду
}

void loop() {
  myservo.update(); // Обязательно для плавных движений
}
```

## Примеры

В папке examples находятся:
- **Basic**: Простое управление сервоприводом
- **SmoothMovement**: Примеры плавной анимации
- **MultipleServos**: Управление несколькими сервоприводами

## Справочник команд

### Основные команды

| Команда | Параметры | Описание | Возвращаемое значение |
|---------|-----------|----------|------------------------|
| **Конструктор** | - | Создает объект сервопривода | Объект ServoM |
| `attach(pin)` | `pin` - номер пина | Подключает сервопривод к пину | `true` при успехе |
| `attach(pin, minPulse, maxPulse)` | `pin`, `minPulse`, `maxPulse` | Подключение с кастомными импульсами | `true` при успехе |
| `detach()` | - | Отключает сервопривод | - |
| `write(angle)` | `angle` - угол (0-180) | Немедленное движение к углу | - |
| `writeMicroseconds(pulseWidth)` | `pulseWidth` - ширина импульса | Установка ширины импульса | - |
| `read()` | - | Чтение текущего угла | Текущий угол (0-180) |

### Команды плавного движения

| Команда | Параметры | Описание | Возвращаемое значение |
|---------|-----------|----------|------------------------|
| `smoothMove(targetAngle, duration)` | `targetAngle` - целевой угол<br>`duration` - длительность (мс) | Плавное движение к цели | - |
| `update()` | - | Обновление плавного движения (вызывать в loop) | - |
| `stop()` | - | Остановка текущего движения | - |
| `isMoving()` | - | Проверка, движется ли сервопривод | `true` если движется |

### Команды конфигурации

| Команда | Параметры | Описание | Возвращаемое значение |
|---------|-----------|----------|------------------------|
| `setLimits(minAngle, maxAngle)` | `minAngle`, `maxAngle` | Установка границ углов | - |
| `setMinPulse(minPulse)` | `minPulse` - минимальный импульс | Установка минимальной ширины импульса | - |
| `setMaxPulse(maxPulse)` | `maxPulse` - максимальный импульс | Установка максимальной ширины импульса | - |

### Команды статуса

| Команда | Параметры | Описание | Возвращаемое значение |
|---------|-----------|----------|------------------------|
| `isAttached()` | - | Проверка подключения сервопривода | `true` если подключен |
| `getCurrentAngle()` | - | Получение текущего угла | Текущий угол (0-180) |

## Примеры использования

### Базовое управление
```cpp
ServoM servo;
servo.attach(9);
servo.write(90); // Немедленно в 90°
delay(1000);
servo.write(0);  // Немедленно в 0°
```

### Плавное движение
```cpp
ServoM servo;
servo.attach(9);

// Плавное движение от 0 до 180 за 2 секунды
servo.smoothMove(0, 2000);
while(servo.isMoving()) {
  servo.update();
  delay(10);
}
```

### Работа с ограничениями
```cpp
ServoM servo;
servo.attach(9);
servo.setLimits(30, 150); // Только от 30° до 150°
servo.write(90); // OK
servo.write(20); // Игнорируется (ниже минимума)
```

### Несколько сервоприводов
```cpp
ServoM servo1, servo2, servo3;

void setup() {
  servo1.attach(9);
  servo2.attach(10);
  servo3.attach(11);
  
  // Разные скорости движения
  servo1.smoothMove(45, 1000);
  servo2.smoothMove(90, 2000);
  servo3.smoothMove(135, 1500);
}

void loop() {
  // Обновляем все сервоприводы
  servo1.update();
  servo2.update();
  servo3.update();
}
```

## Технические детали

### Временные параметры
- Минимальная ширина импульса: 544 мкс
- Максимальная ширина импульса: 2400 мкс
- Частота обновления: 50 Гц (20 мс период)
- Диапазон углов по умолчанию: 0-180°

### Ограничения
- Поддерживает стандартные 180-градусные сервоприводы
- Может работать с сервоприводами на 270° при соответствующей настройке импульсов
- Не предназначен для сервоприводов непрерывного вращения

## Часто задаваемые вопросы

**Q: Почему сервопривод дергается?**  
A: Убедитесь, что питание достаточное и стабильное. Добавьте конденсатор к питанию сервопривода.

**Q: Как использовать с сервоприводом на 270°?**  
A: Настройте импульсы: `servo.attach(pin, 500, 2500);`

**Q: Можно ли управлять сервоприводом в фоне?**  
A: Да, используйте `smoothMove()` и `update()` в основном цикле.

**Q: Сколько сервоприводов можно управлять одновременно?**  
A: Количество ограничено только количеством доступных пинов и памятью.

## Совместимость

- Arduino Uno, Nano, Mega
- ESP32, ESP8266
- STM32 (чесли PlatformIO)
- Другие AVR и ARM платы

## Лицензия

MIT License - подробности в файле LICENSE

## Поддержка

Для сообщений об ошибках и запросов функций используйте Issues на GitHub.

## Автор

Разработано для сообщества Arduino. Вклад приветствуется!

---

### Сравнение с стандартной библиотекой Servo

| Функция | Стандартная Servo | ServoM |
|---------|-------------------|--------|
| Плавные движения | ❌ Нет | ✅ Да |
| Ограничения углов | ❌ Нет | ✅ Да |
| Неблокирующая работа | ❌ Нет | ✅ Да |
| Проверка граничных значений | ❌ Нет | ✅ Да |
| Поддержка ESP32/ESP8266 | ⚠️ Ограниченная | ✅ Полная |
| Управление импульсами | ⚠️ Базовое | ✅ Расширенное |

---

**Версия:** 1.0.0  
**Последнее обновление:** Декабрь 2024  
**GitHub:** https://github.com/KITTEN2008/ServoM
