/*
 * SoftwareAssuraVisionSerial.ino - ตัวอย่างการใช้งาน AssuraVisionSerial Library กับ SoftwareSerial
 *
 * ตัวอย่างนี้แสดงวิธีการใช้งาน AssuraVisionSerial Library กับ SoftwareSerial
 * เพื่อสื่อสารกับโปรแกรม Desktop ผ่าน Serial Port อื่นที่ไม่ใช่ Hardware Serial
 * This example demonstrates how to use AssuraVisionSerial Library with SoftwareSerial
 * to communicate with Desktop application via alternative Serial Port
 *
 * Protocol: C# Compatible DataFrame
 * - Frame Format: [SOF][SOF][CMD][LEN][SEQ][PAYLOAD][CRC][EOF][EOF]
 * - CRC-8 checksum (polynomial 0x07)
 *
 * Hardware:
 * - Arduino (Uno, Nano, etc.)
 * - USB-to-Serial converter (FTDI, CH340, etc.)
 * - Connect:
 *   - Arduino Pin 10 (RX) -> TX ของ USB-to-Serial
 *   - Arduino Pin 11 (TX) -> RX ของ USB-to-Serial
 *   - GND -> GND
 *
 * Usage:
 * 1. Upload sketch to Arduino
 * 2. Open Serial Monitor (115200 baud) สำหรับดู Debug messages
 * 3. ใช้โปรแกรม Desktop เชื่อมต่อกับ SoftwareSerial port (9600 baud)
 */

#include <SoftwareSerial.h>
#include <AssuraVisionSerial.h>

// กำหนดขา RX และ TX สำหรับ SoftwareSerial
const int RX_PIN = 10;
const int TX_PIN = 11;

// สร้าง SoftwareSerial object
SoftwareSerial mySerial(RX_PIN, TX_PIN);

// สร้าง AssuraVisionSerial object โดยใช้ SoftwareSerial
AssuraVisionSerial comm(mySerial);

// ตัวแปรสำหรับ LED (ใช้ built-in LED)
const int LED_PIN = LED_BUILTIN;
bool ledState = false;

// ตัวแปรสำหรับ Heartbeat
unsigned long lastHeartbeat = 0;
const unsigned long HEARTBEAT_INTERVAL = 3000; // ส่ง Heartbeat ทุก 3 วินาที

void setup() {
    // Initialize LED pin
    pinMode(LED_PIN, OUTPUT);
    digitalWrite(LED_PIN, LOW);

    // Initialize Hardware Serial สำหรับ Debug
    Serial.begin(115200);
    Serial.println(F("\n=== SoftwareSerial Communication Example ==="));
    Serial.print(F("SoftwareSerial RX: Pin "));
    Serial.println(RX_PIN);
    Serial.print(F("SoftwareSerial TX: Pin "));
    Serial.println(TX_PIN);
    Serial.println(F("SoftwareSerial Baud: 9600"));
    Serial.println(F("Protocol: C# Compatible DataFrame"));

    // Initialize SoftwareSerial Communication
    mySerial.begin(9600); // SoftwareSerial มักใช้ Baud rate ต่ำกว่า
    comm.begin(9600);

    // ลงทะเบียน callback functions สำหรับคำสั่งต่างๆ
    comm.onCommand(CMD_PING, onPingReceived);
    comm.onCommand(CMD_HEARTBEAT, onHeartbeat);
    comm.onCommand(CMD_LED_RED, onLedRed);
    comm.onCommand(CMD_LED_GREEN, onLedGreen);
    comm.onCommand(CMD_LED_BLUE, onLedBlue);
    comm.onCommand(CMD_LED_OFF, onLedOff);
    comm.onCommand(CMD_RELAY1, onRelay1);
    comm.onCommand(CMD_RESET, onReset);
    comm.onCommand(CMD_SENSOR, onSensor);

    // รอให้ Serial พร้อม
    delay(1000);

    // ส่ง HEARTBEAT เพื่อแจ้ง Desktop ว่า Arduino พร้อมแล้ว
    comm.sendHeartbeat();

    Serial.println(F("\n[READY] Arduino ready!"));
    Serial.println(F("[READY] HEARTBEAT sent - waiting for PC discovery..."));
    Serial.println(F("[READY] Waiting for commands on SoftwareSerial...\n"));
}

void loop() {
    // อัพเดท AssuraVisionSerial (ตรวจสอบข้อมูลที่รับจาก SoftwareSerial)
    comm.update();

    // ส่ง Heartbeat เป็นระยะ
    if (millis() - lastHeartbeat >= HEARTBEAT_INTERVAL) {
        lastHeartbeat = millis();
        comm.sendHeartbeat();
        Serial.println(F("[HEARTBEAT] Sent"));
    }
}

// ===== Callback Functions =====

// Callback เมื่อได้รับคำสั่ง PING
void onPingReceived(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> PING received from PC (Discovery)"));
    // ✅ ตอบด้วย HEARTBEAT (ไม่ใช่ PING!)
    // เพราะ Arduino เป็น Slave ส่ง HEARTBEAT เพื่อบอกว่า alive
    comm.sendHeartbeat();
}

// Callback เมื่อได้รับคำสั่ง Heartbeat
void onHeartbeat(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> HEARTBEAT received"));
    // ไม่ต้องตอบกลับ (Desktop monitor อัตโนมัติ)
}

// Callback เมื่อได้รับคำสั่ง LED_RED
void onLedRed(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> LED_RED"));
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง LED_GREEN
void onLedGreen(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> LED_GREEN"));
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง LED_BLUE
void onLedBlue(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> LED_BLUE"));
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง LED_OFF
void onLedOff(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> LED_OFF"));
    ledState = false;
    digitalWrite(LED_PIN, LOW);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง RELAY1
void onRelay1(byte* payload, byte length) {
    Serial.print(F("[CMD] >>> RELAY1"));

    if (length >= 1) {
        bool relayOn = (payload[0] != 0);
        Serial.print(F(" - "));
        Serial.println(relayOn ? F("ON") : F("OFF"));
    } else {
        Serial.println();
    }

    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง RESET
void onReset(byte* payload, byte length) {
    Serial.println(F("[CMD] >>> RESET"));

    // Reset all outputs
    ledState = false;
    digitalWrite(LED_PIN, LOW);

    Serial.println(F("[SYSTEM] Reset complete"));
    // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}

// Callback เมื่อได้รับคำสั่ง SENSOR
void onSensor(byte* payload, byte length) {
    Serial.print(F("[CMD] >>> SENSOR"));

    if (length >= 1) {
        bool sensorOn = (payload[0] != 0);
        Serial.print(F(" - "));
        Serial.println(sensorOn ? F("READ") : F("OFF"));

        // ✅ ส่งข้อมูลเซ็นเซอร์กลับไป (ถ้า ON)
        // กรณีนี้ส่งข้อมูลกลับเพราะ PC ขอข้อมูล
        if (sensorOn) {
            byte sensorData[2];
            int value = analogRead(A0);
            sensorData[0] = (value >> 8) & 0xFF;
            sensorData[1] = value & 0xFF;

            comm.sendFrame(CMD_SENSOR, sensorData, sizeof(sensorData));

            Serial.print(F("[SENSOR] Value: "));
            Serial.println(value);
        }
        // ❌ ถ้า OFF ก็ไม่ต้องตอบ - Heartbeat จะบอก PC อยู่แล้ว
    } else {
        Serial.println();
        // ❌ ไม่ต้องตอบกลับ! Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
    }
}