/*
 * DebugSoftwareSerial.ino - Debug และทดสอบการสื่อสารผ่าน SoftwareSerial
 *
 * โปรแกรมนี้ช่วยในการ Debug และตรวจสอบการทำงานของ AssuraVisionSerial กับ SoftwareSerial
 * แสดงข้อมูล Debug ละเอียด รวมถึง Frames และ Statistics
 *
 * Protocol: C# Compatible DataFrame
 * - Frame Format: [SOF][SOF][CMD][LEN][SEQ][PAYLOAD][CRC][EOF][EOF]
 * - CRC-8 checksum (polynomial 0x07)
 *
 * Hardware:
 * - Arduino (Uno R4, Mega, Nano, etc.)
 * - USB-to-Serial converter
 * - Connect:
 *   - Arduino Pin 10 (RX) -> TX ของ USB-to-Serial
 *   - Arduino Pin 11 (TX) -> RX ของ USB-to-Serial
 *   - GND -> GND
 *
 * Debug Output:
 * - Hardware Serial (115200 baud) - สำหรับ Debug messages
 * - SoftwareSerial (9600 baud) - สำหรับสื่อสารกับ Desktop
 *
 * Commands:
 * ส่งคำสั่งผ่าน Hardware Serial Monitor:
 * - 't' = ทดสอบส่ง PING
 * - 'h' = ทดสอบส่ง HEARTBEAT
 * - 'l' = สลับ LED
 * - 's' = แสดง Statistics
 * - 'b' = แสดง Buffer content
 * - 'r' = Reset statistics
 * - 'c' = Clear buffer
 * - '?' = แสดง Help
 */

#include <SoftwareSerial.h>
#include <AssuraVisionSerial.h>

// กำหนดขา RX และ TX สำหรับ SoftwareSerial
const int RX_PIN = 10;
const int TX_PIN = 11;

// สร้าง SoftwareSerial object
SoftwareSerial mySerial(RX_PIN, TX_PIN);

// สร้าง AssuraVisionSerial object
AssuraVisionSerial comm(mySerial);

// ตัวแปรสำหรับ LED
const int LED_PIN = LED_BUILTIN;
bool ledState = false;

// Debug flags
bool debugFrames = true;        // แสดง frame details
bool debugCallbacks = true;     // แสดงเมื่อ callback ถูกเรียก

// Statistics
unsigned long lastStatsDisplay = 0;
const unsigned long STATS_INTERVAL = 10000; // แสดง stats ทุก 10 วินาที

// Heartbeat
unsigned long lastHeartbeat = 0;
const unsigned long HEARTBEAT_INTERVAL = 3000; // ส่ง heartbeat ทุก 3 วินาที

void setup() {
    // Initialize LED
    pinMode(LED_PIN, OUTPUT);
    digitalWrite(LED_PIN, LOW);

    // Initialize Hardware Serial สำหรับ Debug
    Serial.begin(9600);
    delay(100);

    printHeader();

    Serial.println(F("\n=== SoftwareSerial Debug Tool ==="));
    Serial.print(F("RX Pin: "));
    Serial.println(RX_PIN);
    Serial.print(F("TX Pin: "));
    Serial.println(TX_PIN);
    Serial.println(F("SoftwareSerial Baud: 9600"));
    Serial.println(F("Protocol: C# Compatible DataFrame"));
    Serial.println(F("CRC: CRC-8 (polynomial 0x07)"));

    // Initialize SoftwareSerial
    mySerial.begin(9600);
    comm.begin(9600);

    // ลงทะเบียน callbacks
    comm.onCommand(CMD_PING, onPingReceived);
    comm.onCommand(CMD_HEARTBEAT, onHeartbeat);
    comm.onCommand(CMD_LED_RED, onLedRed);
    comm.onCommand(CMD_LED_GREEN, onLedGreen);
    comm.onCommand(CMD_LED_BLUE, onLedBlue);
    comm.onCommand(CMD_LED_OFF, onLedOff);
    comm.onCommand(CMD_RELAY1, onRelay1);
    comm.onCommand(CMD_RESET, onReset);
    comm.onCommand(CMD_SENSOR, onSensor);

    // ส่ง HEARTBEAT เพื่อบอกว่าพร้อม
    delay(1000);
    Serial.println(F("\n>>> Sending HEARTBEAT (Arduino Ready)"));
    comm.sendHeartbeat();

    printHelp();
    Serial.println(F("\n[READY] Waiting for commands...\n"));
}

void loop() {
    // ตรวจสอบคำสั่งจาก Hardware Serial (Debug console)
    handleDebugCommands();

    // อัพเดท AssuraVisionSerial
    comm.update();

    // ส่ง Heartbeat เป็นระยะ
    // if (millis() - lastHeartbeat >= HEARTBEAT_INTERVAL) {
    //     lastHeartbeat = millis();
    //     comm.sendHeartbeat();
    //     Serial.println(F("[TX] >>> Heartbeat sent"));
    // }

    // แสดง Statistics เป็นระยะ
    if (millis() - lastStatsDisplay >= STATS_INTERVAL) {
        lastStatsDisplay = millis();
        printStatistics();
    }
}

// ===== Callback Functions =====

void onPingReceived(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< PING received from PC (Discovery)"));
        Serial.print(F("  Payload length: "));
        Serial.println(length);
    }
    // ✅ ตอบด้วย HEARTBEAT (ไม่ใช่ PING!)
    comm.sendHeartbeat();
    Serial.println(F("[TX] >>> HEARTBEAT sent"));
}

void onHeartbeat(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< HEARTBEAT received"));
    }
}

void onLedRed(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< LED_RED"));
    }
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onLedGreen(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< LED_GREEN"));
    }
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onLedBlue(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< LED_BLUE"));
    }
    ledState = true;
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onLedOff(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< LED_OFF"));
    }
    ledState = false;
    digitalWrite(LED_PIN, LOW);
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onRelay1(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.print(F("\n[RX] <<< RELAY1"));
        if (length >= 1) {
            Serial.print(F(" - "));
            Serial.println(payload[0] ? F("ON") : F("OFF"));
        } else {
            Serial.println();
        }
    }
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onReset(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.println(F("\n[RX] <<< RESET"));
    }
    ledState = false;
    digitalWrite(LED_PIN, LOW);
    Serial.println(F("  System reset"));
    // ❌ ไม่ต้องตอบกลับ! Heartbeat จะบอก PC อยู่แล้ว
}

void onSensor(byte* payload, byte length) {
    if (debugCallbacks) {
        Serial.print(F("\n[RX] <<< SENSOR"));
        if (length >= 1) {
            Serial.print(F(" - "));
            Serial.println(payload[0] ? F("READ") : F("OFF"));
        } else {
            Serial.println();
        }
    }

    if (length >= 1 && payload[0]) {
        // ✅ ส่งข้อมูลเซ็นเซอร์ เมื่อ PC ร้องขอ
        byte sensorData[2];
        int value = analogRead(A0);
        sensorData[0] = (value >> 8) & 0xFF;
        sensorData[1] = value & 0xFF;
        comm.sendFrame(CMD_SENSOR, sensorData, sizeof(sensorData));
        Serial.print(F("  Sensor value: "));
        Serial.println(value);
    }
    // ❌ ไม่ต้องตอบ PING - Heartbeat จะบอก PC อยู่แล้ว
}

// ===== Debug Command Handler =====

void handleDebugCommands() {
    if (Serial.available()) {
        char cmd = Serial.read();

        switch (cmd) {
            case 't': // Test PING
                Serial.println(F("\n[TEST] Sending PING..."));
                comm.sendPing();
                break;

            case 'h': // Send Heartbeat
                Serial.println(F("\n[TEST] Sending HEARTBEAT..."));
                comm.sendHeartbeat();
                break;

            case 'l': // Toggle LED
                ledState = !ledState;
                digitalWrite(LED_PIN, ledState ? HIGH : LOW);
                Serial.print(F("\n[TEST] LED toggled: "));
                Serial.println(ledState ? F("ON") : F("OFF"));
                break;

            case 's': // Show statistics
                Serial.println();
                printStatistics();
                break;

            case 'b': // Show buffer
                Serial.println(F("\n[DEBUG] Buffer content:"));
                comm.printBuffer();
                break;

            case 'r': // Reset statistics
                comm.resetStatistics();
                Serial.println(F("\n[DEBUG] Statistics reset"));
                break;

            case 'c': // Clear buffer
                comm.clearBuffer();
                Serial.println(F("\n[DEBUG] Buffer cleared"));
                break;

            case 'f': // Toggle frame debug
                debugFrames = !debugFrames;
                Serial.print(F("\n[CONFIG] Frame debug: "));
                Serial.println(debugFrames ? F("ON") : F("OFF"));
                break;

            case 'k': // Toggle callback debug
                debugCallbacks = !debugCallbacks;
                Serial.print(F("\n[CONFIG] Callback debug: "));
                Serial.println(debugCallbacks ? F("ON") : F("OFF"));
                break;

            case '?': // Help
                printHelp();
                break;

            case '\n':
            case '\r':
                // Ignore newlines
                break;

            default:
                Serial.print(F("\n[ERROR] Unknown command: "));
                Serial.write(cmd);
                Serial.println(F("\nPress '?' for help"));
                break;
        }
    }
}

// ===== Helper Functions =====

void printHeader() {
    Serial.println(F("\n"));
    Serial.println(F("╔════════════════════════════════════════╗"));
    Serial.println(F("║  AssuraVisionSerial SoftwareSerial Debugger   ║"));
    Serial.println(F("║      Arduino Slave Debug Tool         ║"));
    Serial.println(F("║       C# DataFrame Compatible         ║"));
    Serial.println(F("╚════════════════════════════════════════╝"));
}

void printHelp() {
    Serial.println(F("\n===== Debug Commands ====="));
    Serial.println(F("Test:"));
    Serial.println(F("  t - Send PING"));
    Serial.println(F("  h - Send HEARTBEAT"));
    Serial.println(F("  l - Toggle LED"));
    Serial.println(F("\nMonitor:"));
    Serial.println(F("  s - Show Statistics"));
    Serial.println(F("  b - Show Buffer"));
    Serial.println(F("\nControl:"));
    Serial.println(F("  r - Reset statistics"));
    Serial.println(F("  c - Clear buffer"));
    Serial.println(F("  f - Toggle frame debug"));
    Serial.println(F("  k - Toggle callback debug"));
    Serial.println(F("\nHelp:"));
    Serial.println(F("  ? - Show this help"));
    Serial.println(F("=========================="));
}

void printStatistics() {
    Serial.println(F("\n===== Statistics ====="));
    Serial.print(F("Uptime: "));
    Serial.print(millis() / 1000);
    Serial.println(F(" seconds"));

    Serial.print(F("Frames Sent: "));
    Serial.println(comm.getFramesSent());

    Serial.print(F("Frames Received: "));
    Serial.println(comm.getFramesReceived());

    Serial.print(F("Frames Error: "));
    Serial.println(comm.getFramesError());

    Serial.print(F("LED State: "));
    Serial.println(ledState ? F("ON") : F("OFF"));

    #if defined(ARDUINO_ARCH_AVR)
    // แสดง Free RAM เฉพาะ AVR boards (Uno, Mega, Nano)
    Serial.print(F("Free RAM: "));
    Serial.print(freeRam());
    Serial.println(F(" bytes"));
    #endif

    Serial.println(F("======================"));
}

#if defined(ARDUINO_ARCH_AVR)
// ฟังก์ชันตรวจสอบ Free RAM (เฉพาะ AVR boards)
int freeRam() {
    extern int __heap_start, *__brkval;
    int v;
    return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval);
}
#endif