/*
 * AssuraVisionSerial.h - Arduino Serial Communication Library
 *
 * ไลบรารีสำหรับการสื่อสารระหว่าง Arduino กับโปรแกรม Desktop ผ่าน Serial Port
 * Library for communication between Arduino and Desktop application via Serial Port
 *
 * Protocol Format (ตรงกับ C# DataFrame):
 * [SOF][SOF][Command][Length][SequenceId][Payload...][CRC][EOF][EOF]
 * - SOF (2 bytes): Start of Frame (0xFF, 0xCC)
 * - Command (1 byte): ประเภทคำสั่ง
 * - Length (1 byte): ความยาวของ SequenceId + Payload (1 + N)
 * - SequenceId (1 byte): Sequence number สำหรับ tracking
 * - Payload (0-254 bytes): ข้อมูล
 * - CRC (1 byte): CRC-8 checksum (polynomial 0x07)
 * - EOF (2 bytes): End of Frame (0xFF, 0x55)
 */

#ifndef SERIAL_COMM_H
#define SERIAL_COMM_H

#include <Arduino.h>

// Frame delimiters
#define FRAME_SOF_1 0xFF  // Start of Frame byte 1
#define FRAME_SOF_2 0xCC  // Start of Frame byte 2
#define FRAME_EOF_1 0xFF  // End of Frame byte 1
#define FRAME_EOF_2 0x55  // End of Frame byte 2

// Maximum data length
#define MAX_PAYLOAD_LENGTH 254
#define MAX_FRAME_LENGTH (MAX_PAYLOAD_LENGTH + 8) // SOF(2) + CMD + LEN + SEQ + PAYLOAD + CRC + EOF(2)

// Command Types (ต้องตรงกับ CommandType.cs ฝั่ง C#)
enum CommandType : byte {
    // Connection
    CMD_HEARTBEAT = 0x0A,   // Heartbeat to check connection status
    CMD_PING = 0x0B,        // Ping command for auto-discovery

    // Relay Control
    CMD_RELAY1 = 0x01,
    CMD_RELAY2 = 0x02,
    CMD_RELAY3 = 0x03,
    CMD_RELAY4 = 0x04,
    CMD_RELAY_OFF = 0x05,

    // Buzzer Control
    CMD_BUZZER = 0x06,

    // Sensor Control
    CMD_SENSOR = 0x07,      // Payload: 0x01 = ON, 0x00 = OFF
    // Global Control
    CMD_RESET = 0x08,       // Reset all relays and LEDs
    CMD_TEST_RESULT = 0x09, // Test result: 0x00 = OK, 0x01 = NG
    TESTING = 0x0C, // Testing command
    // LED Control
    CMD_LED_RED = 0xFB,
    CMD_LED_GREEN = 0xFC,
    CMD_LED_BLUE = 0xFD,
    CMD_LED_OFF = 0xFE,

    // Text Messages
    CMD_TXT_MESSAGE = 0xFA
};

// Data Frame structure
struct DataFrame {
    CommandType command;
    byte sequenceId;
    byte payloadLength;
    byte payload[MAX_PAYLOAD_LENGTH];
    byte crc;

    DataFrame() : command(CMD_PING), sequenceId(0), payloadLength(0), crc(0) {
        memset(payload, 0, MAX_PAYLOAD_LENGTH);
    }
};

// Callback function types
typedef void (*FrameReceivedCallback)(DataFrame* frame);
typedef void (*CommandCallback)(byte* payload, byte length);

class AssuraVisionSerial {
private:
    Stream* _serial;
    byte _rxBuffer[MAX_FRAME_LENGTH * 2]; // Double size for safety
    uint16_t _rxIndex;
    byte _sequenceId;  // Sequence ID counter

    // Statistics
    unsigned long _framesSent;
    unsigned long _framesReceived;
    unsigned long _framesError;

    // Callbacks
    FrameReceivedCallback _onFrameReceived;
    CommandCallback _commandCallbacks[256]; // One for each command type

    // Internal methods
    byte computeCRC8(byte* data, uint16_t length);
    bool parseFrame(byte* buffer, uint16_t length, DataFrame* frame);
    int findSOF(byte* buffer, uint16_t length);

public:
    AssuraVisionSerial(Stream& serial);

    // Initialization
    void begin(unsigned long baud = 115200);
    void begin(Stream& serial, unsigned long baud = 115200);

    // Send methods
    bool sendFrame(CommandType cmd, byte* payload = nullptr, byte length = 0);
    bool sendPing();
    bool sendHeartbeat();

    // Receive methods
    void update();  // Call this in loop()
    bool available();
    bool readFrame(DataFrame* frame);

    // Callback registration
    void onFrameReceived(FrameReceivedCallback callback);
    void onCommand(CommandType cmd, CommandCallback callback);

    // Utility methods
    void clearBuffer();
    unsigned long getFramesSent() { return _framesSent; }
    unsigned long getFramesReceived() { return _framesReceived; }
    unsigned long getFramesError() { return _framesError; }
    void resetStatistics();

    // Debug methods
    void printFrame(DataFrame* frame);
    void printBuffer();
};

#endif // SERIAL_COMM_H
