/*
 * RawSerialDebug.ino - ดูข้อมูล Raw Bytes จาก SoftwareSerial
 *
 * โปรแกรมนี้ใช้สำหรับ Debug และดูข้อมูล Raw Bytes ที่รับ/ส่งผ่าน Serial
 * ไม่ใช้ AssuraVisionSerial Library - แสดงข้อมูลแบบ Raw ทุก byte
 * เหมาะสำหรับการตรวจสอบ Protocol, Timing, และ Data Format
 *
 * Hardware:
 * - Arduino (Uno R4, Mega, Nano, etc.)
 * - USB-to-Serial converter (optional)
 * - Connect:
 *   - Arduino Pin 10 (RX) -> TX ของ USB-to-Serial
 *   - Arduino Pin 11 (TX) -> RX ของ USB-to-Serial
 *   - GND -> GND
 *
 * Serial Ports:
 * - Hardware Serial (115200 baud) - Debug output + Command input
 * - SoftwareSerial (9600 baud) - Data communication
 *
 * Debug Commands:
 * - 'h' = Help
 * - 'x' = Toggle HEX/DEC display
 * - 't' = Toggle timestamp
 * - 'a' = Toggle ASCII display
 * - 'c' = Clear screen
 * - 's' = Send test data
 * - 'p' = Send PING frame
 * - 'r' = Show statistics
 * - 'z' = Reset statistics
 */

#include <SoftwareSerial.h>

// กำหนดขา RX และ TX
const int RX_PIN = 10;
const int TX_PIN = 11;

// สร้าง SoftwareSerial object
SoftwareSerial mySerial(RX_PIN, TX_PIN);

// Display settings
bool showHex = true;           // แสดงเป็น HEX (false = DEC)
bool showTimestamp = true;     // แสดง timestamp
bool showASCII = true;         // แสดง ASCII ข้างๆ
bool showDirection = true;     // แสดงทิศทาง RX/TX

// Statistics
unsigned long bytesReceived = 0;
unsigned long bytesSent = 0;
unsigned long startTime = 0;

// Buffer for display
const int LINE_BUFFER_SIZE = 16;
byte lineBuffer[LINE_BUFFER_SIZE];
int lineIndex = 0;
unsigned long lastByteTime = 0;
const unsigned long LINE_TIMEOUT = 100; // ms - ถ้าเว้นช่วงนานกว่านี้ให้ขึ้นบรรทัดใหม่

void setup() {
    // Initialize Hardware Serial
    Serial.begin(9600);
    delay(100);

    printHeader();

    Serial.println(F("\n=== Raw Serial Bytes Debugger ==="));
    Serial.print(F("SoftwareSerial RX Pin: "));
    Serial.println(RX_PIN);
    Serial.print(F("SoftwareSerial TX Pin: "));
    Serial.println(TX_PIN);
    Serial.println(F("SoftwareSerial Baud: 9600"));
    Serial.println();

    // Initialize SoftwareSerial
    mySerial.begin(9600);

    startTime = millis();

    printHelp();
    Serial.println(F("\n[READY] Monitoring Serial data...\n"));
}

void loop() {
    // Handle debug commands from Hardware Serial
    handleDebugCommands();

    // Read and display data from SoftwareSerial
    if (mySerial.available()) {
        byte b = mySerial.read();
        bytesReceived++;
        displayByte(b, true); // true = RX
    }

    // Check line timeout
    if (lineIndex > 0 && (millis() - lastByteTime > LINE_TIMEOUT)) {
        finishLine();
    }
}

// ===== Display Functions =====

void displayByte(byte b, bool isRX) {
     Serial.println( b, DEC);

    // Start new line if buffer is full or direction changed
    if (lineIndex >= LINE_BUFFER_SIZE) {
        finishLine();
    }

    // Start new line on first byte
    if (lineIndex == 0) {
        startNewLine(isRX);
    }

    // Add byte to buffer
    lineBuffer[lineIndex++] = b;
    lastByteTime = millis();

    // Display byte
    if (showHex) {
        if (b < 0x10) Serial.print('0');
        Serial.print(b, HEX);
    } else {
        if (b < 100) Serial.print(' ');
        if (b < 10) Serial.print(' ');
        Serial.print(b, DEC);
    }
    Serial.print(' ');

    // Auto finish line if buffer full
    if (lineIndex >= LINE_BUFFER_SIZE) {
        finishLine();
    }
}

void startNewLine(bool isRX) {
    // Timestamp
    if (showTimestamp) {
        unsigned long elapsed = millis() - startTime;
        char timeStr[12];
        sprintf(timeStr, "[%07lu] ", elapsed);
        Serial.print(timeStr);
    }

    // Direction
    if (showDirection) {
        Serial.print(isRX ? "RX << " : "TX >> ");
    }
}

void finishLine() {
    if (lineIndex == 0) return;

    // Pad with spaces if not full line
    if (showHex) {
        for (int i = lineIndex; i < LINE_BUFFER_SIZE; i++) {
            Serial.print("   ");
        }
    } else {
        for (int i = lineIndex; i < LINE_BUFFER_SIZE; i++) {
            Serial.print("    ");
        }
    }

    // Show ASCII representation
    if (showASCII) {
        Serial.print(" | ");
        for (int i = 0; i < lineIndex; i++) {
            byte b = lineBuffer[i];
            if (b >= 32 && b <= 126) {
                Serial.print((char)b);
            } else if (b == 0x02) {
                Serial.print(F("\u25B6")); // STX symbol (may not display correctly)
            } else if (b == 0x03) {
                Serial.print(F("\u25C0")); // ETX symbol (may not display correctly)
            } else {
                Serial.print('.');
            }
            Serial.print(' ');
        }
    }

    Serial.println();
    lineIndex = 0;
}

// ===== Debug Commands =====

void handleDebugCommands() {
    if (Serial.available()) {
        char cmd = Serial.read();

        switch (cmd) {
            case 'h': // Help
                printHelp();
                break;

            case 'x': // Toggle HEX/DEC
                finishLine();
                showHex = !showHex;
                Serial.print(F("\n[CONFIG] Display mode: "));
                Serial.println(showHex ? F("HEX") : F("DEC"));
                Serial.println();
                break;

            case 't': // Toggle timestamp
                finishLine();
                showTimestamp = !showTimestamp;
                Serial.print(F("\n[CONFIG] Timestamp: "));
                Serial.println(showTimestamp ? F("ON") : F("OFF"));
                Serial.println();
                break;

            case 'a': // Toggle ASCII
                finishLine();
                showASCII = !showASCII;
                Serial.print(F("\n[CONFIG] ASCII display: "));
                Serial.println(showASCII ? F("ON") : F("OFF"));
                Serial.println();
                break;

            case 'd': // Toggle direction
                finishLine();
                showDirection = !showDirection;
                Serial.print(F("\n[CONFIG] Direction indicator: "));
                Serial.println(showDirection ? F("ON") : F("OFF"));
                Serial.println();
                break;

            case 'c': // Clear screen
                finishLine();
                clearScreen();
                Serial.println(F("\n[Screen cleared]\n"));
                break;

            case 's': // Send test data
                finishLine();
                sendTestData();
                break;

            case 'p': // Send PING frame
                finishLine();
                sendPingFrame();
                break;

            case 'r': // Show statistics
                finishLine();
                showStatistics();
                break;

            case 'z': // Reset statistics
                finishLine();
                bytesReceived = 0;
                bytesSent = 0;
                startTime = millis();
                Serial.println(F("\n[STATS] Statistics reset\n"));
                break;

            case '\n':
            case '\r':
                // Ignore newlines
                break;

            default:
                finishLine();
                Serial.print(F("\n[ERROR] Unknown command: "));
                Serial.write(cmd);
                Serial.println(F("\nPress 'h' for help\n"));
                break;
        }
    }
}

// ===== Test Functions =====

void sendTestData() {
    Serial.println(F("\n[TEST] Sending test data..."));

    byte testData[] = {0x48, 0x65, 0x6C, 0x6C, 0x6F}; // "Hello"

    for (int i = 0; i < sizeof(testData); i++) {
        mySerial.write(testData[i]);
        bytesSent++;
    }

    Serial.print(F("Sent "));
    Serial.print(sizeof(testData));
    Serial.println(F(" bytes: \"Hello\"\n"));
}

void sendPingFrame() {
    Serial.println(F("\n[TEST] Sending PING frame..."));

    // PING frame format: [STX][Length][Command][Checksum][ETX]
    // STX=0x02, Length=0, Command=0x01 (PING), Checksum=0x01, ETX=0x03
    byte pingFrame[] = {0x02, 0x00, 0x01, 0x01, 0x03};

    for (int i = 0; i < sizeof(pingFrame); i++) {
        mySerial.write(pingFrame[i]);
        bytesSent++;
    }

    Serial.print(F("Sent "));
    Serial.print(sizeof(pingFrame));
    Serial.println(F(" bytes: [STX][00][01][01][ETX]\n"));
}

// ===== Helper Functions =====

void printHeader() {
    Serial.println(F("\n"));
    Serial.println(F("╔════════════════════════════════════════╗"));
    Serial.println(F("║      Raw Serial Bytes Debugger        ║"));
    Serial.println(F("║     Monitor SoftwareSerial Traffic    ║"));
    Serial.println(F("╚════════════════════════════════════════╝"));
}

void printHelp() {
    Serial.println(F("\n===== Debug Commands ====="));
    Serial.println(F("Display Controls:"));
    Serial.println(F("  x - Toggle HEX/DEC display"));
    Serial.println(F("  t - Toggle timestamp"));
    Serial.println(F("  a - Toggle ASCII display"));
    Serial.println(F("  d - Toggle direction indicator"));
    Serial.println(F("  c - Clear screen"));
    Serial.println(F("\nTest Commands:"));
    Serial.println(F("  s - Send test data (Hello)"));
    Serial.println(F("  p - Send PING frame"));
    Serial.println(F("\nStatistics:"));
    Serial.println(F("  r - Show statistics"));
    Serial.println(F("  z - Reset statistics"));
    Serial.println(F("\nHelp:"));
    Serial.println(F("  h - Show this help"));
    Serial.println(F("=========================="));
}

void showStatistics() {
    Serial.println(F("\n===== Statistics ====="));

    unsigned long uptime = millis() - startTime;
    Serial.print(F("Runtime: "));
    Serial.print(uptime / 1000);
    Serial.print(F("."));
    Serial.print(uptime % 1000);
    Serial.println(F(" seconds"));

    Serial.print(F("Bytes Received: "));
    Serial.println(bytesReceived);

    Serial.print(F("Bytes Sent: "));
    Serial.println(bytesSent);

    Serial.print(F("Total Bytes: "));
    Serial.println(bytesReceived + bytesSent);

    if (uptime > 0) {
        float rxRate = (bytesReceived * 1000.0) / uptime;
        float txRate = (bytesSent * 1000.0) / uptime;

        Serial.print(F("RX Rate: "));
        Serial.print(rxRate, 2);
        Serial.println(F(" bytes/sec"));

        Serial.print(F("TX Rate: "));
        Serial.print(txRate, 2);
        Serial.println(F(" bytes/sec"));
    }

    Serial.println(F("======================\n"));
}

void clearScreen() {
    // Send ANSI escape codes to clear screen
    Serial.write(27);       // ESC
    Serial.print(F("[2J")); // Clear screen
    Serial.write(27);       // ESC
    Serial.print(F("[H"));  // Home cursor

    printHeader();
}