/*
 * BasicComm.ino - ตัวอย่างการใช้งาน AssuraVisionSerial Library
 *
 * ตัวอย่างนี้แสดงวิธีการใช้งาน AssuraVisionSerial Library เพื่อสื่อสารกับโปรแกรม Desktop
 * This example demonstrates how to use AssuraVisionSerial Library to communicate with Desktop application
 *
 * Protocol: C# Compatible DataFrame
 * - Frame Format: [SOF][SOF][CMD][LEN][SEQ][PAYLOAD][CRC][EOF][EOF]
 * - CRC-8 checksum (polynomial 0x07)
 *
 * Hardware:
 * - Arduino (Uno, Mega, Nano, etc.)
 * - USB Cable for Serial communication
 *
 * Usage:
 * 1. Upload sketch to Arduino
 * 2. Arduino will send PING to Desktop automatically
 * 3. Desktop will respond and establish connection
 */
#include <SoftwareSerial.h>
#include <AssuraVisionSerial.h>
#include <TcBUTTON.h>

// กำหนดขา RX และ TX สำหรับ SoftwareSerial
const int RX_PIN = 10;
const int TX_PIN = 11;

// สร้าง SoftwareSerial object
SoftwareSerial mySerial(RX_PIN, TX_PIN);

// สร้าง AssuraVisionSerial object
AssuraVisionSerial comm(mySerial);

// ตัวแปรสำหรับ LED (ใช้ built-in LED)
const int LED_PIN = LED_BUILTIN;
bool ledState = false;

// ตัวแปรสำหรับ Heartbeat
unsigned long lastHeartbeat = 0;
const unsigned long HEARTBEAT_INTERVAL = 2000; // ส่ง Heartbeat ทุก 2 วินาที

#define SENSR5_PIN 5 // ขาเซ็นเซอร์ Clear
void onState5Change(bool state);

TcBUTTON sensor5Button(SENSR5_PIN);

#define SENSR6_PIN 6 // ขาเซ็นเซอร์ START
void onState6Change(bool state);
TcBUTTON sensor6Button(SENSR6_PIN);

void setup() {
    // Initialize LED pin
    pinMode(LED_PIN, OUTPUT);
    digitalWrite(LED_PIN, LOW);
    sensor5Button.setOnEventChange(onState5Change);
    sensor6Button.setOnEventChange(onState6Change);
    
    mySerial.begin(9600);
    // Initialize Serial Communication (9600 baud)
    comm.begin(9600);
    Serial.begin(9600);
    comm.onCommand(CMD_PING, onPingReceived);
    comm.onCommand(CMD_TEST_RESULT, onTestResultReceived);
    comm.onCommand(CMD_TXT_MESSAGE, onTextMessageReceived);
    comm.onCommand(TESTING, onTestingCommandReceived);

    delay(1000);

    Serial.println(F("Arduino ready!"));
    Serial.println(F("HEARTBEAT sent - waiting for PC discovery..."));
    Serial.println(F("Waiting for commands..."));
}

void loop() {
    // อัพเดท AssuraVisionSerial (ตรวจสอบข้อมูลที่รับ)
    comm.update();
    sensor5Button.update();
    sensor6Button.update();
    // ส่ง Heartbeat เป็นระยะ
    if (millis() - lastHeartbeat >= HEARTBEAT_INTERVAL) {
        lastHeartbeat = millis();
        // comm.sendHeartbeat();
        // Serial.println(F("Heartbeat sent"));
    }
}

// ===== Callback Functions =====

// Callback เมื่อได้รับคำสั่ง PING (Auto-Discovery)
void onPingReceived(byte* payload, byte length) {
    delay(40);
    comm.sendHeartbeat();

    Serial.println(F("\n[RX] <<< PING received from PC (Discovery)"));
    Serial.print(F("  Payload length: "));
    Serial.println(length);
   
    ledState != ledState;
    digitalWrite(LED_PIN, ledState ? HIGH : LOW);
}

// Callback เมื่อได้รับคำสั่ง TEST_RESULT
void onTestResultReceived(byte* payload, byte length) {
    if (length < 1) return;
    byte result = payload[0];
    if (result == 0x00) {
        Serial.println(F("Test Result: OK"));
    } else if (result == 0x01) {
        Serial.println(F("Test Result: NG"));
    } else {
        Serial.println(F("Test Result: Unknown"));
    }
}

// Callback เมื่อได้รับคำสั่ง TXT_MESSAGE
void onTextMessageReceived(byte* payload, byte length) {
    if (length == 0) return;
    String message;
    for (byte i = 0; i < length; i++) {
        message += (char)payload[i];
    }
    Serial.print(F("Message from PC: "));
    Serial.println(message);
}

void onTestingCommandReceived(byte* payload, byte length) {
    Serial.println(F("Received TESTING command from PC."));
}

void onState5Change(bool state) {
    Serial.print(F("Sensor 5 State Changed: "));
    Serial.println(state ? F("HIGH") : F("LOW"));
    byte payload[1];
    payload[0] = state ? 0x01 : 0x00;
    comm.sendFrame(CMD_RESET, payload, 1);
}

void onState6Change(bool state) {
    Serial.print(F("Sensor 6 State Changed: "));
    Serial.println(state ? F("HIGH") : F("LOW"));
    byte payload[1];
    payload[0] = state ? 0x01 : 0x00;
    comm.sendFrame(CMD_SENSOR, payload, 1);
}