/*
 * AssuraVisionSerial.cpp - Arduino Serial Communication Library Implementation
 */

#include "AssuraVisionSerial.h"

AssuraVisionSerial::AssuraVisionSerial(Stream& serial)
    : _serial(&serial), _rxIndex(0), _sequenceId(0),
      _framesSent(0), _framesReceived(0), _framesError(0),
      _onFrameReceived(nullptr) {
    memset(_rxBuffer, 0, sizeof(_rxBuffer));
    memset(_commandCallbacks, 0, sizeof(_commandCallbacks));
}

void AssuraVisionSerial::begin(unsigned long baud) {
    if (_serial == &Serial) {
        Serial.begin(baud);
    }
    clearBuffer();
    resetStatistics();
    _sequenceId = 0;
}

void AssuraVisionSerial::begin(Stream& serial, unsigned long baud) {
    _serial = &serial;
    begin(baud);
}

// CRC-8 calculation (ตรงกับ C# - polynomial 0x07)
byte AssuraVisionSerial::computeCRC8(byte* data, uint16_t length) {
    byte crc = 0;
    for (uint16_t i = 0; i < length; i++) {
        crc ^= data[i];
        for (byte j = 0; j < 8; j++) {
            if (crc & 0x80) {
                crc = (crc << 1) ^ 0x07;
            } else {
                crc = crc << 1;
            }
        }
    }
    return crc;
}

bool AssuraVisionSerial::sendFrame(CommandType cmd, byte* payload, byte length) {
    if (length > MAX_PAYLOAD_LENGTH) {
        return false;
    }

    // Build frame: [SOF][SOF][CMD][LEN][SEQ][PAYLOAD][CRC][EOF][EOF]
    byte frame[MAX_FRAME_LENGTH];
    uint16_t frameIndex = 0;

    // SOF (2 bytes)
    frame[frameIndex++] = FRAME_SOF_1;
    frame[frameIndex++] = FRAME_SOF_2;

    // Command
    frame[frameIndex++] = (byte)cmd;

    // Length = SequenceId + Payload length
    byte len = 1 + length;  // 1 for sequenceId + payload length
    frame[frameIndex++] = len;

    // Sequence ID
    frame[frameIndex++] = _sequenceId++;

    // Payload
    if (payload != nullptr && length > 0) {
        memcpy(&frame[frameIndex], payload, length);
        frameIndex += length;
    }

    // Calculate CRC-8 (from CMD to end of PAYLOAD)
    // CRC data = CMD + LEN + SEQ + PAYLOAD
    byte crcData[MAX_PAYLOAD_LENGTH + 3];
    crcData[0] = (byte)cmd;
    crcData[1] = len;
    crcData[2] = _sequenceId - 1;  // Use the sequenceId we just sent
    if (length > 0) {
        memcpy(&crcData[3], payload, length);
    }
    byte crc = computeCRC8(crcData, 3 + length);
    frame[frameIndex++] = crc;

    // EOF (2 bytes)
    frame[frameIndex++] = FRAME_EOF_1;
    frame[frameIndex++] = FRAME_EOF_2;

    // Send frame
    size_t written = _serial->write(frame, frameIndex);

    if (written == frameIndex) {
        _framesSent++;
        return true;
    }

    return false;
}

bool AssuraVisionSerial::sendPing() {
    return sendFrame(CMD_PING);
}

bool AssuraVisionSerial::sendHeartbeat() {
    return sendFrame(CMD_HEARTBEAT);
}

// Find Start of Frame (SOF) in buffer
int AssuraVisionSerial::findSOF(byte* buffer, uint16_t length) {
    for (uint16_t i = 0; i < length - 1; i++) {
        if (buffer[i] == FRAME_SOF_1 && buffer[i + 1] == FRAME_SOF_2) {
            return i;
        }
    }
    return -1;
}

// Parse incoming frame: [SOF][SOF][CMD][LEN][SEQ][PAYLOAD][CRC][EOF][EOF]
bool AssuraVisionSerial::parseFrame(byte* buffer, uint16_t bufferLength, DataFrame* frame) {
    // Minimum frame length check (8 bytes minimum)
    // SOF(2) + CMD(1) + LEN(1) + SEQ(1) + CRC(1) + EOF(2) = 8 bytes
    if (bufferLength < 8) {
        return false;
    }

    // Check SOF
    if (buffer[0] != FRAME_SOF_1 || buffer[1] != FRAME_SOF_2) {
        return false;
    }

    // Read frame fields
    byte command = buffer[2];
    byte len = buffer[3];  // len = SEQ(1) + PAYLOAD length

    // Calculate expected frame length
    // SOF(2) + CMD(1) + LEN(1) + SEQ(1) + PAYLOAD(len-1) + CRC(1) + EOF(2)
    uint16_t expectedLength = 7 + len;

    if (bufferLength != expectedLength) {
        return false;
    }

    // Check EOF
    if (buffer[expectedLength - 2] != FRAME_EOF_1 ||
        buffer[expectedLength - 1] != FRAME_EOF_2) {
        return false;
    }

    // Extract sequence ID and payload
    byte sequenceId = buffer[4];
    byte payloadLength = len - 1;  // len includes SEQ

    // Extract payload
    byte payload[MAX_PAYLOAD_LENGTH];
    if (payloadLength > 0) {
        memcpy(payload, &buffer[5], payloadLength);
    }

    // Get received CRC
    byte receivedCRC = buffer[4 + len];

    // Calculate CRC (from CMD to end of PAYLOAD)
    byte crcData[MAX_PAYLOAD_LENGTH + 3];
    crcData[0] = command;
    crcData[1] = len;
    crcData[2] = sequenceId;
    if (payloadLength > 0) {
        memcpy(&crcData[3], payload, payloadLength);
    }
    byte calculatedCRC = computeCRC8(crcData, 3 + payloadLength);

    // Verify CRC
    if (receivedCRC != calculatedCRC) {
        return false;
    }

    // Fill frame structure
    frame->command = (CommandType)command;
    frame->sequenceId = sequenceId;
    frame->payloadLength = payloadLength;
    frame->crc = receivedCRC;
    if (payloadLength > 0) {
        memcpy(frame->payload, payload, payloadLength);
    }

    return true;
}

void AssuraVisionSerial::update() {
    // Read available bytes
    while (_serial->available()) {
        byte b = _serial->read();

        // Prevent buffer overflow
        if (_rxIndex >= sizeof(_rxBuffer)) {
            // Buffer full, clear it
            clearBuffer();
        }

        _rxBuffer[_rxIndex++] = b;

        // Check if we have potential frame (at least 8 bytes)
        if (_rxIndex >= 8) {
            // Find SOF
            int sofPos = findSOF(_rxBuffer, _rxIndex);

            if (sofPos < 0) {
                // No SOF found, keep only last byte (might be start of SOF)
                if (_rxIndex > 0) {
                    _rxBuffer[0] = _rxBuffer[_rxIndex - 1];
                    _rxIndex = 1;
                }
                continue;
            }

            // Move SOF to beginning if not already there
            if (sofPos > 0) {
                memmove(_rxBuffer, &_rxBuffer[sofPos], _rxIndex - sofPos);
                _rxIndex -= sofPos;
            }

            // Check if we have enough bytes to read length field
            if (_rxIndex < 4) {
                continue;
            }

            // Read expected frame length
            byte len = _rxBuffer[3];  // LEN field
            uint16_t expectedFrameLength = 7 + len;  // SOF(2) + CMD + LEN + DATA(len) + CRC + EOF(2)

            // Check if we have complete frame
            if (_rxIndex >= expectedFrameLength) {
                // Check EOF
                if (_rxBuffer[expectedFrameLength - 2] == FRAME_EOF_1 &&
                    _rxBuffer[expectedFrameLength - 1] == FRAME_EOF_2) {
                    // Try to parse frame
                    DataFrame frame;
                    if (parseFrame(_rxBuffer, expectedFrameLength, &frame)) {
                        _framesReceived++;

                        // Call general callback
                        if (_onFrameReceived != nullptr) {
                            _onFrameReceived(&frame);
                        }

                        // Call specific command callback
                        if (_commandCallbacks[frame.command] != nullptr) {
                            _commandCallbacks[frame.command](frame.payload, frame.payloadLength);
                        }

                        // Remove processed frame from buffer
                        if (_rxIndex > expectedFrameLength) {
                            memmove(_rxBuffer, &_rxBuffer[expectedFrameLength],
                                    _rxIndex - expectedFrameLength);
                            _rxIndex -= expectedFrameLength;
                        } else {
                            clearBuffer();
                        }
                    } else {
                        // Parse failed
                        _framesError++;
                        // Remove bad SOF and try again
                        memmove(_rxBuffer, &_rxBuffer[2], _rxIndex - 2);
                        _rxIndex -= 2;
                    }
                } else {
                    // Bad frame, remove SOF and continue
                    _framesError++;
                    memmove(_rxBuffer, &_rxBuffer[2], _rxIndex - 2);
                    _rxIndex -= 2;
                }
            }
        }
    }
}

bool AssuraVisionSerial::available() {
    // Check if we have SOF in buffer
    return findSOF(_rxBuffer, _rxIndex) >= 0;
}

bool AssuraVisionSerial::readFrame(DataFrame* frame) {
    if (!available()) {
        return false;
    }

    // Find SOF
    int sofPos = findSOF(_rxBuffer, _rxIndex);
    if (sofPos < 0) {
        return false;
    }

    // Move to beginning if needed
    if (sofPos > 0) {
        memmove(_rxBuffer, &_rxBuffer[sofPos], _rxIndex - sofPos);
        _rxIndex -= sofPos;
    }

    // Check if we have enough data to read length
    if (_rxIndex < 4) {
        return false;
    }

    byte len = _rxBuffer[3];
    uint16_t expectedFrameLength = 7 + len;

    if (_rxIndex < expectedFrameLength) {
        return false;
    }

    // Parse frame
    if (parseFrame(_rxBuffer, expectedFrameLength, frame)) {
        // Remove from buffer
        if (_rxIndex > expectedFrameLength) {
            memmove(_rxBuffer, &_rxBuffer[expectedFrameLength],
                    _rxIndex - expectedFrameLength);
            _rxIndex -= expectedFrameLength;
        } else {
            clearBuffer();
        }
        _framesReceived++;
        return true;
    }

    // Parse failed
    _framesError++;
    memmove(_rxBuffer, &_rxBuffer[2], _rxIndex - 2);
    _rxIndex -= 2;
    return false;
}

void AssuraVisionSerial::onFrameReceived(FrameReceivedCallback callback) {
    _onFrameReceived = callback;
}

void AssuraVisionSerial::onCommand(CommandType cmd, CommandCallback callback) {
    _commandCallbacks[cmd] = callback;
}

void AssuraVisionSerial::clearBuffer() {
    _rxIndex = 0;
    memset(_rxBuffer, 0, sizeof(_rxBuffer));
}

void AssuraVisionSerial::resetStatistics() {
    _framesSent = 0;
    _framesReceived = 0;
    _framesError = 0;
}

void AssuraVisionSerial::printFrame(DataFrame* frame) {
    Serial.print(F("Frame [Cmd: 0x"));
    Serial.print(frame->command, HEX);
    Serial.print(F(", Seq: "));
    Serial.print(frame->sequenceId);
    Serial.print(F(", PayloadLen: "));
    Serial.print(frame->payloadLength);
    Serial.print(F(", Payload: "));

    for (byte i = 0; i < frame->payloadLength; i++) {
        if (frame->payload[i] < 0x10) Serial.print('0');
        Serial.print(frame->payload[i], HEX);
        Serial.print(' ');
    }

    Serial.print(F(", CRC: 0x"));
    Serial.print(frame->crc, HEX);
    Serial.println(']');
}

void AssuraVisionSerial::printBuffer() {
    Serial.print(F("Buffer ("));
    Serial.print(_rxIndex);
    Serial.print(F(" bytes): "));

    for (uint16_t i = 0; i < _rxIndex; i++) {
        if (_rxBuffer[i] < 0x10) Serial.print('0');
        Serial.print(_rxBuffer[i], HEX);
        Serial.print(' ');
    }

    Serial.println();
}
