# AssuraVisionSerial - Arduino Serial Communication Library

ไลบรารีสำหรับการสื่อสารระหว่าง Arduino กับโปรแกรม Desktop ผ่าน Serial Port โดยใช้โปรโตคอลแบบ Frame-based พร้อม Checksum Validation

Library for communication between Arduino and Desktop applications via Serial Port using a frame-based protocol with checksum validation.

## คุณสมบัติ (Features)

- โปรโตคอลการสื่อสารที่มั่นคงด้วย STX/ETX markers
- Checksum validation ด้วย XOR
- รองรับข้อมูลได้สูงสุด 255 bytes ต่อ frame
- Callback system สำหรับจัดการคำสั่งต่างๆ
- Buffer management อัตโนมัติ
- สถิติการส่ง/รับข้อมูล
- รองรับ Arduino ทุกรุ่น

## โปรโตคอลการสื่อสาร (Protocol)

```
[STX][Length][Command][Data...][Checksum][ETX]
```

| Field    | Size    | Description                                       |
|----------|---------|---------------------------------------------------|
| STX      | 1 byte  | Start of Text (0x02)                             |
| Length   | 1 byte  | ความยาวของข้อมูล (0-255)                          |
| Command  | 1 byte  | ประเภทคำสั่ง (ดู CommandType)                     |
| Data     | 0-255   | ข้อมูล (ถ้ามี)                                    |
| Checksum | 1 byte  | XOR ของ Length, Command และ Data                 |
| ETX      | 1 byte  | End of Text (0x03)                               |

## ประเภทคำสั่ง (Command Types)

| Command           | Value | Description              |
|-------------------|-------|--------------------------|
| CMD_NONE          | 0x00  | ไม่มีคำสั่ง               |
| CMD_PING          | 0x01  | ตรวจสอบการเชื่อมต่อ       |
| CMD_DATA          | 0x02  | ส่งข้อมูล                 |
| CMD_READ          | 0x03  | อ่านข้อมูล                |
| CMD_WRITE         | 0x04  | เขียนข้อมูล               |
| CMD_RESET         | 0x05  | รีเซ็ต                    |
| CMD_ACK           | 0x06  | ตอบรับสำเร็จ              |
| CMD_NACK          | 0x15  | ตอบรับข้อผิดพลาด          |
| CMD_START         | 0x10  | เริ่มการทำงาน            |
| CMD_STOP          | 0x11  | หยุดการทำงาน             |
| CMD_STATUS        | 0x12  | ขอสถานะ                   |
| CMD_CONFIG        | 0x13  | ตั้งค่า                   |
| CMD_SENSOR_DATA   | 0x20  | ข้อมูลเซ็นเซอร์            |
| CMD_MOTOR_CONTROL | 0x21  | ควบคุมมอเตอร์             |
| CMD_LED_CONTROL   | 0x22  | ควบคุม LED                |
| CMD_CUSTOM        | 0xFF  | คำสั่งกำหนดเอง            |

## การติดตั้ง (Installation)

### วิธีที่ 1: ใช้ Arduino Library Manager

1. เปิด Arduino IDE
2. ไปที่ Sketch > Include Library > Add .ZIP Library
3. เลือกโฟลเดอร์ AssuraVisionSerial

### วิธีที่ 2: Manual Installation

1. คัดลอกโฟลเดอร์ `AssuraVisionSerial` ไปที่ `Documents/Arduino/libraries/`
2. รีสตาร์ท Arduino IDE

## การใช้งาน (Usage)

### การเริ่มต้น

```cpp
#include <AssuraVisionSerial.h>

AssuraVisionSerial comm(Serial);

void setup() {
    comm.begin(115200);
}

void loop() {
    comm.update();  // เรียกทุกครั้งใน loop
}
```

### การส่งข้อมูล

```cpp
// ส่ง Ping
comm.sendPing();

// ส่ง Ack
comm.sendAck();

// ส่ง Nack พร้อมข้อความ error
comm.sendNack("Error message");

// ส่งข้อมูล
byte data[] = {0x01, 0x02, 0x03};
comm.sendData(data, sizeof(data));

// ส่งคำสั่งแบบกำหนดเอง
byte customData[] = {0xAA, 0xBB};
comm.sendFrame(CMD_CUSTOM, customData, sizeof(customData));
```

### การรับข้อมูลด้วย Callback

```cpp
void setup() {
    comm.begin(115200);

    // ลงทะเบียน callback สำหรับคำสั่ง PING
    comm.onCommand(CMD_PING, onPingReceived);

    // ลงทะเบียน callback สำหรับคำสั่ง LED
    comm.onCommand(CMD_LED_CONTROL, onLedControl);
}

void onPingReceived(byte* data, byte length) {
    Serial.println("Ping received!");
    comm.sendAck();
}

void onLedControl(byte* data, byte length) {
    if (length >= 1) {
        bool ledOn = (data[0] != 0);
        digitalWrite(LED_BUILTIN, ledOn ? HIGH : LOW);
        comm.sendAck();
    }
}
```

### การรับข้อมูลแบบ Manual

```cpp
void loop() {
    comm.update();

    DataFrame frame;
    if (comm.readFrame(&frame)) {
        // ประมวลผล frame
        Serial.print("Command: 0x");
        Serial.println(frame.command, HEX);

        // เข้าถึงข้อมูล
        for (byte i = 0; i < frame.dataLength; i++) {
            Serial.print(frame.data[i], HEX);
            Serial.print(" ");
        }
        Serial.println();
    }
}
```

## ตัวอย่าง (Examples)

### ตัวอย่าง 1: Basic Communication

```cpp
#include <AssuraVisionSerial.h>

AssuraVisionSerial comm(Serial);

void setup() {
    pinMode(LED_BUILTIN, OUTPUT);
    comm.begin(115200);
    comm.onCommand(CMD_LED_CONTROL, onLedControl);
    comm.sendAck();  // บอกว่าพร้อมแล้ว
}

void loop() {
    comm.update();
}

void onLedControl(byte* data, byte length) {
    if (length >= 1) {
        digitalWrite(LED_BUILTIN, data[0] ? HIGH : LOW);
        comm.sendAck();
    }
}
```

### ตัวอย่าง 2: Sensor Reading

```cpp
unsigned long lastRead = 0;

void loop() {
    comm.update();

    // อ่านเซ็นเซอร์ทุก 1 วินาที
    if (millis() - lastRead >= 1000) {
        lastRead = millis();

        int sensorValue = analogRead(A0);
        byte data[2];
        data[0] = (sensorValue >> 8) & 0xFF;
        data[1] = sensorValue & 0xFF;

        comm.sendFrame(CMD_SENSOR_DATA, data, 2);
    }
}
```

## API Reference

### Methods

#### `void begin(unsigned long baud)`
เริ่มต้น Serial Communication

#### `void update()`
ตรวจสอบและประมวลผลข้อมูลที่รับ (เรียกใน loop)

#### `bool sendFrame(CommandType cmd, byte* data, byte length)`
ส่ง frame พร้อมคำสั่งและข้อมูล

#### `bool sendPing()`
ส่งคำสั่ง Ping

#### `bool sendAck()`
ส่งคำสั่ง Acknowledge

#### `bool sendNack(const char* errorMsg)`
ส่งคำสั่ง Not Acknowledge พร้อมข้อความ error

#### `bool sendData(byte* data, byte length)`
ส่งข้อมูล

#### `void onCommand(CommandType cmd, CommandCallback callback)`
ลงทะเบียน callback สำหรับคำสั่งที่กำหนด

#### `bool readFrame(DataFrame* frame)`
อ่าน frame จาก buffer

#### `void clearBuffer()`
ล้าง buffer

#### `unsigned long getFramesSent()`
ดึงจำนวน frames ที่ส่ง

#### `unsigned long getFramesReceived()`
ดึงจำนวน frames ที่รับ

#### `unsigned long getFramesError()`
ดึงจำนวน frames ที่มี error

## การใช้งานกับโปรแกรม Desktop (C#)

ฝั่ง C# มี classes ที่สอดคล้องกันในโฟลเดอร์ `src/Hardware/Services/`:
- `CommandType.cs` - ประเภทคำสั่ง
- `DataFrame.cs` - โครงสร้างข้อมูล frame
- `DataBuffer.cs` - จัดการ buffer
- `SerialPortManager.cs` - จัดการ Serial Port

## License

MIT License

## Author

Toodev - AssuraVision Controller Project
