# 🔄 Communication Flow - Arduino AssuraVisionSerial

## สถาปัตยกรรม Master-Slave

```
┌──────────────────┐                    ┌──────────────────┐
│   C# Desktop     │                    │     Arduino      │
│     (Master)     │  ◄────────────►    │     (Slave)      │
│                  │   Serial Port      │                  │
│  - ส่งคำสั่ง      │                    │  - รับคำสั่ง      │
│  - Monitor        │                    │  - ทำตามคำสั่ง    │
│  - Discovery      │                    │  - ส่ง Heartbeat │
└──────────────────┘                    └──────────────────┘
```

---

## 🎯 บทบาทหน้าที่

### **C# PC = Master**
- ✅ ส่ง PING สำหรับ Auto-Discovery
- ✅ ส่งคำสั่งควบคุม (LED, Relay, etc.)
- ✅ Monitor Heartbeat จาก Arduino
- ✅ ตัดสินใจและประมวลผล

### **Arduino = Slave**
- ✅ ตอบ PING เมื่อถูก Discovery (ส่ง HEARTBEAT)
- ✅ ส่ง HEARTBEAT ทุก 2-3 วินาที
- ✅ รับและทำตามคำสั่งจาก PC
- ✅ ส่งข้อมูล Sensor กลับไป PC
- ✅ แจ้งเหตุการณ์ (Sensor trigger)

---

## 📡 การทำงานแบบละเอียด

### 1. Auto-Discovery (PC หา Arduino)

```
┌─────────┐                                    ┌─────────┐
│  C# PC  │                                    │ Arduino │
└────┬────┘                                    └────┬────┘
     │                                              │
     │ 1. PC เริ่ม Auto-Discovery                  │
     │ Scan: COM1, COM3, COM5...                   │
     │                                              │
     │ 2. PC ส่ง PING -> COM1                      │
     │────────────────────────────────────> X       │ (ไม่มีอุปกรณ์)
     │                                              │
     │ 3. PC ส่ง PING -> COM3                      │
     │────────────────────────────────────>         │ Arduino รับ PING!
     │                                              │
     │                                              │ onPingReceived()
     │                                              │
     │         4. Arduino ตอบด้วย HEARTBEAT         │
     │    <────────────────────────────────────     │ sendHeartbeat()
     │    Frame: [FF CC 0A 01 xx CRC FF 55]        │
     │                                              │
     │ 5. PC ตรวจสอบ: Valid response ✅             │
     │ 6. PC เชื่อมต่อกับ COM3                     │
     │ 7. PortDiscovered event fired!               │
     │                                              │
```

**โค้ด Arduino:**
```cpp
void onPingReceived(byte* payload, byte length) {
    Serial.println(">>> PING from PC (Discovery)");
    comm.sendHeartbeat(); // ตอบกลับทันที
}
```

---

### 2. Heartbeat Monitoring (Keep-Alive)

```
┌─────────┐                                    ┌─────────┐
│  C# PC  │                                    │ Arduino │
└────┬────┘                                    └────┬────┘
     │                                              │
     │                              loop() {        │
     │                                if (2s passed)│
     │         HEARTBEAT (0x0A)                     │
     │    <────────────────────────────────────     │ sendHeartbeat()
     │    Frame: [FF CC 0A 01 xx CRC FF 55]        │   }
     │                                              │
     │ PC รับและอัพเดท                              │
     │ LastHeartbeat = Now                          │
     │ EquipmentConnected = True ✅                 │
     │                                              │
     │ ... 2 วินาทีผ่านไป ...                       │
     │                                              │
     │         HEARTBEAT (0x0A)                     │
     │    <────────────────────────────────────     │ sendHeartbeat()
     │                                              │
     │ LastHeartbeat = Now                          │
     │                                              │
     │ Monitor: if (Now - LastHB > 5s)              │
     │     ❌ Connection Lost!                      │
     │     ErrorOccurred event                      │
     │                                              │
```

**โค้ด Arduino:**
```cpp
void loop() {
    comm.update();

    if (millis() - lastHeartbeat >= 2000) {
        lastHeartbeat = millis();
        comm.sendHeartbeat(); // ส่งทุก 2 วินาที
    }
}
```

**โค้ด C#:**
```csharp
// C# รับและเช็ค
if (frame.Command == (byte)CommandType.Heartbeat) {
    _status.LastHeartbeat = DateTime.Now;
    _status.EquipmentConnected = true;
}

// Monitor timeout
if ((DateTime.Now - _status.LastHeartbeat).TotalSeconds > 5) {
    // Connection lost!
    _status.EquipmentConnected = false;
    ErrorOccurred?.Invoke(this, "Heartbeat timeout");
}
```

---

### 3. Command Flow (PC ส่งคำสั่ง)

#### **3.1 Simple Command (LED Control)**

```
┌─────────┐                                    ┌─────────┐
│  C# PC  │                                    │ Arduino │
└────┬────┘                                    └────┬────┘
     │                                              │
     │ User กดปุ่ม "LED RED"                        │
     │                                              │
     │ 1. PC ส่งคำสั่ง LED_RED (0xFB)              │
     │────────────────────────────────────>         │
     │    Frame: [FF CC FB 01 xx CRC FF 55]        │
     │                                              │
     │                                              │ update()
     │                                              │ parseFrame()
     │                                              │ onLedRed()
     │                                              │ digitalWrite(LED, HIGH)
     │                                              │
     │ *** Arduino ไม่ตอบกลับ! ***                 │
     │ PC รู้ว่าสำเร็จจาก Heartbeat                │
     │                                              │
     │ ... 2 วินาทีผ่านไป ...                       │
     │                                              │
     │         HEARTBEAT (still alive)              │
     │    <────────────────────────────────────     │
     │                                              │
```

**โค้ด Arduino:**
```cpp
void onLedRed(byte* payload, byte length) {
    digitalWrite(LED_PIN, HIGH);
    // ❌ ไม่ส่ง PING กลับ!
    // ✅ Heartbeat อัตโนมัติจะบอก PC ว่า Arduino ยัง alive
}
```

**เหตุผลที่ไม่ต้องตอบกลับ:**
- Heartbeat ส่งทุก 2 วินาที อยู่แล้ว
- PC จะรู้ว่า Arduino ยัง alive
- ลดจำนวน frames ที่ส่ง = ลด overhead

---

#### **3.2 Command with Response (Sensor Data Request)**

```
┌─────────┐                                    ┌─────────┐
│  C# PC  │                                    │ Arduino │
└────┬────┘                                    └────┬────┘
     │                                              │
     │ PC ต้องการอ่านค่า Sensor                    │
     │                                              │
     │ 1. PC ส่งคำสั่ง SENSOR (0x07) + [0x01]      │
     │────────────────────────────────────>         │
     │    Frame: [FF CC 07 02 xx 01 CRC FF 55]     │
     │           └─Len=2 (SEQ + 0x01)              │
     │                                              │
     │                                              │ onSensor([0x01])
     │                                              │ value = analogRead(A0)
     │                                              │ value = 512
     │                                              │
     │         2. Arduino ส่งข้อมูล SENSOR กลับ     │
     │    <────────────────────────────────────     │
     │    Frame: [FF CC 07 03 xx 02 00 CRC FF 55]  │
     │           └─Len=3  └─Value: 0x0200 (512)    │
     │                                              │
     │ 3. PC รับค่า = 512 ✅                        │
     │ FrameReceived event                          │
     │ Display value on UI                          │
     │                                              │
```

**โค้ด Arduino:**
```cpp
void onSensor(byte* payload, byte length) {
    if (length >= 1 && payload[0] == 0x01) {
        // อ่านค่า sensor
        int value = analogRead(A0);
        byte data[2] = {
            (value >> 8) & 0xFF,  // High byte
            value & 0xFF           // Low byte
        };

        // ส่งกลับไป PC
        comm.sendFrame(CMD_SENSOR, data, 2);
    }
}
```

---

#### **3.3 Event-Driven (Sensor Trigger)**

```
┌─────────┐                                    ┌─────────┐
│  C# PC  │                                    │ Arduino │
└────┬────┘                                    └────┬────┘
     │                                              │
     │                                              │ มีวัตถุผ่าน!
     │                                              │ Sensor triggered!
     │                                              │
     │         1. Arduino แจ้ง SENSOR + [0x01]      │
     │    <────────────────────────────────────     │
     │    Frame: [FF CC 07 02 xx 01 CRC FF 55]     │
     │           └─Payload: 0x01 = Triggered       │
     │                                              │
     │ 2. PC รับ Frame                               │
     │ 3. FrameReceived event                       │
     │ 4. if (CMD==SENSOR && data[0]==1)            │
     │    StartInspection() 📸                      │
     │                                              │
     │ 5. PC ส่ง LED_BLUE (กำลังตรวจสอบ)           │
     │────────────────────────────────────>         │
     │                                              │ LED สีน้ำเงินติด
     │                                              │
     │ ... กำลังตรวจสอบภาพ ...                      │
     │                                              │
     │ 6. PC ส่ง TEST_RESULT (0x00 = OK)           │
     │────────────────────────────────────>         │
     │                                              │
     │ 7. PC ส่ง LED_GREEN (ผ่าน!)                 │
     │────────────────────────────────────>         │
     │                                              │ LED สีเขียวติด ✅
     │                                              │
```

---

## 📊 สรุป Protocol Rules

### ✅ **DO (ควรทำ)**
1. **Arduino ส่ง HEARTBEAT ทุก 2-3 วินาที**
2. **Arduino ตอบ PING ด้วย HEARTBEAT** (เมื่อถูก discovery)
3. **Arduino ส่งข้อมูลกลับเฉพาะเมื่อมีข้อมูล** (Sensor data, Status)
4. **PC Monitor HEARTBEAT timeout (5 วินาที)**
5. **Arduino แจ้ง Event สำคัญ** (Sensor trigger)

### ❌ **DON'T (ไม่ควรทำ)**
1. ❌ **Arduino ตอบ PING ด้วย PING** (ใช้ HEARTBEAT แทน)
2. ❌ **Arduino ตอบทุกคำสั่งด้วย ACK** (ไม่จำเป็น - มี Heartbeat)
3. ❌ **PC ส่ง HEARTBEAT** (เฉพาะ Arduino ส่ง)
4. ❌ **Arduino รอคำสั่งจาก PC ก่อนส่ง Heartbeat** (ส่งอัตโนมัติ)

---

## 🎯 ประโยชน์ของ Design นี้

1. **ลด Overhead** - ไม่ต้อง ACK ทุกคำสั่ง
2. **Connection Monitoring** - รู้ทันทีถ้า Arduino หลุด
3. **Auto-Discovery** - หา Arduino อัตโนมัติ
4. **Event-Driven** - Arduino แจ้ง Event ได้ทันที
5. **Simple Protocol** - เข้าใจง่าย maintain ง่าย

---

**สร้างเมื่อ:** 2025-01-XX
**Protocol Version:** 1.0 (C# DataFrame Compatible)
