#include "sdkconfig.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_system.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "nvs_flash.h"
#include "lwip/err.h"
#include "lwip/sys.h"

#define CHRONOLOG_REMOTE_ENABLE         1           // Enable remote logging server
#define CHRONOLOG_THREAD_SAFE           1           // Enabled thread-safe logging
#define CHRONOLOG_COLOR_ENABLE          1           // 0 = Disable colors for ANSII-incompatible terminals

#include "ChronoLog.h"

// WiFi credentials
#define WIFI_SSID                       "YourSSID"
#define WIFI_PASS                       "YourPassword"
#define WIFI_MAXIMUM_RETRY              5

static EventGroupHandle_t               s_wifi_event_group;
#define WIFI_CONNECTED_BIT              BIT0
#define WIFI_FAIL_BIT                   BIT1

static int                              s_retry_num = 0;
static ChronoLogger                     logger("RemoteLogger", CHRONOLOG_LEVEL_DEBUG);
static ChronoLogger                     networkLogger("Network", CHRONOLOG_LEVEL_DEBUG);
static ChronoLogger                     sensorLogger("Sensors", CHRONOLOG_LEVEL_DEBUG);

// Task handles
TaskHandle_t                            sensorTaskHandle  = NULL;
TaskHandle_t                            networkTaskHandle = NULL;

// Task 1: Sensor Data Logger (Simulated)
void sensorTask(void *pvParameters) {
    logger.info("Sensor task started on core %d", xPortGetCoreID());
    
    while (true) {
        float temp = 20.0 + (rand() % 100) / 10.0;
        sensorLogger.debug("Temperature reading: %.1f°C", temp);
        
        if (temp > 28.0) {
            sensorLogger.warn("High temperature detected: %.1f°C", temp);
        }
        
        vTaskDelay(pdMS_TO_TICKS(2000));
    }
}

// Task 2: Network Activity Logger (Simulated)
void networkTask(void *pvParameters) {
    logger.info("Network task started on core %d", xPortGetCoreID());
    
    while (true) {
        networkLogger.info("Network status: Connected");
        
        if (rand() % 10 == 0) {
            networkLogger.error("Packet loss detected!");
        }
        
        vTaskDelay(pdMS_TO_TICKS(3000));
    }
}

static void event_handler(void* arg, esp_event_base_t event_base, int32_t event_id, void* event_data) {
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_START) {
        esp_wifi_connect();
    } else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_DISCONNECTED) {
        if (s_retry_num < WIFI_MAXIMUM_RETRY) {
            esp_wifi_connect();
            s_retry_num++;
            networkLogger.warn("Retrying WiFi connection... (%d/%d)", s_retry_num, WIFI_MAXIMUM_RETRY);
        } else {
            xEventGroupSetBits(s_wifi_event_group, WIFI_FAIL_BIT);
            networkLogger.error("WiFi Connection failed after %d attempts", WIFI_MAXIMUM_RETRY);
        }
    } else if (event_base == IP_EVENT && event_id == IP_EVENT_STA_GOT_IP) {
        ip_event_got_ip_t* event = (ip_event_got_ip_t*) event_data;
        networkLogger.info("Got IP Address: " IPSTR, IP2STR(&event->ip_info.ip));
        s_retry_num = 0;
        xEventGroupSetBits(s_wifi_event_group, WIFI_CONNECTED_BIT);
        
        // Start remote logging server after getting IP
        ChronoLogRemote::getInstance()->start(9999);
        networkLogger.info("Remote logging server started on port 9999");
        
        networkLogger.info("Connect using: nc " IPSTR " 9999", IP2STR(&event->ip_info.ip));
    }
}

void wifi_init_sta(void) {
    s_wifi_event_group = xEventGroupCreate();

    ESP_ERROR_CHECK(esp_netif_init());
    ESP_ERROR_CHECK(esp_event_loop_create_default());
    esp_netif_create_default_wifi_sta();

    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));

    esp_event_handler_instance_t instance_any_id;
    esp_event_handler_instance_t instance_got_ip;

    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT, ESP_EVENT_ANY_ID, &event_handler, NULL, &instance_any_id));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(IP_EVENT, IP_EVENT_STA_GOT_IP, &event_handler, NULL, &instance_got_ip));

    wifi_config_t wifi_config = {};
    strcpy((char*)wifi_config.sta.ssid, WIFI_SSID);
    strcpy((char*)wifi_config.sta.password, WIFI_PASS);
    
    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_STA));
    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_STA, &wifi_config));
    ESP_ERROR_CHECK(esp_wifi_start());

    networkLogger.info("WiFi initialization complete");
}

extern "C" void app_main(void) {
    // Initialize NVS
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    logger.info("Starting ESP-IDF Remote Logging Example...");
    logger.info("ChronoLog Thread Safety: %s", CHRONOLOG_THREAD_SAFE ? "ENABLED" : "DISABLED");

    networkLogger.info("Starting WiFi connection...");
    wifi_init_sta();

    // Create tasks to demonstrate thread safety with remote logging
    xTaskCreatePinnedToCore(sensorTask, "SensorTask", 4096, NULL, 2, &sensorTaskHandle, 0);
    xTaskCreatePinnedToCore(networkTask, "NetworkTask", 4096, NULL, 2, &networkTaskHandle, 1);

    logger.info("Tasks created and running concurrently on different cores.");
    
    // Main loop - simulate some system events
    int counter = 0;
    while(1) {
        logger.info("Main loop heartbeat #%d", ++counter);
        
        if (counter % 5 == 0) {
            logger.warn("Free heap: %d bytes", esp_get_free_heap_size());
        }
        
        vTaskDelay(pdMS_TO_TICKS(5000));
    }
}
