## Пассивные виджеты
### Label, LabelNum, LabelFloat
Виджет для отображения текста или цифры. Значение обновляется через `update`, поддерживает изменение цвета через `updateColor`.

### LED
Светодиод, имеет два цвета. Состояние переключает между этими цветами, по умолчанию красный и зелёный, можно задать свои два цвета. Значение состояния обновляется через `update`, также поддерживает изменение цвета напрямую через `updateColor` независимо от состояния.

### Image
Вывод изображения:
- Из Интернета - ссылка начинается с `http`
- Из Flash-памяти - ссылка начинается с `/`

При отправке обновления `update` изображение будет обновлено на новый url.

### Paragraph
Вывод строки текста.

### HTML
Вывод кода в HTML разметке:

```cpp
b.HTML("", R"(<a href="http://google.com">Google</a>)");
```

### Stream
Для запуска стрима с ESP32-CAM нужно подключить `SettingsCamera.h`, в нём можно настроить тип камеры и подключение. Для плат ESP32-CAM AiThinker ничего трогать не нужно. Для запуска стрима нужно инициализировать камеру и запустить стрим, он будет выводиться в виджет `Stream`:

```cpp
bool sets::cameraInit(framesize_t frame_size = FRAMESIZE_VGA, pixformat_t pixel_format = PIXFORMAT_JPEG, int jpeg_quality = 12);
void sets::streamBegin(uint16_t fps = 30, uint16_t port = 82, const char* path = "/stream");
void sets::streamEnd();
```

```cpp
#include <SettingsCamera.h>

void build(sets::Builder& b) {
    b.Stream();
}

void setup() {
    sets::cameraInit();
    sets::streamBegin();
}
```

### Link
Виджет-ссылка, открывает ссылку в новой вкладке.

### Log
Готовый инструмент для ведения логов и отправки в вебморду (почти Web-Serial) - `Logger`:

```cpp
sets::Logger logger(150);   // размер буфера

void build(sets::Builder& b) {
    b.Log(H(log), logger);

    if (b.Button("Test")) {
        // печатать как в Serial в любом месте в программе
        logger.println(millis());
    }
}

void update(sets::Updater& upd) {
    // отправить лог
    upd.update(H(log), logger);
}
```

Логгер автоматически окрашивает строки в цвет статуса, если он передан (указыается в начале строки):

- Не указан - цвет темы
- `info:` - чёрный цвет
- `warn:` - оранжевый цвет
- `err:` - красный цвет

Например `logger.println("warn: some warning);`. Также есть готовые строки, функция возвращает `String` с нужным префиксом:

```cpp
logger.println(sets::Logger::info() + "info text");
logger.println(sets::Logger::warn() + "warn text");
logger.println(sets::Logger::error() + "error text");
```

### Table
Таблица. Передаётся в формате CSV строкой или как путь к файлу на флешке:
- С расширением `.csv` - CSV, горизонтальный разделитель `;`, вертикальный `\n`. Внутри ячеек использование разделителей не допускается
- С расширением `.tbl` - бинарная таблица [Table](https://github.com/GyverLibs/Table)

Подписи - список с разделением `;`.

### LinearGauge
Линейная шкала с заполнением, обновляется через апдейт.

## Графики
- У всех вариантов неограниченное количество осей
- Все могут обновляться в реальном времени
- Названия осей передаются в виде списка с разделителем `;` - `"axis 1;test;test 2"`
- Единицы измерения для графиков-линий опционально добавляются в конце названия оси в квадратных скобках - `"Value;Temp[°C];USD[ $]"`
- Виджеты графиков поддерживают загрузку данных напрямую из файла с таблицей в формате:
  - Текстовая CSV таблица (разделитель столбцов `;`, строк - `\n`) с расширением `.csv`
  - Бинарная таблица из библиотеки [Table](https://github.com/GyverLibs/Table) с расширением `.tbl`

> [!TIP]
> Если график хранится в файле бинарной таблицы с расширением `.tbl`, то при скачивании его из файлового менеджера он будет конвертирован в читаемый `.csv`

### PlotRunning
График с линиями. Данные передаются через апдейт, между апдейтами график двигается с заданной скоростью и показывает предыдущие значения осей.

```cpp
// билдер
b.PlotRunning(H(run), "kek1;kek2");

// апдейт по запросу или через вебсокет
float v[] = {random(100), random(100)};
sett.updater().updatePlot(H(run), v);

// или так
sett.updater().updatePlot(H(run), (const float[]){random(100), random(100)});
```

### PlotStack
График с линиями. Данные передаются через апдейт, график обновляется при получении данных.

```cpp
// билдер
b.PlotStack(H(stack), "kek1;kek2;kek3");

// апдейт по запросу или через вебсокет
float v[] = {random(100), random(100), random(100)};
sett.updater().updatePlot(H(stack), v);

// или так
sett.updater().updatePlot(H(stack), (const float[]){random(100), random(100), random(100)});
```

### Plot
График с линиями. Первый столбец таблицы - unix-время, остальные - значения осей:

```cpp
время   значения...
unix1    v1  v2..
unix2    v1  v2..
unix3    v1  v2..
```

```cpp
b.Plot(H(plot1), "/plot.tbl");
b.Plot(H(plot2), "/table.csv");

// TableFile t(&LittleFS, "/plot.tbl");
// t.begin();
// b.Plot(H(plot3), t);
```

### PlotTimeline
График с состояниями осей вкл/выкл в виде блоков. Первый столбец таблицы - unix-время, остальные - значения осей. Есть три формата данных:

1. Режим `sets::TMode::All`, **остальные** столбцы - состояния (1 или 0) всех осей по одному в столбце:

```cpp
время   значения...
unix1    1 0 1..
unix2    0 0 1..
unix3    1 1 0..
```

2. Режим `sets::TMode::Mask`, **второй** столбец - состояния всех осей в виде битовой маски (1-64 бит), *младший бит - первая ось*:

```cpp
время   значения
unix1    0b1010
unix2    0b0010
unix3    0b1000
```

3. Режим `sets::TMode::Single`, **второй** столбец - номер оси, **третий** - состояние (1 или 0):

```cpp
время   ось значение
unix1    1  1
unix2    2  1
unix3    0  0
```

```cpp
b.PlotTimeline("/tline.csv", sets::TMode::All, "1;2;3;4");
b.PlotTimeline("/tline_mask.csv", sets::TMode::Mask, "1;2;3;4");
b.PlotTimeline("/tline_per.csv", sets::TMode::Single, "1;2;3;4");
```

#### Table
`Table` - класс для удобного создания и редактирования динамических таблиц со столбцами разных типов. Использовать очень просто:

```cpp
// таблица с временем и двумя float столбцами (3 столбца, 0 строк)
Table t(0, 3, cell_t::Uint32, cell_t::Float, cell_t::Float);

// прибавить строки
t.append(unix1, random(100), random(100));
t.append(unix2, random(100), random(100));
```

И теперь её можно отправлять.

#### TableFileStatic
Для ведения долгосрочных логов и построения графика гораздо интереснее использовать `TableFileStatic` - эта таблица хранится в файле и не загружается в оперативную память, что позволяет не ограничивать её размер объёмом оперативки и хранить большие объёмы данных (сотни килобайт), а виджет в свою очередь просто скачивает файл. Этот тип таблиц позволяет прибавлять к себе новые строки, сохраняя настроенный лимит на количество строк, например хранить данные о температуре за последние 3 месяца (чтобы файл не увеличивался до бесконечности).

```cpp
void build(sets::Builder& b) {
    b.Plot(H(plot1), "/file_plot1.tbl");
}

// условно вызывается каждую минуту
void everyMinute() {
    TableFileStatic t(&LittleFS, "/file_plot1.tbl", 100);   // макс. 100 строк, будет смещаться при append()

    // инициализация, должна быть вызвана хотя бы один раз после непосредственного создания файла
    t.init(3, cell_t::Uint32, cell_t::Float, cell_t::Int8);

    // добавление в файл
    t.append(sett.rtc.getUnix(), (random(100) - 50) / 2.0, random(-100, 100));
}
```

## Активные виджеты
### Input
Ввод текста. Поддерживает регулярные выражения и комментарий при ошибке ввода:

```cpp
b.Input("");
b.Input("", nullptr, R"(^\d+$)");
b.Input("", nullptr, R"(^\d+$)", "Только цифры");
```

### Pass
Ввод пароля.

### Number
Ввод цифр.

### Color
Колор-пике, принимает и отправляет цвет в 24-битном формате RRGGBB.

### Switch
Переключатель. поддерживает изменение цвета через `updateColor`.

### Date, DateTime
Дата и дата-время, принимает и отправляет unix-секунды (GMT+0). В браузере выводится с учётом часового пояса браузера, также локальный часовой пояс можно указать вручную.

### Time
Принимает и отправляет время в секундах с начала суток независимо от часового пояса.

### Spinner
Как Number, но с кнопками + и -, увеличивают значение на заданный шаг.

### Slider
Слайдер для ввода значений, при движении отправляет значения с указанным в настройках вебморды периодом. Строка результата кликабельная, можно задать значение вручную - подчиняется настройкам min/max/step.

### Slider2
Двойной слайдер, имеет два ID для минимума и максимума, переменных тоже можно подключить две. Обновляется через метод `update2`.

### Select
Всплывающий список выбора опции, принимает и отправляет индекс опции (начиная с `0`):

- Названия опций передаются в виде строки с разделителем `;`: `"option 1;option 2;my option"`
- Опции можно группировать - имя группы заключается в квадратные скобки, список **должен** начинаться с группы: `"[group 1]option 1;option 2[group2]my option"`
- Опции можно скрывать из списка с сохранением индексации - для этого название опции должно начинаться с `~`: `"option 1;~option 2;my option"` (option 2 будет скрыта)
- Группы тоже можно скрывать из списка с сохранением индексации опций - имя группы должно начинаться с `~`: `"[group 1]option 1;option 2[~group2]my option"` (group2 будет скрыта)

> [!TIP]
> Список опций (в том числе в группе) может заканчиваться `;` - пустая опция добавлена не будет. Это позволяет удобно "собирать" список опций, прибавляя к строке опций `"опция;"` (сразу с разделителем на конце), например `"опция;опция;опция;"`

### SelectText
То же самое что Select, но является просто выбором из списка, не хранит выбранный пункт - просто отправляет его. Прочитать можно в `b.build.value` - `if (b.SelectText("", "foo;bar;option")) Serial.println(b.build.value);`.

Например вот так можно сделать выбор файла и открытие при выборе:

```cpp
if (b.SelectText("Select file", sett.fs.listDir())) {
    File f = sett.fs.openRead(b.build.value.c_str());
    Serial.print("File ");
    Serial.print(b.build.value);
    Serial.print(" size ");
    Serial.println(f.size());
}
```

### Tabs
Вкладки, принимает и отправляет индекс вкладки (начиная с 0). Имена вкладок передаются в виде строки с разделителем `;`: `"tab 1;tab 2;my tab"`. Если вкладок слишком много - их можно перемещать пальцем и курсором. Вернёт `true` при клике по вкладке. Далее в программе можно строить билдер по условиям или в `switch` исходя из значения вкладки.

```cpp
void build(sets::Builder& b) {
    static uint8_t tab; // статическая

    if (b.Tabs("Slider;Button;Input", &tab)) {
        // при нажатии перезагружаемся и выходим
        b.reload();
        return;
    }

    if (tab == 0) {
        b.Slider();
    } else if (tab == 1) {
        b.Button();
    } else if (tab == 2) {
        b.Input();
    }
}
```

Пример с использованием `enum` для читаемости кода:

```cpp
void build(sets::Builder& b) {
    enum Tabs : uint8_t {
        Slider,
        Button,
        Input,
    } static tab;

    if (b.Tabs("Slider;Button;Input", (uint8_t*)&tab)) {
        // при нажатии перезагружаемся и выходим
        b.reload();
        return;
    }

    switch (tab) {
        case Tabs::Slider:
            b.Slider();
            break;
            
        case Tabs::Button:
            b.Button();
            break;

        case Tabs::Input:
            b.Input();
            break;
    }
}
```

### Button
Кнопка. Можно обновлять текст на кнопке через `update` и цвет через `updateColor`. функция вернёт `true` при клике по кнопке.

```cpp
if (b.Button()) {
    Serial.println("Click!");
}
```

### ButtonHold
Кнопка с двумя состояниями. Можно обновлять текст на кнопке через `update` и цвет через `updateColor`. Функция вернёт `true` при нажатии и при отпускании. Для определения состояния нужно опросить `b.build.pressed()`.

```cpp
if (b.ButtonHold()) {
    Serial.println(b.build.pressed());
}
```

### Confirm
Всплывающее окно подтверждения. Функция вернёт `true` при любом выборе юзера. Для определения что именно выбрал юзер можно опросить `b.build.value.toBool()` или подключить `bool` переменную. Для вызова окна нужно отправить обновление с id виджета:

- Если отправить обновление `upd.confirm(id)` - вызовется окно с текстом, заданным в билдере
- Если отправить `upd.update(id, текст)` - вызовется окно с текстом, указанным в апдейте, а текст виджета обновится

```cpp
// вызов Confirm по кнопке
bool cfm_f;

void build(sets::Builder& b) {
    bool res;
    if (b.Confirm(kk::conf, "Confirm", &res)) {
        Serial.println(res);
        // Serial.println(b.build.value.toBool());
    }

    if (b.Button()) cfm_f = true;
}

void update(sets::Updater& u) {
    if (cfm_f) {
        cfm_f = false;
        u.confirm(kk::conf);
    }
}
```

### Joystick
Активный джойстик, принимает переменную для хранения позиции типа `sets::Pos`, которая содержит поля `x` и `y` с координатами от `-255` до `255`. Вернёт `true`, если было изменение:

```cpp
sets::Pos pos;

void builder() {
    b.Joystick(pos);
}

void loop() {
    if (pos) {
        Serial.print(pos.x);
        Serial.print(',');
        Serial.println(pos.y);
    }
}
```