## Установка
Библиотека доступна в менеджере библиотек Arduino и PlatformIO

### Зависимости
- [GTL](https://github.com/GyverLibs/GTL) v1.3.0+
- [GyverDB](https://github.com/GyverLibs/GyverDB) v1.3.0+
- [StringUtils](https://github.com/GyverLibs/StringUtils) v1.4.30+
- [GyverHTTP](https://github.com/GyverLibs/GyverHTTP) v1.0.26+
- [BSON](https://github.com/GyverLibs/BSON) v2.1.0+
- [Stamp](https://github.com/GyverLibs/Stamp) v1.4.0+
- [Table](https://github.com/GyverLibs/Table) v1.2.0+
- [StreamIO](https://github.com/GyverLibs/StreamIO) v1.0.5+

> [!TIP]
> При установке из реестра PIO или Arduino IDE все зависимости установятся автоматически!

Дополнительно (ставится вручную):
- [arduinoWebSockets](https://github.com/Links2004/arduinoWebSockets) для версии SettingsGyverWS/SettingsESPWS
- [ESPAsyncTCP](https://github.com/esphome/ESPAsyncTCP) для версии SettingsAsync/SettingsAsyncWS
- [ESPAsyncWebServer](https://github.com/esphome/ESPAsyncWebServer) для версии SettingsAsync/SettingsAsyncWS

<details>
<summary>platformio.ini</summary>

```ini
[env]
framework = arduino
lib_deps =
    GyverLibs/Settings
    # esphome/ESPAsyncWebServer-esphome   ; для версии SettingsAsync/SettingsAsyncWS
    # esphome/ESPAsyncTCP-esphome         ; для версии SettingsAsync/SettingsAsyncWS
    # links2004/WebSockets                ; для версии SettingsGyverWS/SettingsESPWS

[env:d1_mini]
platform = espressif8266
board = d1_mini
upload_speed = 921600
monitor_speed = 115200
monitor_filters = esp8266_exception_decoder, default
build_type = debug
board_build.filesystem = littlefs

[env:esp32dev]
monitor_speed = 115200
platform = espressif32
board = esp32dev
upload_speed = 921600
board_build.filesystem = littlefs

[env:esp32-c3]
monitor_speed = 115200
platform = espressif32
board = esp32dev
board_build.mcu = esp32c3
upload_speed = 2000000
board_build.f_cpu = 80000000L
board_build.filesystem = littlefs
```
</details>

### Пример для PIO
Есть встроенный пример для PlatformIO - `arduino_pio`: просто перетащи его в окно VS Code, библиотека и нужные платформы установятся автоматически

## Введение
### Термины
- **Вебморда** - веб-приложение, которое хостится на веб-сервере ESP и открывается в браузере
- **Виджет** - единица интерфейса вебморды (кнопка, поле ввода, слайдер)

### Как это работает
Вебморда является самостоятельным веб-приложением, написанным на html/css/js. Её файлы минифицированы, сжаты в gz и вшиты в код библиотеки как PROGMEM массив. В библиотеке настроен вебсервер, который отправляет файлы вебморды при заходе на IP платы в браузере. Лёгкий html файл подгружается каждый раз, а скрипты и стили кешируются браузером для ускорения загрузки. Вебморда общается с платой по http: например при загрузке запрашивает пакет с виджетами и прочей информацией.

### Captive portal
Во всех трёх реализациях сервера из коробки настроен DNS для работы как Captive portal - если ESP работает в режиме точки доступа (AP или AP_STA), то при подключении к точке автоматически откроется окно браузера со страницей настроек.

### Приложение для поиска
Позволяет находить устройства с библиотекой в локальной сети и заменяет браузер, вебморда открывается сразу в приложении, кнопка назад возвращает к списку устройств. Чтобы удалить устройство - долгое удержание на нём на смартфоне или правой кнопкой мыши на ПК. Для поиска смартфон/ПК должны быть в одной локальной сети с устройством. В приложении должна быть указана корректная маска подсети (настраивается в роутере). Если в роутере она не менялась - то она там стандартная *255.255.255.0*, как и в приложении по умолчанию.

- [Скачать .apk](https://github.com/GyverLibs/Settings-discover/releases/latest/download/Settings.apk
) последней версии
- [Скачать .html](https://github.com/GyverLibs/Settings-discover/releases/latest/download/index.html) последней версии

## Минимальный пример
```cpp
#include <Arduino.h>

#define WIFI_SSID ""
#define WIFI_PASS ""

#include <SettingsGyver.h>
SettingsGyver sett("My Settings");

void build(sets::Builder& b) {
}

void setup() {
    Serial.begin(115200);
    Serial.println();

    // wifi
    WiFi.mode(WIFI_STA);
    WiFi.begin(WIFI_SSID, WIFI_PASS);
    while (WiFi.status() != WL_CONNECTED) {
        delay(500);
        Serial.print(".");
    }
    Serial.println();
    Serial.print("Connected: ");
    Serial.println(WiFi.localIP());

    // settings
    sett.begin();
    sett.onBuild(build);
}

void loop() {
    sett.tick();
}
```

## База данных
Библиотека интегрирована с [GyverDB](https://github.com/GyverLibs/GyverDB) - относительно быстрой базой данных для хранения данных любого типа. В данном случае она используется для хранения "настроек" - значений виджетов вебморды: Settings автоматически читает и обновляет данные в БД. Рекомендуется изучить как работать с БД на странице описания GyverDB по ссылке выше. При использовании `GyverDBFile` база данных будет автоматически записываться в файл при изменениях с браузера, а файловая система позаботится об оптимальном износе Flash памяти.

При запуске нужно **инициализировать** БД, указав ключи и соответствующие им начальные значения и типы. Эти значения будут записаны только в том случае, если запись в БД ещё не существует, т.е. при первом запуске и при добавлении новых записей. В то же время автоматическое обновление БД работает только для существующих записей, т.е. Settings будет работать только с сущестующими ячейками и не создаст новых.

Минимальный пример:

```cpp
#include <LittleFS.h>
#include <GyverDBFile.h>
GyverDBFile db(&LittleFS, "/data.db");  // создать БД в файле

#include <SettingsGyver.h>
SettingsGyver sett("My Settings", &db);   // подключить БД

// объявить хэш-ключи БД через макрос
DB_KEYS(
    keys,
    input,
    slider
);

void setup() {
    // подключение к WiFi...

    // settings
    sett.begin();

    // запуск файловой системы
#ifdef ESP32
    LittleFS.begin(true); // format on fail
#else
    LittleFS.begin();
#endif

    // запуск БД и чтение из файла
    db.begin();
    
    // инициализация БД начальными значениями
    db.init(keys::input, "text");
    db.init(keys::slider, 30);

    // или через макрос инициализации
    // DB_INIT(
    //     db,
    //     (keys::input, "text"),
    //     (keys::slider, 30)
    // );

    // чтение из БД
    Serial.println(db[keys::input]);
    int sld = db[keys::slider];
}

void loop() {
    sett.tick();    // вызывать в loop
}
```

### Несколько БД
Можно использовать несколько баз данных, например одна для сохраняемых в память настроек, вторая для "временных" настроек, которые не нужно сохранять при перезагрузке: `GyverDBFile` сохраняет в файл, а обычная `GyverDB` - нет, живёт чисто в оперативной памяти. Переключаться между БД нужно в билдере таким образом, чтобы после смены БД шли только виджеты с ключами из этой БД. Например

```cpp
GyverDBFile db_flash(&LittleFS, "/data.db");
GyverDB db_ram;

void build(sets::Builder& b) {
    settings.attachDB(&db_ram);
    b.Input("input2"_h, "...");

    settings.attachDB(&db_flash);
    b.Input("input1"_h, "...");
}
```

> После выхода из билдера нужно оставлять подключенной ту БД, для которой нужны автоматические обновления, система не сможет обновляться одновременно с нескольких БД. Также нужно оставлять последней подключенной БД, которая пишет на флешку, чтобы система автоматически вызывала её тикер.

## Файловый менеджер
Библиотека нативно поддерживает бортовую флэш-память с файловой системой `LittleFS`, список файлов выводится в меню при нажатии на кнопку в углу справа. Файлы можно изменять, удалять, скачивать и загружать. Для корректной работы нужно самостоятельно запустить файловую систему в блоке `setup`, если она нужна:

```cpp
void setup() {
#ifdef ESP32
    LittleFS.begin(true); // format on fail
#else
    LittleFS.begin();
#endif
}
```

### SD карта
Библиотека также поддерживает внешнюю SD карту в паре со встроенной флешкой - в файловом менеджере файлы с SD карты выводятся начиная с пути `/sd/`: это **виртуальный путь** - на самой карте они хранятся без него. Если загрузить через менеджер файл по пути `/sd/file.txt`, то на самой SD карте он будет записан просто как `file.txt`.

В программе также используется виртуальный путь - это позволяет работать с файлами не выбирая вручную файловую систему - библиотека сделает это сама. Например для вывода графиков:

```cpp
void build(sets::Builder& b) {
    b.Plot("/plot2.csv");     // из flash
    b.Plot("/sd/plot2.csv");  // из SD
}
```

Для запуска SD карты нужно:

```cpp
// подключить библиотеку
#ifdef ESP8266
#include <SDFS.h>
#else  // ESP32
#include <SD.h>
#endif

void setup() {
    // wifi, settings...

#ifdef ESP8266
    LittleFS.begin();

    // ESP8266 SPI
    // MOSI     MISO    CLK     CS
    // D7(13)   D6(12)  D5(14)  D2(4) default!
    SPI.begin();

    // карта подключена к SPI(MOSI/MISO/CLK) + CS на пине D8 (по умолчанию там D4)
    SDFSConfig cfg(D8);
    SDFS.setConfig(cfg);

    SDFS.begin();
    // sett.fs.setFS(LittleFS, SDFS);  // подключение обеих FS
    sett.fs.sd.setFS(SDFS); // подключение SD
#else
    LittleFS.begin(true);

    // ESP32 SPI
    // xSPI     MOSI    MISO    CLK     CS
    // HSPI     13      12      14      15
    // VSPI     23      19      18      5
    SPI.begin();

    // карта подключена к VSPI (по умолч. SPI=VSPI)
    SD.begin(5);

    // sett.fs.setFS(LittleFS, SD);  // подключение обеих FS
    sett.fs.sd.setFS(SD); // подключение SD
#endif
}
```

После этого карта доступна в программе и в вебморде.

### Список файлов
Реализован удобный вывод списка файлов:

```cpp
String s;

// sett.fs.flash.listDir(s);    // вывести список файлов flash
// sett.fs.sd.listDir(s);       // вывести список файлов sd (без префикса /sd/)
// sett.fs.listDir(s);          // вывести общий список файлов
```

Список выводится с разделением по новой строке - `\n`. Это позволяет например передать список в виджет `Select`:

```cpp
void build(sets::Builder& b) {
    // селект с файлами
    String s;
    sett.fs.flash.listDir(s);
    b.Select("", s);
}
```

Для определения выбранного пути нужно парсить список по разделителю, например встроенным парсером `Text`:

```cpp
void build(sets::Builder& b) {
    static uint8_t file;

    String s;
    sett.fs.flash.listDir(s);
    
    if (b.Select("", s, &file)) {
        // выводим выбранный файл при выборе
        String sel = Text(s).getSub(file, '\n').toString();
        Serial.println(sel);
    }
}
```