## Билдер и виджеты
Пакет с **виджетами** собирается устройством в **билдере** - функция в программе, которая вызывается, когда приходит запрос от вебморды. Внутри билдера нужно вызвать методы виджетов в том порядке, в котором они должны находиться в вебморде:

```cpp
// минимальный код
SettingsGyver sett;

// билдер
void build(sets::Builder& b) {
    // b.Input(...);
    // b.Button(...);
}

void setup() {
    // подключение к WiFi...

    sett.begin();
    sett.onBuild(build);
}

void loop() {
    sett.tick();
}
```

### ID виджета
У всех виджетов есть вариант функции с ID и без ID. ID виджета нужен для:
- Работа с подключенной базой данных на чтение и запись значений
- Отправка обновлений на виджет
- Разбор действий отдельно от вывода виджетов, чтобы разделить UI и обработку

ID в библиотеке задаётся числом, точно так же как в GyverDB. Для читаемости кода и подсказок IDE удобнее всего задавать ID ключами `DB_KEYS`, как и для базы данных. **Виджеты и база данных - единое целое**, виджет берёт и пишет значение в БД, а мы можем потом прочитать это значение в другом месте в программе.

> Если ID не задан (**анонимный виджет**) - он будет присваиваться библиотекой автоматически (только для активных виджетов). Автоматический ID - это число от `UINT32_MAX`, уменьшается на 1 с каждым вызовом. Если автоматический ID совпадёт с каким-то из вручную заданных - в вебморде высветится ошибка `Duplicated ID`.

```cpp
DB_KEYS(
    kk,
    my_inp,
    button
);

void build(sets::Builder& b) {
    b.Input("My input");                // без ID
    b.Input("my_inp"_h, "My input");    // хэш-строка
    b.Input(SH("my_inp"), "My input");  // хэш-строка
    b.Input(H(my_inp), "My input");     // хэш-строка
    b.Input(kk::my_inp, "My input");    // GyverDB-хэш
}
```

### Взаимодействие с виджетами
Есть несколько способов взаимодействия с виджетами, т.е. отправки и получения значений:
- У виджета можно задать ID, по которому библиотека будет автоматически читать и писать данные в базу данных GyverDB при загрузке вебморды и получения новых значений с неё соответственно. Получить значение можно из любого места в программе, обратившись к БД
- К виджету можно подключить обычную переменную - библиотека будет читать из неё значение и писать при изменениях с вебморды. К БД в этом случае не будет обращений даже при заданном ID
- У виджета без ID и переменной будет значение по умолчанию (0 или пустая строка). Новое значение с вебморды никуда не запишется, но можно прочитать его в момент получения из инфо о билде
- Активный виджет (значение можно менять из вебморды, функция виджета возвращает `bool`) возвращает `true` при изменении значения пользователем и при клике по кнопке

```cpp
String str;
char cstr[20];

void build(sets::Builder& b) {
    // виджет без id и начального значения
    // При установке с вебморды получаем значение напрямую
    if (b.Input("My input")) {
        Serial.println(b.build.value);
    }

    // виджет без id с привязанной String-строкой
    // при установке с вебморды значение запишется в строку
    b.Input("My input", &str);
    b.Input("My input", AnyPtr(cstr, 20));  // для char-массивов

    // виджет с id без привязанной переменной
    // будет работать с базой данных по указанному ключу
    b.Input("my_inp"_h, "My input");

    // действие возвращается независимо от наличия id
    if (b.Button()) Serial.println("btn 1");
    if (b.Button("my_btn2"_h)) Serial.println("btn 2");
}
```

#### Инфо о билде
```cpp
void build(sets::Builder& b) {
    // можно узнать, было ли действие по виджету
    if (b.build.isAction()) {
        Serial.print("Set: 0x");
        Serial.print(b.build.id, HEX);
        Serial.print(" = ");
        Serial.println(b.build.value);
    }
}
```

#### Разделение UI и действий
```cpp
void build(sets::Builder& b) {
    // вывод UI
    b.Input("my_inp"_h, "My input");
    b.Button("my_btn"_h, "My button");

    // обработка действий
    switch (b.build.id) {
        case "my_inp"_h:
            Serial.print("input: ");
            Serial.println(b.build.value);
            break;

        case "my_btn"_h:
            Serial.println("btn click");
            break;
    }
}
```

> Если ID виджета задан и переменная не привязана - будет использоваться БД. Если привязана переменная - будет использоваться она

### Динамические виджеты
Виджеты собираются линейно, вызов функции виджета добавляет его в вебморду. Это означает, что виджеты можно выводить и динамически, особенно удобно это работает с автоматическим id. Например:

```cpp
int numbers[5];

void build(sets::Builder& b) {
    // вывод в цикле
    for (int i = 0; i < 5; i++) {
        b.Input();
    }
    
    // обработка действий также будет работать
    for (int i = 0; i < 5; i++) {
        if (b.Input()) Serial.println(b.build.value);
    }
    
    // массив number с привязанными переменными
    for (int i = 0; i < 5; i++) {
        b.Number(String("number #") + i, &numbers[i]);
    }

    // обработка и действий
    for (int i = 0; i < 5; i++) {
        if (b.Number(String("number #") + i, &numbers[i])) {
            Serial.print(String("number #") + i + ": ");
            Serial.println(numbers[i]);
        }
    }

    // можно и так
    for (int i = 0; i < 5; i++) {
        b.Number(String("number #") + i, &numbers[i]);

        if (b.wasSet()) {
            Serial.print(String("number #") + i + ": ");
            Serial.println(numbers[i]);
            b.clearSet();
        }
    }
}
```

Также можно динамически скрывать виджеты, например по флагу:

```cpp
void build(sets::Builder& b) {
    if (flag) {
        b.Input();
        b.Slider();
        // ...
    }
}
```

Частым сценарием является открытие группы настроек с активацией режима, это можно сделать так:

```cpp
void build(sets::Builder& b) {
    if (b.Switch()) {
        b.reload(); // перезагрузить вебморду по клику на свитч
    }

    // здесь flag должен быть прочитан из БД или переменной
    if (flag) {
        b.Input();
        b.Slider();
        // ...
    }
}
```

Например с БД

```cpp
DB_KEYS(
    kk,
    mode_sw
);

void build(sets::Builder& b) {
    // запись в БД и перезагрузка
    if (b.Switch(kk::mode_sw)) b.reload();

    // чтение из БД
    if (db[kk::mode_sw]) {
        b.Input();
        b.Slider();
        // ...
    }
}
```

### Иконки лейблов
Можно использовать emoji, они неплохо смотрятся в меню. Например с [удобного сайта](https://symbl.cc/ru/emoji/)

```cpp
b.Input(kk::intw, "🔈Громкость");
```