## Динамические задачи
Все типы задач можно создавать динамически через `new` и удалять через `delete`. Важно понимать особенности работы с динамической памятью в C/C++, откуда она берётся и куда девается, как происходит фрагментация и чем это грозит. При неграмотном использовании динамических задач программа может перестать работать спустя какое то время.

```cpp
void setup() {
    new LoopTicker([](){
        if (...) delete LP.thisTask();
    });
}
```

При удалении задачи она автоматически убирается из диспетчера задач. Есть два паттерна создания и удаления задач.

### Отложенное удаление
Если задача поддерживает статус выхода, то можно убрать её из диспетчера через remove, а удалить из памяти уже в статусе Exit. Это будет максимально безопасно, т.к. статус Exit вызывается только из loop, без рекурсий и возможности сломать программу:

```cpp
new LoopTicker([](){
    switch (LP.thisState()) {
        case tState::Setup:
            break;

        case tState::Loop:
            if (...) LP.removeThis();
            // if (...) LP.remove(LP.thisTask());     // или так
            // if (...) LP.thisTask()->removeLoop();  // или так
            break;

        case tState::Exit:
            delete LP.thisTask();   // удаление из памяти
            // здесь уже нельзя обращаться к this-
            break;
    }
});
```

Для задач-таймеров можно включить статусы, если это нужно.

### Мгновенное удаление
Можно удалить задачу сразу "на месте" - в этом случае **вызова со статусом Exit не будет**, даже если задача его поддерживает:

```cpp
void setup() {
    Serial.begin(115200);
    
    new LoopTimer(500, [](){
        Serial.println("Timer");
        delete LP.thisTask();
    });
    
    new LoopTicker([](){
        Serial.println("Ticker");
        delete LP.thisTask();
    });
}
```

Особенности мгновенного удаления:

- Можно удалять только **текущую задачу** или **другую задачу вне обработчиков**, т.е. откуда-то из программы, вне задач
- Удаление задачи из обработчика другой задачи (например по ID) может сломать программу. Для такого сценария используем отложенное удаление через remove
- После удаления задачи очевидно нельзя к ней обращаться, т.е. к `LP.thisTask()`. Лучше после `delete` сразу выходить через `return`

```cpp
void setup() {
    Serial.begin(115200);

    new LoopTicker("ticker", []() {
        if (LP.thisExit()) delete LP.thisTask();
    });

    new LoopTimer(500, []() {
        // удалить себя - ок
        delete LP.thisTask();

        // удалить отсюда другую задачу - нельзя
        // delete LP["ticker"];

        // убрать через remove - можно
        // но задача должна сама удалиться в своём Exit
        // LP.remove(LP["ticker"]);
    });

    // удалить отсюда - можно
    // вызова со статусом Exit не будет
    delete LP["ticker"];
}
```

Если нужно безопасное удаление (первого типа) для задачи, которая не поддерживает статус `Exit`, то вместо неё нужно использовать тип `LoopTask`. А для таймера `LoopTimer` эти статусы можно включить в конструкторе.

## Задачи с данными
У всех задач есть варинт, позволяющий подключить обработчик с данными, указатель на данные передаётся при созданини объекта или в процессе работы. Данные могут быть любого типа, тип указывается как параметр шаблона. Это позволяет прокинуть в обработчик задачи какие то данные, чтобы при вызове обработчика он имел к ним доступ. Например создадим тикер, который меняет переменную каждый тик:

```cpp
int val = 0;

LoopTickerData<int> ticker(&val, [](int* data){
    (*data)++;
});
```

Пример имеет мало смысла, ведь переменная глобальная и находится в области видимости обработчика. Интереснее будет так:

```cpp
void setup() {
    new LoopTickerData<int>(new int(123), [](int* data){
        (*data)++;
    });
}
```

Здесь у задачи есть **свои личные данные**, с которыми она вызывается. Это позволяет очень просто описывать различную сложную логику, сделать которую без Looper'а будет довольно трудно. Например таймер с заданным периодом и количеством срабатываний:

```cpp
void setup() {
    new LoopTimerData<int>(300, new int(5), [](int* counter) {
        Serial.print("Timer ");
        Serial.println(*counter);
        if (!--(*counter)) delete LP.thisTask();
    });
}
```

Можно прокинуть в обработчик любые данные, например таймер со счётчиком и какими то данными:

```cpp
void setup() {
    struct tdata_t {
        int count;
        const char* text;
    };

    new LoopTimerData<tdata_t>(200, new tdata_t{5, "hello:"}, [](tdata_t* data) {
        Serial.print(data->text);
        Serial.println(data->count);

        if (!--data->count) {
            delete data;
            delete LP.thisTask();
        }
    });
}
```
