
## События
Диспетчер позволяет отправлять события задачам двумя способами:
- `sendEvent` - отправить прямо сейчас, из текущего контекста
- `pushEvent` - сохранить в стек событий и отправить из главного loop

> По умолчанию Looper имеет стек отложенных событий на 3 ячейки, т.е. можно отложить 3 события в рамках одной задачи и всех вложенных в неё рекурсивных вызовов. Размер стека настраивается в `platform.h`

Задача может принять событие, если:
- Поддерживает вызов со статусом Event
- Подключена к диспетчеру
- Не находится в состояниях Setup/Exit
- Не отключена через `disable()`
- Не находится в ожидании `LP.delay()` (во избежание рекурсий)
- Имеет ненулевой id
- Отправителем является другая задача
- Совпадает ID получателя или это широковещательное событие

```cpp
LP_LISTENER_("lisn", []() {
    Serial.println("EVENT!");

    // данные события, void*
    LP.eventData();

    // задача, которая отправила событие
    // например выведем ID
    Serial.println(LP.eventSource()->id());

    // событие было отправлено всем
    if (LP.eventBroadcast()) {
    }
});

void setup() {
    // сразу прилетит в обработчик
    LP.sendEvent("lisn");

    // будет отправлено в loop
    LP.pushEvent("lisn");
}
void loop() {
    LP.loop();
}
```

Это очень важно например в следующих ситуациях:

- Событие активирует долгую и блокирующую выполнение задачу. В то же время оно отправляется из места программы, где нельзя долго ждать, например в обработчике ответа сервера. Будет более правильно отложить событие, закончить текущую задачу и вызвать целевую долгую задачу. Таким образом, событие выполняет роль флага
- Отправка события на долгую задачу из прерывания. Гораздо правильнее отложить такую задачу и вызвать позже
- Отправка события в задачу, которая будет активно потреблять память. Отправка из задачи, которая сама потребляет много памяти, которую нет возможности освободить прямо сейчас. Будет логично отложить событие и вызвать его из loop, когда память освободится

### Источник события
В примере выше показано, как обратиться к задаче, которая отправила событие (`eventSource()`). С этим нужно быть аккуратнее, т.к. источник может быть нулевым указателем `nullptr`, если событие было отправлено не из другой задачи, а просто откуда-то из программы. Также все отложенные `pushEvent` события будут иметь нулевой источник, так как отправляются из loop.

```cpp
LP_LISTENER_("lisn", []() {
    if (LP.eventSource()) {
        Serial.println(LP.eventSource()->id());
    }
});
```

### Широковещательные события
Диспетчер может отправить событие всем задачам, для этого нужно отправить его на id `0` или использовать макрос `LP_BROADCAST`. Обработчик в свою очередь знает, было ли событие широковещательным:

```cpp
LP_LISTENER_("lisn", []() {
    // событие было отправлено всем
    if (LP.eventBroadcast()) { }
});

void setup() {
    LP.sendEvent(LP_BROADCAST);
}
```

### Перехват событий
Диспетчер задач позволяет перехватывать события на свой обработчик `onEvent` (функция вида `void f(hash_t id)`). В этот обработчик будут поступать все события, отправленные из других задач или из программы. В этом обработчике корректно работают геттеры `eventData`, `eventSource` и `eventBroadcast`:

```cpp
void setup() {
    // подключаем обработчик для перехвата событий
    LP.onEvent([](hash_t id){
        Serial.println(id);
        Serial.println((const char*)LP.eventData());
    });

    // тест
    LP.sendEvent(1, (void*)"send");
    LP.pushEvent(2, (void*)"push");
}
```

Таким образом для классического механизма "флагов запуска из loop" необязательно создавать отдельные `LoopListener`-задачи, можно ограничиться одним общим обработчиком перехвата событий, используя ID события как имя флага. Такие события можно отправлять из прерываний и прочих нагруженных участков программы (через push) и они будут обрабатываться в loop после завершения тяжёлой задачи. Пример обработки хэш-строк:

```cpp
// где то в программе
LP.pushEvent(LPH("print_flag"));

// ......
void setup() {
    // подключаем обработчик для перехвата событий
    LP.onEvent([](hash_t id){
        switch (id) {
            case LPH("print_flag"):
                // print code
                break;
            
            case LPH("reconnect_flag"):
                // reconnect code
                break;
        }
    });
}
```

Точно так же отлично будет работать и с "базой" ID из прошлой главы:

```cpp
// IDs.h
enum IDs : hash_t {
    my_task = 1,
    timer0,
    web_listener,
};

// main
void setup() {
    LP.onEvent([](hash_t id){
        switch (id) {
            case IDs::my_task:
                break;
            
            case IDs::web_listener:
                break;
        }
    });

    // 
    LP.pushEvent(IDs::web_listener);
}
```

### Отправка данных
К событию можно подключить данные, т.е. отправить событие с данными. Библиотека предлагает универсальный способ - прикрепить данные через указатель `void*`, т.е. пользователь может отправлять в событие абсолютно любые данные. Тут важно понимать, что если эти данные - динамические, то отправить событие сразу будет безопасно. Если событие отложено - нужно быть увереным, что данные ещё будут находиться в памяти на момент вызова отложенного события. Также принимающий обработчик должен знать, данные какого типа ему будут отправлять.

> Для работы с передачей данных рекомендуется хорошо разбираться в особенностях статических и динамических данных в Си

```cpp
LP_LISTENER_("lisn0", []() {
    Serial.println("lisn0: ");
    Serial.println((const char*)LP.eventData());
});

LP_LISTENER_("lisn1", []() {
    Serial.println("lisn1: ");
    int* data = (int*)LP.eventData();
    Serial.println(*data);
    delete data;
});

LP_LISTENER_("lisn2", []() {
    Serial.println("lisn2: ");
    int* data = (int*)LP.eventData();
    Serial.println(*data);
});

LP_LISTENER_("lisn3", []() {
    Serial.println("lisn3: ");
    int* data = (int*)LP.eventData();
    Serial.println(*data);
});

void setup() {
    // статическая строка
    LP.sendEvent("lisn0", (void*)"hello!");

    // динамические данные. Обработчик сам их удалит
    // тут можно pushEvent
    LP.sendEvent("lisn1", new int(12345));

    // динамические данные с отправкой и удалением
    // нельзя делать pushEvent!
    int* data = new int(123);
    LP.sendEvent("lisn2", data);
    delete data;

    // статические локальные данные
    // нельзя делать pushEvent!
    int val = 4321;
    LP.sendEvent("lisn3", &data);
}
```