## База
### Как это работает
Глобально определён объект `LP` (и его синоним `Looper`) - диспетчер задач, который должен вызываться в основном цикле программы, в Arduino - в `void loop`. При каждом вызове диспетчер вызывает все подключенные задачи и проверяет таймеры. Минимальный код использования библиотеки:

```cpp
#include <Looper.h>

void setup() {
}

void loop() {
    LP.loop();
}
```

При создании новой задачи она автоматически добавляется в диспетчер. Создадим тикер:

```cpp
#include <Looper.h>

LoopTicker myTicker([]() {
    // этот код будет вызываться из loop
});

void setup() {
}

void loop() {
    LP.loop();
}
```

### Окружение
Библиотека разработана для однопоточного окружения, в многопоточных средах корректная работа не гарантируется. Например в RTOS управлять состоянием задач из других потоков (отличных от того, в котором крутится `LP.loop()`).

### Сущности и термины
- **Задача** - сам объект задачи с подключенным обработчиком
- **Обработчик** - подключенная к задаче функция, которая вызывается библиотекой в нужный момент
- `LP`/`Looper` - диспетчер задач, глобально объявленный объект библиотеки, который выполняет всю работу
- `LoopTask` - базовая задача, которая может добавляться в диспетчер. В явном виде они не используются в программе, на их основе работает всё остальное:
  - `LoopTicker` - задача-тикер, обработчик вызывается постоянно в loop
  - `LoopTimer` - задача-таймер, обработчик вызывается с заданным периодом из loop
  - `LoopListener` - задача-обработчик событий, обработчик вызывается только при получении события
  - `LoopThread` - задача-поток с асинхронными задержками и ожиданиями
- **Текущая задача** - задача, в обработчике которой сейчас выполняется код программы

### Обработчик задачи
Обработчик может быть как внешней, так и лямбда-функцией. Принципиальной разницы нет, но лямбда лаконичнее:

```cpp
// 1. Внешняя функция
void myCallback() {
    // код тикера 1
}
LoopTicker myTicker1(myCallback);
```
```cpp
// 2. Лямбда-функция
LoopTicker myTicker2([]() {
    // код тикера 2
});
```

### Макросы
Для более удобного создания задач предусмотрены макросы (см. документацию). Статические макросы генерируют уникальное имя для переменной, а также автоматически применяют хэш-функцию к строковому ID (строка не хранится в программе, хэш считается компилятором). Например:

```cpp
// статический тикер без ID
// развернётся в что-то типа static LoopTicker __loop_obj_5([]() { })
LP_TICKER([]() {
});

// статический таймер на 500мс с ID
LP_TIMER_("tick", 500, []() {
});
LP_TIMER_(IDs::timer0, 500, []() {
});
```

### Текущая задача
Диспетчер вызывает задачу и хранит указатель и прочую информацию о ней, эту информацию можно получить внутри обработчика задачи. У диспетчера есть набор методов, которые начинаются с `this` (см. документацию) - они позволяют обратиться к текущей задаче. Например:

```cpp
LoopTicker myTicker([]() {
    // получить указатель на текущую задачу типа LoopTicker*
    LP.thisTask() == &myTicker; // true

    if (LP.thisSetup()) {
        // текущий статус - Setup
    }

    // отключить текущую задачу
    LP.thisTask()->disable();
});
```

Так как диспетчер выдаёт указатель на текущую задачу - нет нужды где то самому хранить указатель на задачу или делать удобное имя переменной. Например, можно создавать динамические задачи и точно так же с ними работать (подробнее о динамических задачах в другой главе):

```cpp
void setup() {
    new LoopTicker([]() {
        // вывести id
        Serial.println(LP.thisTask()->id());

        // отключить
        LP.thisTask()->disable();

        // удалить эту задачу
        if (condition) delete LP.thisTask();
    });
}
```

### ID задачи
У каждой задачи есть ID - уникальный номер, по которому задачу можно получить из диспетчера для настройки, а также отправить в задачу событие. Библиотека использует хэш-строки для хранения ID. Хэш - целое число, имеет тип `hash_t` (псевдоним `size_t`). Для задания ID можно использовать:
- Просто цифры
- Строки `const char*`, они будут преобразованы в хэш во время выполнения программы
- Строки внутри функции `LPH` - хэш будет посчитан на этапе компиляции (строка не попадёт в память). Например `LoopTicker(LPH("my_ticker"), ...)`
- Нулевой ID `0` (по умолчанию в конструкторах без ID) считается не заданным, такую задачу нельзя получить из диспетчера по ID и отправить ей событие

Использование `LPH("string")` строк как ID конечно хорошо, но в большой программе придётся помнить или каждый раз искать имя задачи/события. Чтобы IDE могла подсказать весь список имеющихся ID, можно создать "базу" таких ID, например в виде `enum`, **начиная с 1**. Объявить в отдельном файле и подключить во все файлы проекта:

```cpp
// IDs.h
enum IDs : hash_t {
    my_task = 1,    // начиная с 1
    timer0,
    web_listener,
};

LP_TICKER_(IDs::my_task, []() {
});

// main
void setup() {
    LP.pushEvent(IDs::my_task);
    LP.sendEvent(IDs::web_listener);
}
```

### Статусы задач
У диспетчера есть набор статусов, с которыми могут вызываться задачи. Каждый тип задач имеет свой набор статусов:

- `tState::Loop` - обычный обход и вызов активных задач. Может вызываться из разных мест
- `tState::Setup` - добавление задачи в диспетчер. Вызывается только из главного loop. Вызов происходит до вызовов с остальными статусами, что важно для манипуляций с динамическими данными
- `tState::Exit` - удаление задачи из диспетчера. Вызывается только из главного loop. Вызов происходит последним из остальных статусов
- `tState::Event` - событие. Может вызываться из разных мест. Приходит только активным задачам

Стандартная поддержка статусов для типов задач:

|                 | `LoopTicker` | `LoopThread` | `LoopTimer` | `LoopListener` |
| --------------- | :----------: | :----------: | :---------: | :------------: |
| `tState::Setup` |      ✔       |              |             |                |
| `tState::Loop`  |      ✔       |      ✔      |      ✔      |                |
| `tState::Exit`  |      ✔       |              |             |                |
| `tState::Event` |      ✔       |      ✔      |             |       ✔        |

- Поддержку статусов `Setup`+`Exit` и `Event` можно включить у всех типов задач через методы `enableStates()` и `enableEvents()`

> С версии 1.2 изменена логика вызова со статусами `Setup`/`Exit` - теперь они вызываются только из главного цикла во избежание рекурсий и неопределённого поведения программы

### Добавление/удаление
При добавлении задачи (также при создании) она помечается как добавленная и будет вызвана из главного цикла со статусом `Setup`, если поддерживает его.

Когда задача убирается из диспетчера (например `LP.removeThis()`) - она помечается на удаление и будет фактически удалена только при следующем вызове loop, это исключает рекурсивные вызовы и возможность сломать диспетчер, как это было в предыдущих версиях библиотеки.

Задача может удалить как себя, так и другую задачу:

```cpp
LP_TIMER_("timer", 100, []() {
    Serial.println("hello");
});

LP_TIMER(500, []() {
    LP["timer"]->removeLoop();  // удалить timer
    LP.removeThis();            // удалить себя
    Serial.println("end");
});
```

После удаления сам объект задачи остаётся в памяти, о динамических задачах читай в отдельной главе.

## Задачи
### LoopTicker
Тикер вызывается постоянно и принимает все статусы. Обработать их можно так:

```cpp
// в условии
LP_TICKER([]() {
    if (LP.thisSetup()) {
        // код инициализации
    }
    // код loop
});


// свитч
LP_TICKER([]() {
    switch (LP.thisState()) {
        case tState::Setup:
            Serial.println("TICK SETUP");
            break;

        case tState::Loop:
            Serial.println("TICK LOOP");
            break;

        case tState::Exit:
            Serial.println("TICK EXIT");
            break;

        case tState::Event:
            Serial.println("TICK EVENT");
            break;
    }
});
```

### LoopTimer
Периодический таймер вызывается с заданным периодом и по умолчанию только со статусом `Loop`. Можно обратиться к текущему таймеру через `thisTimer()`:

```cpp
// остановить через 500 мс
LP_TIMER(500, []() {
    Serial.println("timeout!");
    LP.thisTimer()->stop();
});

// выполнять циклически
LP_TIMER(1000, []() {
    Serial.println("hello!");
});
```

Можно включить обработку статусов `Setup`+`Exit`, а вызов именно самого события таймера будет в статусе `Loop`:

```cpp
LP_TIMER(500, []() {
        switch (LP.thisState()) {
            case tState::Setup:
                Serial.println("SETUP");
                break;

            case tState::Loop:
                Serial.println("TIMER");
                break;

            case tState::Exit:
                Serial.println("EXIT");
                break;
        } }, true, true);   // <-- true
```

### LoopListener
Обработчик событий по умолчанию вызывается только с `Event`, но можно включить ему и `Setup`+`Exit`:

```cpp
LP_LISTENER_("lisn", []() {
    Serial.println("EVENT!");
});

LP_LISTENER_("lisn", []() {
    LP.thisState(); // Setup, Event, Exit
}, true);   // <-- true
```

### LoopThread
Рассмотрены подробнее в следующей главе. Принимают только события `Loop` и `Event`, т.к. поток можно организовать через свой цикл `while` и асинхронные задержки - статусы `Setup`+`Exit` ему не нужны:

```cpp
LP_THREAD([]() {
    // "Setup"
    while (true) {
        //LP_WAIT_EVENT();    // ждать события
        if (LP.thisEvent());  // проверка события
    }
    // "End"
});
```

## Взаимодействие
Задачи могут взаимодействовать между собой напрямую или по ID, также это может делать и основная программа. Например, можно получить задачу и отключить её:

```cpp
// через getTask
LP.getTask("my_task")->disable();

// доступ через []
Looper["my_task"]->disable();
```

Например, одна задача может включать и выключать другую:

```cpp
LP_TIMER(1000, []() {
    Looper["tmr"]->toggle();
});

LP_TIMER_("tmr", 100, []() {
    Serial.println(millis());
});
```

### Задержки
Обычный Arduino-`delay` работает как ожидается - тормозит полностью всю программу и все задачи.

Если внутри задачи вызвать `LP.delay(x)` - внутри будет вызываться `LP.loop()` на протяжении указанного времени. Это рекурсивный вызов, который корректно работает на любую вложенность (пока не кончится память) - вызвавшая задача приостанавливается на время задержки, а остальные продолжают работу.

Если во время задержки одной задачи какая-то другая задача запустит задержку на большее время, чем осталось у первой, то задержка первой задачи увеличится на эту разницу и так далее в глубину рекурсии. Такие задержки нужно использовать аккуратно! Бывает полезно, если задаче нужно подождать буквально несколько мс и городить ради этого таймер и машину состояний нецелесообразно.

Здесь тикер тормозит всю программу, но во время этой задержки диспетчер работает и таймер будет выводить свой "hello 1":

```cpp
LP_TIMER(500, []() {
    Serial.println("hello 1");
});

LP_TICKER([]() {
    Serial.println("hello 2");
    LP.delay(5000);
});
```

## Многофайловый проект
Глобальный диспетчер задач позволяет создавать задачи в отдельных исполняемых файлах и не тащить их в main, но они всё равно будут вызываться из общего loop в main. Например

```cpp
// main.cpp
#include <Arduino.h>
#include <Looper.h>

void setup() {
}
void loop() {
    LP.loop();
}
```
```cpp
// main1.cpp
#include <Arduino.h>
#include <Looper.h>

LP_TICKER([]() {
    // loop code
});
```
```cpp
// main2.cpp
#include <Arduino.h>
#include <Looper.h>

LP_TICKER([]() {
    // loop code
});
```
