## LoopThread
Внутри этой задачи-потока можно использовать "асинхронные" задержки и ожидания условий и событий, почти как в RTOS системах, что позволяет задачам выполняться по-настоящему одновременно и параллельно. Потоки реализованы на переходах к метке, без сохранения стека функции как в RTOS, поэтому не используют дополнительной памяти, но имеют ряд ограничений и особенностей. Создать поток можно вручную как тикер `LoopTicker`, но нужно дополнительно вызвать `LP_THREAD_BEGIN()` в начале и `LP_THREAD_END()` в конце обработчика:

```cpp
LoopThread th_manual([]() {
    LP_THREAD_BEGIN();

    // код

    LP_THREAD_END();
});
```

Либо использовать макрос, который сам создаст имя для функции и обернёт код в begin-end:

```cpp
LP_THREAD({
    // код
});

LP_THREAD_("thread_id", {
    // код
});
```

> Внутри обработчика точно так же можно обращаться к задаче через `thisTask` и делать всё то же самое, что с `LoopTicker` (остановить, исключить из loop, отправить-поймать событие и так далее).

Для управления потоком используются специальные управляющие макросы:

```cpp
// начало и конец потока для создания вручную
LP_THREAD_BEGIN()
LP_THREAD_END()

// перезапустить поток (начать выполнение с начала)
LP_RESTART()

// выйти из потока и потом вернуться в эту точку
LP_EXIT()

// асинхронно ждать время в потоке, мс
LP_DELAY(ms)

// асинхронно ждать условия в потоке
LP_WAIT(cond)

// асинхронно ждать Looper-события в потоке
LP_WAIT_EVENT()

// ждать семафор
LP_SEM_WAIT(sem)

// освободить семафор
LP_SEM_SIGNAL(sem)
```

### Выход из потока
В любой момент можно покинуть поток и отдать выполнение другим потокам, на следующем круге диспетчера задач управление вернётся к потоку и продолжится после точки выхода. Полезно например если нужно "размазать" какие то долгие сложные вычисления по времени и выполнять в промежутках другие потоки:

```cpp
LP_THREAD({
    Serial.println("test 1");
    LP_EXIT();
    Serial.println("test 2");
    LP_EXIT();
});
```

### Ожидание времени
Поток может асинхронно ждать указанное время, это ожидание **не блокирует выполнение программы**:

```cpp
LP_THREAD({
    LP_DELAY(500);
    Serial.println("test!");
});
```

Например вот так выглядит классический blink:

```cpp
LP_THREAD({
    digitalWrite(LED_BUILTIN, 1);
    LP_DELAY(500);
    digitalWrite(LED_BUILTIN, 0);
    LP_DELAY(500);
});
```

Поток также позволяет обернуть часть кода в бесконечный цикл, чтобы организовать setup-loop как это обычно делается в таких случаях:

```cpp
LP_THREAD({
    pinMode(LED_BUILTIN, OUTPUT);  // выполнится один раз

    while (true) {
        digitalWrite(LED_BUILTIN, 1);
        LP_DELAY(500);
        digitalWrite(LED_BUILTIN, 0);
        LP_DELAY(500);
    }
});
```

> Примечание: оставшееся время до продолжения выполнения учитывается в `Looper.nextTimerLeft()`, как и у задач-таймеров

### Ожидание условия
Поток также может асинхронно ждать наступления указанного условия, например:

```cpp
bool flag = 0;

LP_THREAD({
    LP_WAIT(flag);  // ждём пока флаг станет 1

    Serial.println("flag!");
    flag = 0;
});
```

Если где-то в программе поднимется flag, то выполнение кода продолжится.

### Ожидание события
Также есть готовый макрос ожидания события:

```cpp
LP_THREAD_("thread", {
    Serial.println("wait event");
    LP_WAIT_EVENT();    // ждём, когда кто то отправит событие
    Serial.println("event!");
});

// отправляем событие по таймеру
LP_THREAD({
    LP_DELAY(2000);
    Looper.pushEvent("thread");
});

// или так
LP_TIMER(1000, []() {
    Looper.pushEvent("thread");
});
```

## Ограничения
Реализация потоков очень простая и лёгкая, поэтому есть ограничения. По сути все управляющие макросы создают метку перехода, а весь код находится внутри `switch`, это означает что:
- После макроса ожидания будет выход из функции без сохранения стека, т.е. объявленные выше локальные переменные потеряют значения
- Нельзя создавать внутри потока локальные переменные, которые "живут" между вызовами управляющих макросов
- Для создания "стека" используем статические переменные внутри функции
- Нельзя использовать управляющие макросы внутри `switch`

Например цикл с задержкой делать вот так **нельзя**:
```cpp
LP_THREAD({
    for (int i = 0; i < 10; i++) {
        Serial.println(i);
        LP_DELAY(500);
        // тут мы выйдем из функции и потеряем значение i
    }
});
```

Вот так - **можно** и нужно:

```cpp
LP_THREAD({
    static int i;

    for (i = 0; i < 10; i++) {
        Serial.println(i);
        LP_DELAY(500);
    }
});
```

Вот так - **нельзя**:
```cpp
LP_THREAD({
    switch (foo) {
        case 3:
            // управляющий макрос внутри своего switch
            // может привести к конфликту с глобальным switch
            // и код зависнет тут
            LP_DELAY(500);
            break;
    }
});
```

## Семафоры
Реализована простенькая поддержка семафоров для синхронизации потоков. Например есть два потока, один забирает семафор на какое то время для условного получения значения, второй поток блокируется до освобождения семафора и выводит это значение:

```cpp
LP_SEM sem = 1; // начальное значение 1, чтобы какой-нибудь поток мог забрать себе семафор
int val;

LP_THREAD({
    LP_SEM_WAIT(sem); // ждём тут, пока семафор занят
    // тут мы забираем семафор себе, другие потоки будут заблокированы

    Serial.println(val);

    LP_SEM_SIGNAL(sem); // освобождаем семафор
});

LP_THREAD({
    LP_SEM_WAIT(sem); // ждём тут, пока семафор занят
    // тут мы забираем семафор себе, другие потоки будут заблокированы

    // имитация бурной деятельности
    LP_DELAY(500);
    val = random(100);

    LP_SEM_SIGNAL(sem); // освобождаем семафор
});
```

### Потоки с данными
Тут всё по аналогии с другими типами задач

```cpp
// вручную
LoopThreadData<int> thread_data(new int(3), [](int* data) {
    LP_THREAD_BEGIN();

    LP_DELAY(500);
    Serial.println(*data);
    *data += 1;

    LP_THREAD_END();
});

// макрос
LP_THREAD_DATA(int, new int(3), int* data, {
    LP_DELAY(500);
    Serial.println(*data);
    *data += 1;
});
```