## База
### Как это работает
Глобально определён объект `Looper` - диспетчер задач, который должен вызываться в основном цикле программы `loop`. При каждом вызове диспетчер вызывает все подключенные задачи и проверяет таймеры. Минимальный код использования библиотеки:

```cpp
#include <Looper.h>

void setup() {
}

void loop() {
    Looper.loop();
}
```

При создании новой задачи она автоматически добавляется в стек диспетчера, при удалении (вызов деструктора) - сама убирается из него. Создадим тикер:

```cpp
#include <Looper.h>

LoopTicker myTicker([]() {
    // этот код будет вызываться из loop
});

void setup() {
}

void loop() {
    Looper.loop();
}
```

### Сущности и термины
- Обработчик - подключенная к задаче функция, которая вызывается библиотекой в нужный момент
- `Looper` - диспетчер задач, глобально объявленный объект библиотеки, который выполняет всю работу
- `LoopTask` - базовая задача, которая может добавляться в диспетчер. В явном виде они не используются в программе, на их основе работает всё остальное:
  - `LoopTicker` - задача-тикер, обработчик вызывается постоянно в loop
  - `LoopTimer` - задача-таймер, обработчик вызывается с заданным периодом из loop
  - `LoopListener` - задача-обработчик событий, обработчик вызывается только при получении события
  - `LoopThread` - задача-поток с асинхронными задержками и ожиданиями
- Текущая задача - задача, в обработчике которой сейчас выполняется код программы

### Обработчик задачи
Обработчик - функция, которая подключается к задаче. Может быть как внешней, так и лямбда-функцией. Принципиальной разницы нет, но лямбда лаконичнее и удобнее:

```cpp
// 1. Внешняя функция
void myCallback() {
    // код тикера 1
}
LoopTicker myTicker1(myCallback);

// 2. Лямбда-функция
LoopTicker myTicker2([]() {
    // код тикера 2
});
```

### Текущая задача
Диспетчер вызывает задачу и хранит указатель и прочую информацию о ней, эту информацию можно получить внутри обработчика задачи. Задача, в обработчике которой в данный момент выполняется код, называется **текущей задачей**. У диспетчера есть набор методов, которые начинаются с `this` (см. документацию) - они позволяют обратиться к текущей задаче. Например:

```cpp
LoopTicker myTicker([]() {
    // получить указатель на текущую задачу
    Looper.thisTask() == &myTicker; // true

    if (Looper.thisSetup()) {
        // текущий статус - Setup
    }

    // отключить текущую задачу
    Looper.thisTask()->disable();
});
```

Так как диспетчер выдаёт указатель на текущую задачу - нет нужды где то самому хранить указатель на задачу или делать удобное имя переменной. Например можно создавать динамические задачи и точно так же с ними работать (подробнее о динамических задачах позже):

```cpp
void setup() {
    new LoopTicker([]() {
        // вывести id
        Serial.println(Looper.thisTask()->id());

        // отключить
        Looper.thisTask()->disable();

        // удалить эту задачу
        if (condition) delete Looper.thisTask();
    });
}
```

### ID задачи
У каждой задачи есть ID - уникальный номер, по которому задачу можно получить из диспетчера для настройки, а также отправить в задачу событие. Библиотека использует хэш-строки для хранения ID. Хэш - целое число, имеет тип `hash_t` (псевдоним `size_t`). Для задания ID можно использовать:
- Просто цифры
- Строки `const char*`, они будут преобразованы в хэш во время выполнения программы
- Строки внутри функции `LPH` - хэш будет посчитан на этапе компиляции (строка не попадёт в память). Например `LoopTicker(LPH("my_ticker"), ...)`
- Нулевой ID `0` (по умолчанию в конструкторах без ID) считается не заданным, такую задачу нельзя получить из диспетчера по ID и отправить ей событие

Использование `LPH("string")` строк как ID конечно хорошо, но в большой программе придётся помнить или каждый раз искать имя задачи/события. Чтобы IDE могла подсказать весь список имеющихся ID, можно создать "базу" таких ID, например в виде `enum`. Объявить в отдельном файле и подключить во все файлы проекта:

```cpp
// IDs.h
enum IDs : hash_t {
    my_task = 1,
    timer0,
    web_listener,
};

// main
void setup() {
    Looper.pushEvent(IDs::my_task);
    Looper.sendEvent(IDs::web_listener);
}
```

### Макросы
Для более удобного создания задач предусмотрены макросы (см. документацию). Статические макросы генерируют уникальное имя для переменной, а также автоматически применяют хэш-функцию к строковому ID (строка не хранится в программе, хэш считается компилятором). Например:

```cpp
// статический тикер без ID
LP_TICKER([]() {
});

// статический таймер на 500мс с ID
LP_TIMER_("tick", 500, []() {
});
LP_TIMER_(IDs::timer0, 500, []() {
});
```

### Статусы задач
У диспетчера есть набор статусов, с которыми могут вызываться задачи. Каждый тип задач имеет свой стандартный набор статусов, но их можно гибко настроить при создании задачи или в процессе работы:
- `tState::Setup` - вызывается при добавлении задачи в диспетчер. Если задача создана до первого прохода цикла loop (т.е. глобально или статически), то вызов с этим статусом будет выполнен позже из loop, при первом вызове `Looper.loop()`. Если задача создана позже в процессе работы программы - вызов будет выполнен сразу же из контекста создания задачи
- `tState::Loop` - вызывается из loop
- `tState::Exit` - вызывается при удалении задачи из диспетчера. Если задача удаляет сама себя - вызов со статусом выхода будет рекурсивным
- `tState::Event` - вызывается при поступлении события

Стандартные настройки для всех типов задач:
|                 | `LoopTicker` | `LoopThread` | `LoopTimer` | `LoopListener` |
| --------------- | :----------: | :----------: | :---------: | :------------: |
| `tState::Setup` |      ✔       |      ✔       |             |                |
| `tState::Loop`  |      ✔       |      ✔       |      ✔      |                |
| `tState::Exit`  |      ✔       |      ✔       |             |                |
| `tState::Event` |      ✔       |      ✔       |             |       ✔        |

> Иными словами: тикер вызывается со всеми статусами, таймер только с `Loop`, а обработчик событий - только с событиями

> Настроить поведение задачи можно в конструкторе (см. документацию), флаг `states` отвечает за `Setup` и `Exit`, флаг events - за `Event`

## Задачи
### LoopTicker
Тикер вызывается постоянно и принимает все статусы. Обработать их можно так:

```cpp
// в условии
LP_TICKER([]() {
    if (Looper.thisSetup()) {
        // код инициализации
    }
    // код loop
});


// свитч
LP_TICKER([]() {
    switch (Looper.thisState()) {
        case tState::Setup:
            Serial.println("TICK SETUP");
            break;

        case tState::Loop:
            Serial.println("TICK LOOP");
            break;

        case tState::Exit:
            Serial.println("TICK EXIT");
            break;

        case tState::Event:
            Serial.println("TICK EVENT");
            break;
    }
});
```

### LoopTimer
Периодический таймер вызывается с заданным периодом. Можно обратиться к текущему таймеру через `thisTimer()`:

```cpp
// остановить через 500 мс
LP_TIMER(500, []() {
    Serial.println("timeout!");
    Looper.thisTimer()->stop();
});

// выполнять циклически
LP_TIMER(1000, []() {
    Serial.println("hello!");
});
```

### LoopListener
Обработчик событий по умолчанию реагирует только на события:

```cpp
LP_LISTENER_("lisn", []() {
    Serial.println("EVENT!");
});
```

### LoopThread
Рассмотрены отдельно в следующей главе.

## Взаимодействие
Задачи могут взаимодействовать между собой напрямую или по ID, также это может делать и основная программа. Например можно получить задачу и отключить её:

```cpp
// через getTask
Looper.getTask("my_task")->disable();

// есть доступ через []
Looper["my_task"]->disable();
```

Например одна задача может включать и выключать другую:

```cpp
LP_TIMER(1000, []() {
    Looper["tmr"]->toggle();
});

LP_TIMER_("tmr", 100, []() {
    Serial.println(millis());
});
```

### Задержки
Если внутри задачи вызвать `Looper.delay(x)` - внутри будет вызываться `Looper.loop()` на протяжении указанного времени. Это рекурсивный вызов, который корректно работает на любую вложенность (пока есть память) - задача приостанавливается и не вызывается во время рекурсивного вызова диспетчера.

## Многофайловый проект
Глобальный диспетчер задач позволяет создавать задачи в отдельных исполняемых файлах и не тащить их в main, но они всё равно будут вызываться из общего loop в main. Например

```cpp
// main.cpp
#include <Arduino.h>
#include <Looper.h>

void setup() {
}
void loop() {
    Looper.loop();
}
```
```cpp
// main1.cpp
#include <Arduino.h>
#include <Looper.h>

LP_TICKER([]() {
    // loop code
});
```
```cpp
// main2.cpp
#include <Arduino.h>
#include <Looper.h>

LP_TICKER([]() {
    // loop code
});
```
