## Ближайший таймер и сон
Диспетчер может вывести остаток времени до срабатывания самого ближайшего таймера `LoopTimer`. Если весь проект построен на периодических задачах и должен спать в промежутках между ними, то тут как раз поможет данная фича:

```cpp
void loop() {
    Looper.loop();
    sleep_function(Looper.nextTimerLeft());
}
```

здесь `sleep_function` - абстрактная функция, которая принимает время сна МК в миллисекундах, а `Looper.nextTimerLeft()` - количество миллисекунд до ближайшего срабатывания таймера, либо `0`, если активных таймеров нет

## SimpleTimer
В библиотеке есть класс таймера, который можно использовать отдельно от задач. При срабатывании таймер возвращает `true`, его нужно вручную остановить (таймаут) или перезапустить (интервал). Предусмотрены и готовые функции для этого:

```cpp
// свои таймеры внутри тикера
LP_TICKER([]() {
    static SimpleTimer tmrManual(500);
    static SimpleTimer tmrTimeout(1000);
    static SimpleTimer tmrInterval(1000);

    if (tmrManual) {
        Serial.println("timer");
        tmrManual.restart();
    }

    if (tmrTimeout.timeout()) {
        Serial.println("timeout");
    }

    if (tmrInterval.interval()) {
        Serial.println("interval");
    }
});
```

## Свои классы
Библиотека предлагает удобный API для создания своих классов-задач. Нужно наследовать нужный класс (`LoopTickerBase`/`LoopTimerBase`/`LoopListenerBase`) и реализовать в нём метод `void exec()` - он будет вызываться как задача. Например:

```cpp
class LoopHello : public LoopTimerBase {
    public:
    LoopHello(uint32_t ms) : LoopTimerBase(ms) {}

    void exec() {
        Serial.println("hello!");
    }
};

LoopHello hello(300);
```

Теперь при создании объекта он автоматически будет добавляться в диспетчер задач, а при удалении убираться. Так как мы наследовали таймер - будет периодически вызываться метод `exec`, который в данном примере просто выводит строку в порт каждые 300 мс. Точно таким же образом можно делать удобные классы для кнопок/датчиков/связи, которые будут автоматически интегрироваться в диспетчер задач. Вот например простейший класс кнопки:

```cpp
class LoopButton : public LoopTickerBase {
   public:
    LoopButton(uint8_t pin) : _pin(pin) {
        pinMode(pin, INPUT_PULLUP);
    }

    void exec() {
        if (!digitalRead(_pin)) {
            if (!_flag) {
                _flag = true;
                Serial.println("button pressed!");
            }
        } else {
            if (_flag) _flag = false;
        }
    }

   private:
    uint8_t _pin;
    bool _flag;
};

LoopButton btn(2);
```

При создании кнопка добавится в Looper и начнёт опрашиваться, при нажатии будет выводить текст в порт. Такая кнопка с выводом в порт не имеет смысла - нужно протащить событие нажатия в основную программу. Добавим в класс свой подключаемый обработчик, чтобы и дальше писать в событийном стиле.

### Свой обработчик
В библиотеке есть удобный макрос для создания обработчиков - `LP_MAKE_CALLBACK(name, return_type, arg_types...)` Он создаёт typedef указателя на функцию, с которым удобно работать. Например

```cpp
LP_MAKE_CALLBACK(MyCallback, void, int, float);
// Создаст обработчик для функции вида void func(int a, float b)
```

Для AVR используется просто указатель на функцию, для остальных платформ - `std::function`, она позволяет подключать обработчики внутри класса лямбда-функцией. Добавим обработчик клика в наш класс кнопки. Сделаем это в private секции, т.к. вне класса он нам не пригодится:

```cpp
class LoopButton : public LoopTickerBase {
    // создание типа обработчика
    LP_MAKE_CALLBACK(ClickCallback, void);

   public:
    LoopButton(uint8_t pin, ClickCallback callback) : _pin(pin), _cb(callback) { // инициализация коллбэка
        pinMode(pin, INPUT_PULLUP);
    }

    void exec() {
        if (!digitalRead(_pin)) {
            if (!_flag) {
                _flag = true;
                _cb();  // вызов коллбэка
                // Serial.println("button pressed!"); // вместо этого
            }
        } else {
            if (_flag) _flag = false;
        }
    }

    // функция ничего не делает, см. ниже
    void foo() {}

   private:
    uint8_t _pin;
    ClickCallback _cb;  // храним коллбэк
    bool _flag;
};
```

Теперь к кнопке можно подключить обработчик клика:

```cpp
LoopButton btn(2, []() {
    Serial.println("button pressed!");
});
```

Передавать `this` самой кнопки в обработчик нет нужды, т.к. диспетчер может выдать нам указатель на объект внутри обработчика текущей задачи, можно использовать удобную функцию `thisTaskAs<T>` для каста в свой тип:

```cpp
LoopButton btn(2, []() {
    // например отключим кнопку (базовый API задач)
    Looper.thisTask()->disable();

    // вот так доступны методы самого класса кнопки
    Looper.thisTaskAs<LoopButton>()->foo();
});
```

И кнопку естественно можно создать динамически

```cpp
void setup() {
    new LoopButton(2, []() {
        // code...
    });
}
```

## Другие платформы
Библиотеку можно использовать на отличных от Arduino платформах, для этого нужно отредактировать под себя файл `platform.cpp`, реализовать свой таймер аптайма и при желании yield-функцию, которая будет вызываться в цикле между задачами диспетчера.