## Динамические задачи
Все типы задач можно создавать динамически через `new` и удалять через `delete`. Важно понимать особенности работы с динамической памятью в C/C++, откуда она берётся и куда девается, как происходит фрагментация и чем это грозит. При неграмотном использовании динамических задач программа может перестать работать спустя какое то время.

```cpp
void setup() {
    new LoopTicker([](){
        if (...) delete Looper.thisTask();
    });
}
```

При удалении задачи она автоматически убирается из диспетчера задач. Если это тикер - он будет рекурсивно вызван ещё раз со статусом `Exit`, поэтому **код выше - некорректный**. Лучше использовать разделение по статусам, чтобы после удаления код оказался в блоке выхода:

```cpp
void setup() {
    new LoopTicker([](){
        switch (Looper.thisState()) {
            case tState::Setup:
                Serial.println("TICK SETUP");
                break;

            case tState::Loop:
                Serial.println("TICK LOOP");
                if (...) delete Looper.thisTask();
                break;

            case tState::Exit:
                Serial.println("TICK EXIT");
                // здесь окажемся после delete задачи
                break;

            case tState::Event:
                Serial.println("TICK EVENT");
                break;
        }
    });
}
```

Удалять так таймеры - можно, они не вызываются со статусом выхода. Таймер вызовется через 500мс и удалится из памяти:

```cpp
void setup() {
    new LoopTimer(500, [](){
        delete Looper.thisTask();
    });
}
```

## Задачи с данными
У всех задач есть варинт, позволяющий подключить обработчик с данными, указатель на данные передаётся при созданини объекта или в процессе работы. Данные могут быть любого типа, тип указывается как параметр шаблона. Это позволяет прокинуть в обработчик задачи какие то данные, чтобы при вызове обработчика он имел к ним доступ. Например создадим тикер, который меняет переменную каждый тик:

```cpp
int val = 0;

LoopTickerData<int> ticker(&val, [](int* data){
    (*data)++;
});
```

Пример имеет мало смысла, ведь переменная глобальная и находится в области видимости обработчика. Интереснее будет так:

```cpp
void setup() {
    new LoopTickerData<int>(new int(123), [](int* data){
        (*data)++;
    });
}
```

Здесь у задачи есть **свои личные данные**, с которыми она вызывается. Это позволяет очень просто описывать различную сложную логику, сделать которую без Looper'а будет довольно трудно. Например таймер с заданным периодом и количеством срабатываний:

```cpp
void setup() {
    new LoopTimerData<int>(300, new int(5), [](int* data) {
        Serial.println("Timer");
        if (!--(*data)) delete Looper.thisTask();
    });
}
```

Можно прокинуть в обработчик любые данные, например таймер со счётчиком и какими то данными:

```cpp
void setup() {
    struct tdata_t {
        int count;
        const char* text;
    };

    new LoopTimerData<tdata_t>(200, new tdata_t{5, "hello:"}, [](tdata_t* data) {
        Serial.print(data->text);
        Serial.println(data->count);

        if (!--data->count) {
            delete data;
            delete Looper.thisTask();
        }
    });
}
```
