
## События
Диспетчер позволяет отправлять события задачам. Чтобы принять событие, задача должна иметь ненулевой ID. Диспетчер может отправлять события двумя способами:
- `sendEvent` - отправить прямо сейчас, из текущего контекста
- `pushEvent` - сохранить в стек событий и отправить из loop сразу после выхода из текущей задачи

> По умолчанию Looper имеет стек отложенных событий на 3 ячейки, т.е. можно отложить 3 события в рамках одной задачи и всех вложенных в неё рекурсивных вызовов. Размер стека настраивается в `platform.h`

```cpp
LP_LISTENER_("lisn", []() {
    Serial.println("EVENT!");

    // данные события
    Looper.eventData();

    // задача, которая отправила событие
    // например выведем ID
    Serial.println(Looper.eventSource()->id());

    // событие было отправлено всем
    if (Looper.eventBroadcast()) {
    }
});

void setup() {
    // сразу прилетит в обработчик
    Looper.sendEvent("lisn");

    // будет отправлено в loop
    Looper.pushEvent("lisn");
}
void loop() {
    Looper.loop();
}
```

Это очень важно например в следующих ситуациях:
- Событие активирует долгую и блокирующую выполнение задачу. В то же время оно отправляется из места программы, где нельзя долго ждать, например в обработчике ответа сервера. Будет более правильно отложить событие, закончить текущую задачу и вызвать целевую долгую задачу. Таким образом событие выполняет роль флага
- Отправка события на долгую задачу из прерывания. Гораздо правильнее отложить такую задачу и вызвать позже
- Отправка события в задачу, которая будет активно потреблять память. Отправка из задачи, которая сама потребляет много памяти, которую нет возможности освободить прямо сейчас. Будет логично отложить событие и вызвать его из loop, когда память освободится

### Источник события
В примере выше показано, как обратиться к задаче, которая отправила событие (`eventSource()`). С этим нужно быть аккуратнее, т.к. источник может быть нулевым указателем `nullptr`, если событие было отправлено не из другой задачи, а просто откуда-то из программы. Также все отложенные `pushEvent` события будут иметь нулевой источник, так как отправляются из loop.

```cpp
LP_LISTENER_("lisn", []() {
    if (Looper.eventSource()) {
        Serial.println(Looper.eventSource()->id());
    }
});
```

### Широковещательные события
Диспетчер может отправить событие всем задачам, для этого нужно отправить его на id `0` или использовать макрос `LP_BROADCAST`. Обработчик в свою очередь знает, было ли событие широковещательным:

```cpp
LP_LISTENER_("lisn", []() {
    // событие было отправлено всем
    if (Looper.eventBroadcast()) { }
});

void setup() {
    Looper.sendEvent(LP_BROADCAST);
}
```

### Перехват событий
Диспетчер задач позволяет перехватывать события на свой обработчик `onEvent` (функция вида `void f(hash_t id)`). В этот обработчик будут поступать все события, отправленные из других задач или из программы. В этом обработчике корректно работают геттеры `eventData`, `eventSource` и `eventBroadcast`:

```cpp
void setup() {
    // подключаем обработчик для перехвата событий
    Looper.onEvent([](hash_t id){
        Serial.println(id);
        Serial.println((const char*)Looper.eventData());
    });

    // тест
    Looper.sendEvent(1, (void*)"send");
    Looper.pushEvent(2, (void*)"push");
}
```

Таким образом для классического механизма "флагов запуска из loop" необязательно создавать отдельные `LoopListener`-задачи, можно ограничиться одним общим обработчиком перехвата событий, используя ID события как имя флага. Такие события можно отправлять из прерываний и прочих нагруженных участков программы (через push) и они будут обрабатываться в loop после завершения тяжёлой задачи. Пример обработки хэш-строк:

```cpp
// где то в программе
Looper.pushEvent(LPH("print_flag"));

// ......
void setup() {
    // подключаем обработчик для перехвата событий
    Looper.onEvent([](hash_t id){
        switch (id) {
            case LPH("print_flag"):
                // print code
                break;
            
            case LPH("reconnect_flag"):
                // reconnect code
                break;
        }
    });
}
```

Точно так же отлично будет работать и с "базой" ID из прошлой главы:

```cpp
// IDs.h
enum IDs : hash_t {
    my_task,
    timer0,
    web_listener,
};

// main
void setup() {
    Looper.onEvent([](hash_t id){
        switch (id) {
            case IDs::my_task:
                break;
            
            case IDs::web_listener:
                break;
        }
    });

    // 
    Looper.pushEvent(IDs::web_listener);
}
```

### Отправка данных
К событию можно подключить данные, т.е. отправить событие с данными. Библиотека предлагает универсальный способ - прикрепить данные через указатель `void*`, т.е. пользователь может отправлять в событие абсолютно любые данные. Тут важно понимать, что если эти данные - динамические, то отправить событие сразу будет безопасно. Если событие отложено - нужно быть увереным, что данные ещё будут находиться в памяти на момент вызова отложенного события. Также принимающий обработчик должен знать, данные какого типа ему будут отправлять.

> Для работы с передачей данных рекомендуется хорошо разбираться в особенностях статических и динамических данных в Си

```cpp
LP_LISTENER_("lisn0", []() {
    Serial.println("lisn0: ");
    Serial.println((const char*)Looper.eventData());
});

LP_LISTENER_("lisn1", []() {
    Serial.println("lisn1: ");
    int* data = (int*)Looper.eventData();
    Serial.println(*data);
    delete data;
});

LP_LISTENER_("lisn2", []() {
    Serial.println("lisn2: ");
    int* data = (int*)Looper.eventData();
    Serial.println(*data);
});

LP_LISTENER_("lisn3", []() {
    Serial.println("lisn3: ");
    int* data = (int*)Looper.eventData();
    Serial.println(*data);
});

void setup() {
    // статическая строка
    Looper.sendEvent("lisn0", (void*)"hello!");

    // динамические данные. Обработчик сам их удалит
    // тут можно pushEvent
    Looper.sendEvent("lisn1", new int(12345));

    // динамические данные с отправкой и удалением
    // нельзя делать pushEvent!
    int* data = new int(123);
    Looper.sendEvent("lisn2", data);
    delete data;

    // статические локальные данные
    // нельзя делать pushEvent!
    int val = 4321;
    Looper.sendEvent("lisn3", &data);
}
```

### Вложенные события, рекурсия
Библиотека корректно обрабатывает все рекурсивные вызовы, например внутри задачи отправляется событие, которое в своём обработчике отправляет ещё одно событие. Но есть исключения:
1. Нельзя удалять текущую задачу из вложенных вызовов других задач
2. Нельзя отправлять события после удаления текущей задачи

#### 1 исключение
```cpp
LP_LISTENER_("lisn1", []() {
    Looper.sendEvent("lisn2");
});

LP_LISTENER_("lisn2", []() {
    // удалить lisn1, из которого мы сюда и попали
    Looper.getTask("lisn1")->removeLoop();
});

void setup() {
    Looper.sendEvent("lisn1");
}
```

#### 2 исключение
```cpp
LP_LISTENER_("lisn1", []() {
    Looper.thistask()->removeLoop();

    // отправка после удаления текущей
    Looper.sendEvent("lisn2");
});

LP_LISTENER_("lisn2", []() {
});

void setup() {
    Looper.sendEvent("lisn1");
}
```

#### Правило хорошего тона
Вообще ничего не делать после удаления текущей задачи, это действие должно быть последним в обработчике. Можно вызвать `return` после него для пущей безопасности.