## Документация
### Коллбэки
- `TaskCallback` - функция вида `void f()`
- `DataCallback` - функция вида `void f(T* data)`
- `LooperCallback` - функция вида `void f(hash_t id)`

### Настройки
В файле platform.h можно настроить некоторые параметры библиотеки для уменьшения веса и переноса на отличные от Arduino платформы. Например можно отключить поддержку ID задач для уменьшения веса самих задач и отключить поддержку отправки событий для уменьшения веса всей библиотеки в целом

### Макросы
```cpp
// создать обработчик
LP_MAKE_CALLBACK(name, type, ...)

// отправить событие всем (передать как ID)
LP_BROADCAST

// отправить событие, id - const char*, будет применена LPH
LP_SEND_EVENT(id, data)

// отложить событие, id - const char*, будет применена LPH
LP_PUSH_EVENT(id, data)

// создать статически с указанием типа
LP_MAKE(type, ...)
LP_MAKE_(id, type, ...)

// создать динамически с указанием типа
LP_NEW(type, ...)
LP_NEW_(id, type, ...)

// создать статический обработчик событий с указанием типа
LP_LISTENER_(id, func)

// создать статический тикер
LP_TICKER(func)
LP_TICKER_(id, func)

// создать статический таймер
LP_TIMER(ms, func)
LP_TIMER_(id, ms, func)

// создать статический поток
LP_THREAD(body)
LP_THREAD_(id, body)
LP_THREAD_DATA(T, data, data_arg, body)
LP_THREAD_DATA_(id, T, data, data_arg, body)

// начало и конец потока для создания вручную
LP_THREAD_BEGIN()
LP_THREAD_END()

// асинхронно ждать события в потоке
LP_WAIT_EVENT(cond)

// асинхронно ждать условия в потоке
LP_WAIT(cond)

// асинхронно ждать время в мс в потоке
LP_SLEEP(ms)

// освободить семафор
LP_SEM_SIGNAL(sem)

// ждать семафор
LP_SEM_WAIT(sem)
```

### `Looper`
Набор функций для управления диспетчером и задачами

```cpp
// ========= SYSTEM =========
// вызывать в loop
void loop();

// вызвать сигнал setup у всех задач
void restart();

// количество задач в стеке
uint16_t length();

// время до срабатывания следующего таймера
uint32_t nextTimerLeft();

// аналог delay, внутри него вызывается loop
void delay(uint32_t ms);

// добавить задачу
void add(LoopTask* task);

// убрать задачу
void remove(LoopTask* task);

// получить указатель на задачу по id
LoopTask* getTask(hash_t id);
LoopTask* getTask(const char* id);

// получить указатель на задачу по id
LoopTask* operator[](hash_t id);
LoopTask* operator[](const char* id);

// получить указатель на задачу по id
LoopTimer* getTimer(hash_t id);
LoopTimer* getTimer(const char* id);

// ========= THIS =========

// указатель на текущую задачу
LoopTask* thisTask();

// указатель на текущую задачу-таймер
LoopTimer* thisTimer();

// указатель на текущую задачу, кастуется в указанный тип
template <typename T>
T* thisTaskAs();

// убрать текущую задачу из loop
void removeThis();

// статус текущей задачи
tState thisState();

// статус текущей задачи - setup
bool thisSetup();

// статус текущей задачи - exit
bool thisExit();

// статус текущей задачи - loop
bool thisLoop();

// статус текущей задачи - событие
bool thisEvent();

// ========= EVENT =========

// данные события текущей задачи
void* eventData();

// задача, которая отправила событие
LoopTask* eventSource();

// событие отправлено всем
bool eventBroadcast();

// отправить событие сейчас
void sendEvent(hash_t id, void* data = nullptr);
void sendEvent(const char* id, void* data = nullptr);

// отложить событие (отправить позже из loop)
bool pushEvent(hash_t id, void* data = nullptr);
bool pushEvent(const char* id, void* data = nullptr);

// подключить перехватчик событий вида void f(hash_t id)
void onEvent(LooperCallback callback);
```

### `LoopTask`
Базовая задача, все типы задач имеют этот функционал

```cpp
// добавить в loop
void addLoop();

// убрать из loop
void removeLoop();

// вызвать обработчик
void exec();

// подключить обработчик
void attach(TaskCallback callback);

// отключить обработчик
void detach();

// id задачи
hash_t id();

// включить задачу
void enable();

// выключить задачу
void disable();

// задача запущена
bool isEnabled();

// включить если выключена и наоборот
void toggle();

// включить обработку событий
void enableEvents();

// отключить обработку событий
void disableEvents();

// включена обработка событий
bool hasEvents();

// включить статусы Setup Exit
void enableStates();

// выключить статусы Setup Exit
void disableStates();

// включены статусы Setup Exit
bool hasStates();

// задача - таймер
bool isTimer();

// задача - тикер
bool isTicker();

// задача - обработчик событий
bool isListener();
```

### `SimpleTimer`
Базовый таймер, наследуется LoopTimer, также может работать самостоятельно

```cpp
SimpleTimer(uint32_t ms = 0);

// перезапустить
void restart();

// перезапустить с новым периодом
void restart(uint32_t ms, uint32_t sec = 0, uint16_t min = 0, uint16_t hour = 0, uint16_t day = 0);

// остановить
void stop();

// форсировать срабатывание
void force();

// запущен
bool running();

// время вышло
bool ready();

// получить период таймера
uint32_t getPeriod();

// время вышло с перезапуском
bool interval();

// время вышло с остановкой
bool timeout();

// осталось времени. 0 также если таймер остановлен
uint32_t left();
```

### LoopTicker
Тикер, наследует `LoopTask`
```cpp
LoopTicker(TaskCallback callback, bool states = true, bool events = true);
LoopTicker(const char* id, TaskCallback callback, bool states = true, bool events = true);
LoopTicker(hash_t id, TaskCallback callback, bool states = true, bool events = true);
```

### LoopTickerBase
Тикер для создания своих классов

```cpp
LoopTickerBase();
LoopTickerBase(const char* id);
LoopTickerBase(hash_t id);

virtual void exec() = 0;
```

### LoopTickerData<T>
Тикер с данными

```cpp
LoopTickerData(T* data, DataCallback callback);
LoopTickerData(const char* id, T* data, DataCallback callback);
LoopTickerData(hash_t id, T* data, DataCallback callback);

// подключить новые данные
void setData(T* data);

// получить данные
T* getData();
```

### LoopTimer
Таймер, наследует `LoopTask` и `SimpleTimer`

```cpp
LoopTimer(uint32_t ms, TaskCallback callback, bool states = false, bool events = false);
LoopTimer(const char* id, uint32_t ms, TaskCallback callback, bool states = false, bool events = false);
LoopTimer(hash_t id, uint32_t ms, TaskCallback callback, bool states = false, bool events = false);

// опрос таймера
void poll();
```

### LoopTimerBase
Таймер для создания своих классов

```cpp
LoopTimerBase(uint32_t ms);
LoopTimerBase(const char* id, uint32_t ms);
LoopTimerBase(hash_t id, uint32_t ms);

// выполняется при срабатывании таймера
virtual void exec() = 0;
```

### LoopTimerData<T>
Таймер с данными

```cpp
LoopTimerData(uint32_t ms, T* data, DataCallback callback);
LoopTimerData(const char* id, uint32_t ms, T* data, DataCallback callback);
LoopTimerData(hash_t id, uint32_t ms, T* data, DataCallback callback);

// подключить новые данные
void setData(T* data);

// получить данные
T* getData();
```

### LoopListener
Обработчик событий, наследует `LoopTask`

```cpp
LoopListener(const char* id, TaskCallback callback, bool states = false);
LoopListener(hash_t id, TaskCallback callback, bool states = false);
```

### LoopListenerBase
Обработчик событий для создания своих классов

```cpp
LoopListenerBase(const char* id);
LoopListenerBase(hash_t id);

// выполняется при получении события
virtual void exec() = 0;
```

### LoopListenerData<T>
Обработчик событий с данными

```cpp
LoopListenerData(const char* id, T* data, DataCallback callback);
LoopListenerData(hash_t id, T* data, DataCallback callback);

// подключить новые данные
void setData(T* data);

// получить данные
T* getData();
```

### LoopThread
Тикер, наследует `LoopTicker`
```cpp
LoopThread(TaskCallback callback, bool states = true, bool events = true);
LoopThread(const char* id, TaskCallback callback, bool states = true, bool events = true);
LoopThread(hash_t id, TaskCallback callback, bool states = true, bool events = true);
```

### LoopThreadData<T>
Поток с данными

```cpp
LoopThreadData(T* data, DataCallback callback);
LoopThreadData(const char* id, T* data, DataCallback callback);
LoopThreadData(hash_t id, T* data, DataCallback callback);

// подключить новые данные
void setData(T* data);

// получить данные
T* getData();
```