#include <GyverINA.h>

/*
   Внимание!!! Пределы измерения напряжения шунта у INA226 = +/- 81.92 мВ
   При использовании модуля INA226 с шунтом 0.1 Ом макс. измеряемый ток будет I ~ 820 мА
   При использовании другого шунта, рекомендуется рассчитать его так, чтобы падение напряжения не превышало 82мВ!

   Пример:
   Макс. ожидаемый ток = 5 A
   Предел падения напряжения на шунте = 80 мВ
   R шунта = 0.08 В / 5 А = 0,016 Ом
   Шунт должен иметь сопротивление 0.016 Ом (16 мОм)
*/

// Создаем обьект: INA226 ina(Сопротивление шунта, Макс. ожидаемый ток, I2c адрес);
// INA226 ina(0x41);              // Стандартные настройки для модуля, но измененный адрес
// INA226 ina(0.05f);             // Стандартный адрес и макс. ток, но другой шунт (0.05 Ом)
// INA226 ina(0.05f, 1.5f);       // Стандартный адрес, но другой шунт (0.05 Ом) и макс. ожидаемый ток (1.5 А)
// INA226 ina(0.05f, 1.5f, 0x41); // Полностью настраиваемый вариант, ручное указание параметров
INA226 ina;                       // Стандартный набор параметров для Arduino модуля (0.1, 0.8, 0x40)

void setup() {
  // Открываем последовательный порт
  Serial.begin(9600);
  Serial.print(F("INA226..."));

  // Проверяем наличие и инициализируем INA226
  if (ina.begin()) {	// ina.begin(4, 5) // Для ESP32/ESP8266 можно указать пины I2C
    Serial.println(F("connected!"));
  } else {
    Serial.println(F("not found!"));
    while (1);
  }

  // Можно веревести в режим сна, вызвав .sleep с аргументом true, чтобы разбудить - вызываем повторно с указанием false
  // ina.sleep(true);   // Усыпить INA226
  // ina.sleep(false);  // Разбудить INA226

  // INA226 имеет возможность встроенной калибровки измерения тока, при помощи специального калибровочного значения
  // После запуска библиотека автоматически рассчитает и запишет калибровочное значение на основе введенных данных
  // Полученное значение можно прочитать, используя метод .getCalibration(); для изменения и/или сохранения в EEPROM
  Serial.print(F("Calibration value: ")); Serial.println(ina.getCalibration());
  // Далее полученное значение можно изменять для подстройки под реальное сопротивление шунта и сохранять в EEPROM
  // Чтобы записать калибровочное значение в INA226 существует метод .setCalibration(value);
  // ina.setCalibration(ina.getCalibration() + 10); // Прочитать-модифицировать-записать калибровочное значение
  // Так же, можно использовать метод .adjCalibration(offset); для подстройки калибровки без непосредственного чтения
  // ina.adjCalibration(10);  // Увеличить калибровочное значение на 10
  // ina.adjCalibration(-20); // Уменьшить калибровочное значение на 20
  // Можно хранить в EEPROM и загружать в INA226 именно смещение калибровки вместо непосредственного значения

  // Для повышения помехозащищенности INA226 имеет возможность настроить время выборки напряжения и тока
  // INA226 будет захватывать "кусок" сигнала выбранной продолжительности, что повысит точность на шумном сигнале
  // По умолчанию выборка занимает 1100 мкс, но может быть изменена методом .setSampleTime(канал, время);
  // Варианты времени выборки см. в таблице (файл INA226.h)
  ina.setSampleTime(INA226_VBUS, INA226_CONV_2116US);   // Повысим время выборки напряжения вдвое
  ina.setSampleTime(INA226_VSHUNT, INA226_CONV_8244US); // Повысим время выборки тока в 8 раз

  // Так же имеется возможность использовать встроенное усреднение выборок
  // Усреднение применяется и для напряжения и для тока и пропорционально увеличивает время оцифровки
  // Рекомендуется на шумной нагрузке, устанавливается методом .setAveraging(кол-во усреднений) (см. таблицу в INA226.h)
  ina.setAveraging(INA226_AVG_X4); // Включим встроенное 4х кратное усреднение, по умолчанию усреднения нет 

  Serial.println("");
}

void loop() {
  // Читаем напряжение
  Serial.print(F("Voltage: "));
  Serial.print(ina.getVoltage(), 3);
  Serial.println(F(" V"));

  // Читаем ток
  Serial.print(F("Current: "));
  Serial.print(ina.getCurrent(), 3);
  Serial.println(F(" A"));

  // Читаем мощность
  Serial.print(F("Power: "));
  Serial.print(ina.getPower(), 3);
  Serial.println(F(" W"));

  // Читаем напряжение на шунте
  Serial.print(F("Shunt voltage: "));
  Serial.print(ina.getShuntVoltage(), 6);
  Serial.println(F(" V"));

  Serial.println("");
  delay(1000);
}
