## Начало работы
### ESP8266/ESP32
Библиотека нативно поддерживает МК ESPxx классом `FastBot2`:
```cpp
#include <FastBot2.h>
FastBot2 bot;
```
Для подключения к WiFi можно использовать типичную конструкцию:

```cpp
void setup() {
    Serial.begin(115200);

    WiFi.begin("WIFI_SSID", "WIFI_PASS");
    while (WiFi.status() != WL_CONNECTED) {
        delay(500);
        Serial.print(".");
    }
    Serial.println("Connected");
}
```

### Другие платформы
Библиотека работает напрямую с `Arduino Client` интерфейсом, поэтому можно подключить любую библиотеку, которая предоставляет к нему доступ (для Ethernet, GPRS и так далее). Для этого используется класс `FastBot2Client`:
```cpp
SomeClient client;

#include <FastBot2Client.h>
FastBot2Client bot(client);
```

Пример с [TinyGSM](https://github.com/vshymanskyy/TinyGSM):
```cpp
// обязательно нужен модем с поддержкой HTTPS
#define TINY_GSM_MODEM_SIM7000SSL
#include <TinyGsmClient.h>
TinyGsm modem(Serial);
TinyGsmClient gsmclient(modem);

#include <FastBot2Client.h>
FastBot2Client bot(gsmclient);
```

### Настройка бота
```cpp
void setup() {
    // ОБЯЗАТЕЛЬНЫЕ
    bot.setToken("token");  // установка токена
    // bot.setToken(F("token")); // любым текстовым способом

    // НЕОБЯЗАТЕЛЬНЫЕ (указаны настройки по умолчанию)
    bot.skipUpdates(-1);    // пропустить все входящие обновления
    bot.setLimit(3);        // максимум сообщений в одном обновлении (экономия памяти)
    bot.setPollMode(fb::Poll::Sync, 4000);  // настройка метода опроса (об этом ниже)
}
```

## Получение обновлений
Под *обновлением* понимаются:
- Сообщения и бизнес сообщения, отправленные/пересланные/отредактированные в личке/группе/канале с ботом
- Опросы
- Реакции, бусты, уведомления о присоединении юзера
- Callback запросы

По умолчанию включены все типы обновлений, но можно выбрать конкретные:
```cpp
// сначала очистить все
bot.updates.clearAll();

// и включить нужные
bot.updates.set(fb::Updates::Type::Message | fb::Updates::Type::ChannelPost);

// можно отключить только ненужные
bot.updates.set(fb::Updates::Type::ShippingQuery | fb::Updates::Type::PreCheckoutQuery);
```

### Обработчик
Для получения обновлений нужно подключить обработчик и вызывать тикер в главном цикле программы:

```cpp
// обработчик
void update(fb::Update& u) {
}

void setup() {
    bot.onUpdate(update);
}

void loop() {
    bot.tick();
}
```

Подробнее разбор обновлений описан в следующей главе документации

> Указатель на текущий объект FastBot (тот, который вызвал коллбэк) можно получить из переменной `fb::Core* fb::thisBot`. Кастовать его в нужный класс и использовать:

```cpp
void update(fb::Update& u) {
    static_cast<FastBot2*>(fb::thisBot)->sendMessage(...);
}
```

### Минимальный код (для esp)
```cpp
#define WIFI_SSID ""
#define WIFI_PASS ""
#define BOT_TOKEN ""

#include <FastBot2.h>
FastBot2 bot;

void update(fb::Update& u) {
}

void setup() {
    // ==============
    Serial.begin(115200);
    WiFi.begin(WIFI_SSID, WIFI_PASS);
    while (WiFi.status() != WL_CONNECTED) {
        delay(500);
        Serial.print(".");
    }
    Serial.println("Connected");
    // ==============

    bot.setToken(BOT_TOKEN);
    bot.onUpdate(update);
    // bot.setPollMode(fb::Poll::Long, 30000);
}

void loop() {
    bot.tick();
}
```

## Метод опроса обновлений
### `Sync` (по умолчанию)
```cpp
bot.setPollMode(fb::Poll::Sync, 4000);
```
#### Обновление
Такой же режим, как в первой версии FastBot: библиотека запрашивает обновления и ждёт ответа внутри `tick()`. При плохой связи может "зависать" внутри `tick()` на ощутимое (не дольше таймаута) время. Ставить период меньше `4000` не рекомендуется - сервер Телеграм "накажет" долгим ответом за слишком частый опрос!

#### Отправка сообщений
Отправка сообщений будет быстрой в любом месте программы, т.к. библиотека не ждёт ответа от сервера.

### `Async`
```cpp
bot.setPollMode(fb::Poll::Async, 4000);
```
#### Обновление
"Асинхронный" режим - библиотека запросит обновления и обработает ответ, когда он придёт. Период опроса - такая же особенность, как в `Sync`.

#### Отправка сообщений
Если отправить сообщение НЕ в обработчике обновления и ВО ВРЕМЯ ожидания ответа с обновлением - библиотека переподключится к серверу, что занимает около 1 секунды и блокирует выполнение программы. Для безопасной отправки из любого места в программе нужно отправлять сообщения, когда библиотека не ждёт ответа - проверить можно по флагу `isPolling()`.

### `Long`
```cpp
bot.setPollMode(fb::Poll::Long, 30000);
```
#### Обновление
Самый правильный способ опроса - асинхронный *long polling*. Период можно ставить побольше - от 20 секунд. Библиотека отправляет запрос и получит ответ либо по указанному таймауту, либо по факту поступления новых сообщений в бота. Таймаут переподключения - по умолчанию 300мс. Это самый быстрый и безопасный способ получения обновлений, сообщения доставляются мгновенно.

#### Отправка сообщений
Отправка сообщений в обработчике обновлений - безопасна, сообщения будут отправлены быстро. Если нужно отправить из другого места программы - по флагу `isPolling()` можно узнать, ждёт ли библиотека ответ сервера. Если отправить пока ждёт - будет переподключение ("зависнет" на ~1 секунду). Также `tick()` вернёт `true`, когда были обработаны обновления, в этом условии безопасно делать отправку.