# Updatable

A lightweight framework for **non-blocking Arduino programming**. Write clean, modular code without `delay()` calls.

## Description

The Updatable library enables true non-blocking programming in Arduino projects. Instead of scattering timing logic throughout your code with `millis()` checks and `delay()` calls, simply inherit from `Updatable` and implement your update logic. Objects automatically register themselves and receive periodic updates with delta time, allowing multiple components to run concurrently without blocking.

## Features

- **Non-blocking by design** - No more `delay()` blocking your program
- **Automatic registration** - Objects register themselves on construction
- **Delta time updates** - Each component receives time elapsed since last update
- **Batch processing** - Update all components with a single call
- **Zero configuration** - Just inherit and override `update()`
- **Debug mode support** - Enable debugging for all instances at once
- **Minimal overhead** - Lightweight and efficient

## Important Limitation

**This library is designed for persistent components only.** Once an `Updatable` object is created, it registers itself automatically and remains registered throughout the program's lifetime. There is no automatic deregistration when objects are deleted (this is intentional to keep the library efficient).

**What this means:**
- ✅ Perfect for components created at startup that live throughout program execution
- ✅ Ideal for global objects, static instances, or objects with program lifetime
- ⚠️ Not suitable for dynamically created/destroyed objects (e.g., with `new`/`delete`)
- ⚠️ Deleting an `Updatable` object will cause undefined behavior on the next update cycle

If you need dynamic lifecycle management, please open an issue on GitHub. I'm open to changes that would support this use case while maintaining efficiency for the common persistent component pattern.

## Dependencies

This library requires the [Vector library by Janelia](https://github.com/janelia-arduino/Vector) for managing the collection of registered instances.

## Installation

1. Install this library through the Arduino Library Manager or manually
2. Install the Vector library dependency from the Arduino Library Manager

## Usage

### The Problem: Blocking Code

Traditional Arduino code blocks execution:

```cpp
void loop() {
  digitalWrite(LED1, HIGH);
  delay(1000);  // ❌ Everything stops here!
  digitalWrite(LED1, LOW);
  delay(1000);

  // Can't do other things while waiting
}
```

### The Solution: Non-blocking with Updatable

```cpp
#include <Updatable.h>

// Create a non-blocking LED blinker
class LEDBlinker : public Updatable {
private:
  int pin;
  unsigned long interval;
  unsigned long elapsed;
  bool state;

public:
  LEDBlinker(int ledPin, unsigned long blinkInterval)
    : pin(ledPin), interval(blinkInterval), elapsed(0), state(false) {
    pinMode(pin, OUTPUT);
  }

  void update(unsigned long deltaTime) override {
    elapsed += deltaTime;
    if (elapsed >= interval) {
      elapsed = 0;
      state = !state;
      digitalWrite(pin, state);
    }
  }
};

// Create multiple non-blocking components
LEDBlinker led1(13, 1000);  // Blink every 1 second
LEDBlinker led2(12, 500);   // Blink every 0.5 seconds
// Both run concurrently without blocking!

void setup() {
  // Components are automatically registered
}

void loop() {
  // Option 1: Automatic delta time (recommended, easiest!)
  Updatable::updateAllInstances();  // Calculates delta time automatically

  // Option 2: Manual delta time (if you need more control)
  // static unsigned long lastUpdate = 0;
  // unsigned long now = millis();
  // unsigned long deltaTime = now - lastUpdate;
  // lastUpdate = now;
  // Updatable::updateAllInstances(deltaTime);

  // Your loop remains responsive!
  // Add more non-blocking code here
}
```

**Key Benefits:**
- ✅ Multiple LEDs blink at different rates simultaneously
- ✅ No `delay()` calls blocking your program
- ✅ Loop stays responsive for other tasks
- ✅ Clean, modular, object-oriented code
- ✅ Automatic delta time calculation with overflow protection
- ✅ Ultra-simple: just call `updateAllInstances()` in your loop!

## API Reference

### Constructor
- `Updatable()` - Automatically registers the instance

### Virtual Methods
- `virtual void update(unsigned long deltaTime)` - Override this in derived classes to implement update logic

### Static Methods
- `static void updateAllInstances()` - Updates all registered instances with automatic delta time calculation (recommended)
  - Handles millis() overflow automatically
  - Note: This method is not thread-safe
- `static void updateAllInstances(unsigned long deltaTime)` - Updates all registered instances with manual delta time
- `static void setDebugMode(bool mode)` - Enables/disables debug mode for all instances

### Instance Methods
- `bool debugMode()` - Returns the current debug mode state

## License

This library is licensed under the GNU Lesser General Public License v3.0 (LGPL-3.0).

**What this means:**
- ✅ Free to use in personal and commercial projects
- ✅ You can modify and distribute the library
- ✅ Can be used in proprietary/closed-source applications
- ⚠️ If you modify the library itself, you must share your changes under LGPL-3.0
- ⚠️ Modified versions must be clearly marked as such

See [LICENSE](LICENSE) for full details.

### Why LGPL-3.0?

This license ensures:
1. The library remains free for everyone
2. Improvements benefit the entire community
3. You can still use it in commercial products without restrictions
4. The library can evolve with contributions from users

## Author

Francois Nouaille-Degorce
Copyright (C) 2012-2025
