/*************** FILE DESCRIPTION ***************/

/**
 * Filename: Drv7SegQ595.cpp
 * ----------------------------------------------------------------------------|---------------------------------------|
 * Purpose:  A class for driving a multiplexed 7-segment display using
 *           a single 74HC595 shift register IC and a set of transistors.
 * ----------------------------------------------------------------------------|---------------------------------------|
 * Notes:    Refer to the README for a general library overview and
 *           a basic API usage description.
 *
 *           Intended for displays with 1 to 4 character positions (digits).
 *
 *           seg_byte means a byte that turns ON and OFF individual segments.
 */


/************ PREPROCESSOR DIRECTIVES ***********/

/*--- Includes ---*/

// This source file's own header file.
#include "Drv7SegQ595.h"

// Additional Arduino libraries.
#ifdef DRV7SEGQ595_SPI_PROVIDED
    #include <SPI.h>
#endif


/*************** GLOBAL VARIABLES ***************/

Drv7SegQ595Class Drv7Seg;


/******************* FUNCTIONS ******************/

/*--- Constructors ---*/

Drv7SegQ595Class::Drv7SegQ595Class() {}


/*--- Public methods ---*/

int32_t Drv7SegQ595Class::begin_bb(PosSwitchType pos_switch_type,
                                   uint32_t data_pin,
                                   uint32_t latch_pin,
                                   uint32_t clock_pin,
                                   int32_t pos_1_pin,
                                   int32_t pos_2_pin,
                                   int32_t pos_3_pin,
                                   int32_t pos_4_pin
                                  )
{
    _status = begin_helper(DRV7SEGQ595_VARIANT_BIT_BANGING,
                           pos_switch_type,
                           latch_pin,
                           pos_1_pin,
                           pos_2_pin,
                           pos_3_pin,
                           pos_4_pin
                          );

    if (_status < 0) {
        return _status;
    }

    _data_pin  = data_pin;
    _clock_pin = clock_pin;
    pinMode(_data_pin,  OUTPUT);
    pinMode(_clock_pin, OUTPUT);

    return _status;
}

#ifdef DRV7SEGQ595_SPI_PROVIDED
int32_t Drv7SegQ595Class::begin_spi(PosSwitchType pos_switch_type,
                                    uint32_t latch_pin,
                                    int32_t pos_1_pin,
                                    int32_t pos_2_pin,
                                    int32_t pos_3_pin,
                                    int32_t pos_4_pin
                                   )
{
    _status = begin_helper(DRV7SEGQ595_VARIANT_SPI,
                           pos_switch_type,
                           latch_pin,
                           pos_1_pin,
                           pos_2_pin,
                           pos_3_pin,
                           pos_4_pin
                          );

    if (_status < 0) {
        return _status;
    }

    SPI.begin();

    return _status;
}
#endif

#ifdef DRV7SEGQ595_SPI_PROVIDED_CUSTOM_PINS
int32_t Drv7SegQ595Class::begin_spi_custom_pins(PosSwitchType pos_switch_type,
                                                uint32_t mosi_pin,
                                                uint32_t latch_pin,
                                                uint32_t sck_pin,
                                                int32_t pos_1_pin,
                                                int32_t pos_2_pin,
                                                int32_t pos_3_pin,
                                                int32_t pos_4_pin
                                               )
{
    _status = begin_helper(DRV7SEGQ595_VARIANT_SPI,
                           pos_switch_type,
                           latch_pin,
                           pos_1_pin,
                           pos_2_pin,
                           pos_3_pin,
                           pos_4_pin
                          );

    if (_status < 0) {
        return _status;
    }

    _mosi_pin = mosi_pin;
    _sck_pin  = sck_pin;

    #if defined(ARDUINO_ARCH_ESP32)
        SPI.begin(_sck_pin, -1, _mosi_pin, -1);
    #elif defined(ARDUINO_ARCH_STM32)
        SPI.setMOSI(_mosi_pin);
        SPI.setSCLK(_sck_pin);
        SPI.begin();
    #endif

    return _status;
}
#endif

int32_t Drv7SegQ595Class::get_status()
{
    return _status;
}

int32_t Drv7SegQ595Class::set_glyph_to_pos(uint8_t seg_byte, Pos pos)
{
    /*--- Configuration status check ---*/

    if (_status < 0) {
        return _status;
    }


    /*--- Protection from unexpected casts ---*/

    if (pos < Drv7SegPos1 || pos > Drv7SegPos4) {
        return DRV7SEGQ595_SET_GLYPH_ERR_INVALID_POS;
    }


    /*--- Assign a glyph to a position ---*/

    size_t pos_as_index = static_cast<size_t>(pos) - 1;
    if (_pos_pins[pos_as_index] <= DRV7SEGQ595_POS_PIN_INITIAL) {
        return DRV7SEGQ595_SET_GLYPH_ERR_POS_PIN_NOT_SPECIFIED_FOR_POS;
    } else {
        _pos_glyphs[pos_as_index] = seg_byte;
        return DRV7SEGQ595_SET_GLYPH_OK;
    }
}

int32_t Drv7SegQ595Class::output(uint8_t seg_byte,
                                 Pos pos
                                )
{
    /*--- Configuration status check ---*/

    if (_status < 0) {
        return _status;
    }


    /*--- Protection from unexpected casts ---*/

    if (pos < Drv7SegPos1 || pos > Drv7SegPos4) {
        return DRV7SEGQ595_OUTPUT_ERR_INVALID_POS;
    }


    /*--- Anti-ghosting retention ---*/

    if (_anti_ghosting_retention_duration > 0 && _anti_ghosting_first_output_call == false) {
        /* If this method has been called not for the character position
         * that must be turned on next, return and continue the retention.
         */
        if (pos != anti_ghosting_next_pos_to_output()) {
            return DRV7SEGQ595_OUTPUT_ANTI_GHOSTING_RETENTION_RUNNING;
        }

        // If the retention timer hasn't elapsed, return and continue the retention.
        if (anti_ghosting_timer() == false) {
            return DRV7SEGQ595_OUTPUT_ANTI_GHOSTING_RETENTION_RUNNING;
        }
    } else {
        _anti_ghosting_first_output_call = false;
    }


    /*--- Account for a character position switch type ---*/

    int32_t active = HIGH;
    if (_pos_switch_type == Drv7SegActiveLow) {
        active = !active;
    }


    /*--- Position-control pin check ---*/

    size_t pos_as_index = static_cast<size_t>(pos) - 1;
    if (_pos_pins[pos_as_index] <= DRV7SEGQ595_POS_PIN_INITIAL) {
        return DRV7SEGQ595_OUTPUT_ERR_POS_PIN_NOT_SPECIFIED_FOR_POS;
    }


    /*--- Switching the position-control pins ---*/

    for (size_t i = 0; i < DRV7SEGQ595_POS_MAX; ++i) {
        if (_pos_pins[i] > DRV7SEGQ595_POS_PIN_INITIAL) {
            digitalWrite(_pos_pins[i], !active);
        }
    }


    /*--- Shift data ---*/

    switch (_variant) {
        case DRV7SEGQ595_VARIANT_BIT_BANGING:
            digitalWrite(_latch_pin, LOW);
            shift_out(DRV7SEGQ595_ALL_BITS_CLEARED_MASK);
            digitalWrite(_latch_pin, HIGH);

            digitalWrite(_latch_pin, LOW);
            shift_out(seg_byte);
            digitalWrite(_latch_pin, HIGH);
            break;

        #ifdef DRV7SEGQ595_SPI_PROVIDED
        case DRV7SEGQ595_VARIANT_SPI:
            digitalWrite(_latch_pin, LOW);
            SPI.transfer(DRV7SEGQ595_ALL_BITS_CLEARED_MASK);
            digitalWrite(_latch_pin, HIGH);

            digitalWrite(_latch_pin, LOW);
            SPI.transfer(seg_byte);
            digitalWrite(_latch_pin, HIGH);
            break;
        #endif

        default:
            break;  // Do nothing and hail MISRA.
    }


    /*--- Switching the position-control pins, continued ---*/

    digitalWrite(_pos_pins[pos_as_index], active);


    // Update the values related to the anti-ghosting logic.
    _anti_ghosting_retained_pos = pos;
    _anti_ghosting_timer_previous_micros = micros();

    return DRV7SEGQ595_OUTPUT_NEXT;
}

void Drv7SegQ595Class::output_all()
{
    /*--- Configuration status check ---*/

    if (_status < 0) {
        return;
    }


    /*--- Output ---*/

    for (size_t i = 0; i < DRV7SEGQ595_POS_MAX; ++i) {
         output(_pos_glyphs[i], static_cast<Pos>(i + 1));
    }
}

void Drv7SegQ595Class::set_anti_ghosting_retention_duration(uint32_t new_val)
{
    /*--- Configuration status check ---*/

    if (_status < 0) {
        return;
    }


    /*--- Retention duration control ---*/

    _anti_ghosting_retention_duration = new_val;
}


/*--- Private methods ---*/

int32_t Drv7SegQ595Class::begin_helper(int32_t variant,
                                       PosSwitchType pos_switch_type,
                                       uint32_t latch_pin,
                                       int32_t pos_1_pin,
                                       int32_t pos_2_pin,
                                       int32_t pos_3_pin,
                                       int32_t pos_4_pin
                                      )
{
    /* Highly unlikely to occur without messing with the code,
     * but preserved as a redundant safety measure.
     */
    if (variant < 0) {
        return DRV7SEGQ595_STATUS_ERR_VARIANT_NOT_SPECIFIED;
    }

    // Position switch type validity check.
    if (pos_switch_type != Drv7SegActiveLow && pos_switch_type != Drv7SegActiveHigh) {
        return DRV7SEGQ595_STATUS_ERR_INVALID_POS_SWITCH_TYPE;
    }

    // Position-control pins validity check.
    int32_t pos_pins[DRV7SEGQ595_POS_MAX] = {pos_1_pin, pos_2_pin, pos_3_pin, pos_4_pin};
    for (uint32_t i = 0; i < DRV7SEGQ595_POS_MAX; ++i) {

        // First position-control pin (array index zero) must be >= 0.
        if (i == 0) {
            if (pos_pins[i] <= DRV7SEGQ595_POS_PIN_INITIAL) {
                return DRV7SEGQ595_STATUS_ERR_INVALID_POS_PIN;
            }
        }

        // Other position-control pins must be >= -1.
        if (i > 0) {
            if (pos_pins[i] < DRV7SEGQ595_POS_PIN_INITIAL) {
                return DRV7SEGQ595_STATUS_ERR_INVALID_POS_PIN;
            }
        }

        // Position-control pins duplication check.
        for (uint32_t j = i + 1; j < DRV7SEGQ595_POS_MAX; ++j) {
            if (pos_pins[i] != DRV7SEGQ595_POS_PIN_INITIAL &&
                pos_pins[j] != DRV7SEGQ595_POS_PIN_INITIAL &&
                pos_pins[i] == pos_pins[j]) {
                return DRV7SEGQ595_STATUS_ERR_POS_PIN_DUPLICATION;
            }
        }
    }

    _variant         = variant;
    _pos_switch_type = pos_switch_type;

    _latch_pin = latch_pin;
    pinMode(_latch_pin, OUTPUT);

    _pos_pins[0] = pos_1_pin;
    _pos_pins[1] = pos_2_pin;
    _pos_pins[2] = pos_3_pin;
    _pos_pins[3] = pos_4_pin;
    for (size_t i = 0; i < DRV7SEGQ595_POS_MAX; ++i) {
        if (_pos_pins[i] > DRV7SEGQ595_POS_PIN_INITIAL) {
            pinMode(_pos_pins[i], OUTPUT);
        }
    }

    /* Reset the variables related to the anti-ghosting logic
     * (useful if the driver gets reconfigured mid-use).
     */
    _anti_ghosting_first_output_call     = true;
    _anti_ghosting_retained_pos          = Drv7SegPos1;  /* Safe because position 1 is guaranteed to be valid
                                                          * for output. Not strictly necessary, but preserved
                                                          * as a redundant safety measure.
                                                          */
    _anti_ghosting_timer_previous_micros = 0;            /* Could call micros() instead, either is fine.
                                                          * Not strictly necessary, but preserved
                                                          * as a redundant safety measure.
                                                          */

    return DRV7SEGQ595_STATUS_OK;
}

void Drv7SegQ595Class::shift_out(uint8_t byte_to_shift)
{
    digitalWrite(_clock_pin, LOW);
    for (uint32_t i = 0; i < DRV7SEGQ595_BITS_IN_BYTE; i++) {
        digitalWrite(_data_pin, (byte_to_shift << i) & DRV7SEGQ595_ONLY_MSB_SET_MASK);
        digitalWrite(_clock_pin, HIGH);
        digitalWrite(_clock_pin, LOW);
    }
}

bool Drv7SegQ595Class::anti_ghosting_timer()
{
    if (_anti_ghosting_retention_duration == 0) {
        return true;  /* If the applicable retention duration is zero, the timer elapses
                       * without further calculations and the function returns early.
                       */
    }

    uint32_t current_micros = micros();

    if (current_micros - _anti_ghosting_timer_previous_micros >= _anti_ghosting_retention_duration) {
        return true;   // The timer has elapsed.
    } else {
        return false;  // The timer hasn't elapsed yet.
    }
}

Drv7SegQ595Class::Pos Drv7SegQ595Class::anti_ghosting_next_pos_to_output()
{
    // Subtract 1 because positions are 1-indexed while array members are 0-indexed.
    size_t pos_as_index = static_cast<size_t>(_anti_ghosting_retained_pos) - 1;

    // Add 1 because we're checking the next position, not the current one.
    for (size_t i = pos_as_index + 1; i < DRV7SEGQ595_POS_MAX; ++i) {
        // Search for a position that is valid for output (that was assigned a valid controlling pin).
        if (_pos_pins[i] > DRV7SEGQ595_POS_PIN_INITIAL) {
            // Add 1 because we're hopping back from 0-indexed to 1-indexed.
            return static_cast<Pos>(i + 1);
        }
    }

    // Character position 1 is guaranteed to be valid for output.
    return Drv7SegPos1;
}
