/*
 * DIYables WebApp Library - Web RTC Example
 * 
 * This example demonstrates the Web RTC feature:
 * - Real-time clock display for both ESP32 and client device
 * - One-click time synchronization from web browser to ESP32
 * - Hardware RTC integration for persistent timekeeping
 * - Visual time difference monitoring
 * 
 * Hardware Required:
 * - ESP32 development board
 * - DS3231 RTC module (connected via I2C)
 * 
 * Required Libraries:
 * - RTClib library (install via Library Manager)
 * 
 * Setup:
 * 1. Connect DS3231 RTC module to ESP32 I2C pins (SDA/SCL)
 * 2. Install RTClib library in Arduino IDE
 * 3. Update WiFi credentials below
 * 4. Upload the sketch to your ESP32
 * 5. Open Serial Monitor to see the IP address
 * 6. Navigate to http://[IP_ADDRESS]/web-rtc
 */

#include <DIYables_ESP32_Platform.h>
#include <DIYablesWebApps.h>
#include <RTClib.h>

// RTC object
RTC_DS3231 rtc;

char daysOfWeek[7][12] = {
  "Sunday",
  "Monday",
  "Tuesday",
  "Wednesday",
  "Thursday",
  "Friday",
  "Saturday"
};

// WiFi credentials - UPDATE THESE WITH YOUR NETWORK
const char WIFI_SSID[] = "YOUR_WIFI_SSID";
const char WIFI_PASSWORD[] = "YOUR_WIFI_PASSWORD";

// Create WebApp server and page instances
ESP32ServerFactory serverFactory;
DIYablesWebAppServer webAppsServer(serverFactory, 80, 81);
DIYablesHomePage homePage;
DIYablesWebRTCPage webRTCPage;

void setup() {
  Serial.begin(9600);
  delay(1000);

  Serial.println("DIYables ESP32 WebApp - Web RTC Example");

  // Initialize RTC
  if (!rtc.begin()) {
    Serial.println("RTC module is NOT found");
    Serial.flush();
    while (1);
  }

  // Check if RTC lost power and if so, set the time
  if (rtc.lostPower()) {
    Serial.println("RTC lost power, setting time!");
    // When time needs to be set on a new device, or after a power loss, the
    // following line sets the RTC to the date & time this sketch was compiled
    rtc.adjust(DateTime(F(__DATE__), F(__TIME__)));
    // This line sets the RTC with an explicit date & time, for example to set
    // January 21, 2021 at 3am you would call:
    // rtc.adjust(DateTime(2021, 1, 21, 3, 0, 0));
  }

  // Print initial RTC time
  DateTime initialTime = rtc.now();
  Serial.print("Initial RTC Time: ");
  Serial.print(initialTime.year(), DEC);
  Serial.print("/");
  Serial.print(initialTime.month(), DEC);
  Serial.print("/");
  Serial.print(initialTime.day(), DEC);
  Serial.print(" (");
  Serial.print(daysOfWeek[initialTime.dayOfTheWeek()]);
  Serial.print(") ");
  if (initialTime.hour() < 10) Serial.print("0");
  Serial.print(initialTime.hour(), DEC);
  Serial.print(":");
  if (initialTime.minute() < 10) Serial.print("0");
  Serial.print(initialTime.minute(), DEC);
  Serial.print(":");
  if (initialTime.second() < 10) Serial.print("0");
  Serial.print(initialTime.second(), DEC);
  Serial.println();

  // Add pages to server
  webAppsServer.addApp(&homePage);
  webAppsServer.addApp(&webRTCPage);

  // Optional: Add 404 page for better user experience
  webAppsServer.setNotFoundPage(DIYablesNotFoundPage());

  // Set callback for time sync from web
  webRTCPage.onTimeSyncFromWeb(onTimeSyncReceived);

  // Set callback for time request from web
  webRTCPage.onTimeRequestToWeb(onTimeRequested);

  // Start the WebApp server
  if (!webAppsServer.begin(WIFI_SSID, WIFI_PASSWORD)) {
    while (1) {
      Serial.println("Failed to connect to WiFi");
      delay(1000);
    }
  }
}

void loop() {
  // Handle web server
  webAppsServer.loop();

  // Send current time to web clients and print to Serial every 1 second
  static unsigned long lastUpdate = 0;
  if (millis() - lastUpdate >= 1000) {
    lastUpdate = millis();

    // Get current RTC time
    DateTime currentTime = rtc.now();

    // Send time to web clients in human readable format
    webRTCPage.sendTimeToWeb(currentTime.year(), currentTime.month(),
                             currentTime.day(), currentTime.hour(),
                             currentTime.minute(), currentTime.second());
    
    // Print time to Serial Monitor
    Serial.print("RTC Time: ");
    Serial.print(currentTime.year(), DEC);
    Serial.print("/");
    Serial.print(currentTime.month(), DEC);
    Serial.print("/");
    Serial.print(currentTime.day(), DEC);
    Serial.print(" (");
    Serial.print(daysOfWeek[currentTime.dayOfTheWeek()]);
    Serial.print(") ");
    if (currentTime.hour() < 10) Serial.print("0");
    Serial.print(currentTime.hour(), DEC);
    Serial.print(":");
    if (currentTime.minute() < 10) Serial.print("0");
    Serial.print(currentTime.minute(), DEC);
    Serial.print(":");
    if (currentTime.second() < 10) Serial.print("0");
    Serial.print(currentTime.second(), DEC);
    Serial.println();
  }

  delay(10);
}

// Callback function called when web client sends time sync command
void onTimeSyncReceived(unsigned long unixTimestamp) {
  Serial.print("Time sync received: ");
  Serial.println(unixTimestamp);

  // Convert Unix timestamp to DateTime and set RTC time
  DateTime newTime(unixTimestamp);
  rtc.adjust(newTime);

  Serial.println("ESP32 RTC synchronized!");
}

// Callback function called when web client requests current ESP32 time
void onTimeRequested() {
  // Get current RTC time and send to web in human readable format
  DateTime currentTime = rtc.now();

  webRTCPage.sendTimeToWeb(currentTime.year(), currentTime.month(),
                           currentTime.day(), currentTime.hour(),
                           currentTime.minute(), currentTime.second());
}