#include "ESP32WebSocketServer.h"

// Static instance definition
ESP32WebSocketServer* ESP32WebSocketServer::instance = nullptr;

// Constructor and Destructor
ESP32WebSocketServer::ESP32WebSocketServer(uint16_t serverPort) 
    : server(serverPort), port(serverPort), 
      connectionCallback(nullptr), messageCallback(nullptr), closeCallback(nullptr) {
    instance = this;
}

ESP32WebSocketServer::~ESP32WebSocketServer() {
    if (instance == this) {
        instance = nullptr;
    }
    // Client cleanup is now handled by WebSocketClientManager
}

// Static wrapper functions - Simplified to use core client manager
void ESP32WebSocketServer::onConnectionWrapper(net::WebSocket& ws) {
    if (instance && instance->connectionCallback) {
        // Create wrapper and let client manager handle it
        ESP32WebSocket* wrapper = new ESP32WebSocket(ws);
        IWebSocket* managedWrapper = instance->clientManager.addClient(&ws, wrapper);
        
        if (managedWrapper) {
            instance->connectionCallback(managedWrapper);
        } else {
            delete wrapper; // Client manager rejected it
            Serial.println("WARNING: Failed to add WebSocket client");
        }
    }
}

void ESP32WebSocketServer::onMessageWrapper(net::WebSocket& ws, const net::WebSocket::DataType dataType, const char* message, uint16_t length) {
    if (instance && instance->messageCallback) {
        IWebSocket* wrapper = instance->clientManager.findClient(&ws);
        if (wrapper) {
            IWebSocket::DataType iDataType = (dataType == net::WebSocket::DataType::TEXT) 
                ? IWebSocket::TEXT 
                : IWebSocket::BINARY;
            instance->messageCallback(wrapper, iDataType, message, length);
        }
    }
}

void ESP32WebSocketServer::onCloseWrapper(net::WebSocket& ws, const net::WebSocket::CloseCode code, const char* reason, uint16_t length) {
    if (instance && instance->closeCallback) {
        IWebSocket* wrapper = instance->clientManager.findClient(&ws);
        if (wrapper) {
            IWebSocket::CloseCode iCode = static_cast<IWebSocket::CloseCode>(code);
            instance->closeCallback(wrapper, iCode, reason, length);
            
            // Let client manager handle cleanup
            instance->clientManager.removeClient(&ws);
        }
    }
}

// Server lifecycle
bool ESP32WebSocketServer::begin() {
    return server.begin();
}

void ESP32WebSocketServer::end() {
    // Clear callbacks and let client manager cleanup clients
    connectionCallback = nullptr;
    messageCallback = nullptr;
    closeCallback = nullptr;
    clientManager.cleanupAllClients();
}

void ESP32WebSocketServer::listen() {
    server.loop();
}

// Event callbacks
void ESP32WebSocketServer::onConnection(ConnectionCallback callback) {
    connectionCallback = callback;
    server.onOpen(onConnectionWrapper);
}

void ESP32WebSocketServer::onMessage(MessageCallback callback) {
    messageCallback = callback;
    server.onMessage(onMessageWrapper);
}

void ESP32WebSocketServer::onClose(CloseCallback callback) {
    closeCallback = callback;
    server.onClose(onCloseWrapper);
}

// Broadcasting methods - PURE TRANSPORT LAYER
void ESP32WebSocketServer::broadcastText(const char* message) {
    if (message) {
        yield();
        server.broadcastTXT(message);
    }
}

void ESP32WebSocketServer::broadcastBinary(const uint8_t* data, uint16_t length) {
    yield();
    server.broadcastBIN(data, length);
}

// Configuration
uint16_t ESP32WebSocketServer::getPort() const {
    return port;
}

