#include "ESP32WebSocket.h"

// Constructors and Destructor
ESP32WebSocket::ESP32WebSocket(WebSocket* ws, bool takeOwnership) 
    : webSocket(ws), ownsWebSocket(takeOwnership), messageCallback(nullptr), closeCallback(nullptr) {
}

ESP32WebSocket::ESP32WebSocket(WebSocket& ws) 
    : webSocket(&ws), ownsWebSocket(false), messageCallback(nullptr), closeCallback(nullptr) {
}

ESP32WebSocket::~ESP32WebSocket() {
    if (ownsWebSocket && webSocket) {
        delete webSocket;
    }
}

// Connection management
bool ESP32WebSocket::isConnected() {
    return webSocket != nullptr; // ESP32 WebSocket doesn't have a direct isConnected method
}

void ESP32WebSocket::close(CloseCode code, const char* reason) {
    if (webSocket) {
        // Convert our enum to ESP32 enum
        WebSocket::CloseCode r4Code = static_cast<WebSocket::CloseCode>(code);
        webSocket->close(r4Code, true, reason, reason ? strlen(reason) : 0);
    }
}

// Message handling
bool ESP32WebSocket::send(DataType dataType, const char* message, uint16_t length) {
    if (!webSocket) return false;
    
    // Additional validation like ESP32
    if (!message || length == 0) return false;
    
    // SIMPLE FIX: Add connection check to prevent sending to dead connections
    if (!isConnected()) return false;
    
    // Convert our enum to ESP32 enum
    WebSocket::DataType r4DataType = (dataType == TEXT) 
        ? WebSocket::DataType::TEXT 
        : WebSocket::DataType::BINARY;
    
    // FIXED: Add yield() like ESP32 to prevent frame corruption on ESP32
    yield();
    
    webSocket->send(r4DataType, message, strlen(message));
    
    return true; // ESP32 send method is void, so assume success
}

// Callback management - Simplified (callbacks are managed at server level)
void ESP32WebSocket::onMessage(void (*callback)(IWebSocket* ws, const DataType dataType, const char* message, uint16_t length)) {
    messageCallback = callback;
    // Note: Actual callback registration happens at the server level via WebSocketClientManager
}

void ESP32WebSocket::onClose(void (*callback)(IWebSocket* ws, const CloseCode code, const char* reason, uint16_t length)) {
    closeCallback = callback;
    // Note: Actual callback registration happens at the server level via WebSocketClientManager
}

// Client information
String ESP32WebSocket::getRemoteIP() {
    // ESP32 WebSocket might not expose this directly
    // Return empty string or implement based on available methods
    return String(""); 
}

uint16_t ESP32WebSocket::getRemotePort() {
    // ESP32 WebSocket might not expose this directly
    // Return 0 or implement based on available methods
    return 0;
}

// Internal method
WebSocket* ESP32WebSocket::getUnderlyingWebSocket() { 
    return webSocket; 
}
