/*
  ModbusTCPSlaveWiFiExample

  This example demonstrates how to setup and use the ModbusTCPSlave library (https://github.com/CMB27/ModbusTCPSlave).

  This was tested with the following boards:
  - Arduino Nano 33 IoT (https://store.arduino.cc/products/arduino-nano-33-iot)
  - Arduino Nano RP2040 Connect (https://store.arduino.cc/products/arduino-nano-rp2040-connect)
  - Arduino Nano ESP32 (https://store.arduino.cc/products/nano-esp32)
  - Arduino UNO R4 WiFi (https://store.arduino.cc/products/uno-r4-wifi)
  - Arduino GIGA R1 WiFi (https://store.arduino.cc/products/giga-r1-wifi)

  A ModbusRTU Test Shield (https://github.com/CMB27/ModbusRTU-Test-Shield) was used to provide the buttons, knobs, and LEDs.

  It worked well with all boards except the Arduino UNO R4 WiFi.
  
  Created: 2024-12-14
  By: C. M. Bulliner
  Last Modified: 2025-02-08
  By: C. M. Bulliner
  
*/

#include <ModbusTCPSlave.h>
#include "arduino_secrets.h"

#if defined(ADRUINO_SAMD_MKRWIFI1010) || defined(ARDUINO_SAMD_MKRVIDOR4000) || defined(ARDUINO_SAMD_NANO_33_IOT) || defined(ARDUINO_AVR_UNO_WIFI_REV2) || defined(ARDUINO_NANO_RP2040_CONNECT)
  #include <WiFiNINA.h>
#elif defined(ARDUINO_NANO_ESP32) || defined(ARDUINO_GIGA) || defined(ARDUINO_UNO_R4_WIFI)
  #include <WiFi.h>
#elif defined(ARDUINO_UNOR4_WIFI)
  #include <WiFiS3.h>
#endif



#if (defined(ARDUINO_NANO_RP2040_CONNECT) && !defined(ARDUINO_ARCH_MBED)) || defined(ARDUINO_NANO_ESP32)
  // These boards operate unsing GPIO numbers that don't correspond to the numbers on the boards.
  // However they do have D# values #defined to correct this.
  const int8_t buttonPins[2] = {D2, D3};
  const int8_t ledPins[4] = {D5, D6, D7, D8};
#else
  // Other boards do not have D# values, and will throw an error if you try to use them.
  const int8_t buttonPins[2] = {2, 3};
  const int8_t ledPins[4] = {5, 6, 7, 8};
#endif
const int8_t knobPins[2] = {A0, A1};


WiFiServer server(MODBUS_TCP_SLAVE_DEFAULT_PORT);
ModbusTCPSlave modbus;

const uint8_t numCoils = 2;
const uint8_t numDiscreteInputs = 2;
const uint8_t numHoldingRegisters = 2;
const uint8_t numInputRegisters = 2;

bool coils[numCoils];
bool discreteInputs[numDiscreteInputs];
uint16_t holdingRegisters[numHoldingRegisters];
uint16_t inputRegisters[numInputRegisters];

void setup() {
  Serial.begin(115200);
  while (!Serial && millis() < 2000);

  pinMode(knobPins[0], INPUT);
  pinMode(knobPins[1], INPUT);
  pinMode(buttonPins[0], INPUT_PULLUP);
  pinMode(buttonPins[1], INPUT_PULLUP);
  pinMode(ledPins[0], OUTPUT);
  pinMode(ledPins[1], OUTPUT);
  pinMode(ledPins[2], OUTPUT);
  pinMode(ledPins[3], OUTPUT);
  pinMode(LED_BUILTIN, OUTPUT);

  #if defined(ARDUINO_NANO_ESP32)
    analogReadResolution(10);
  #endif

  modbus.configureCoils(coils, numCoils);
  modbus.configureDiscreteInputs(discreteInputs, numDiscreteInputs);
  modbus.configureHoldingRegisters(holdingRegisters, numHoldingRegisters);
  modbus.configureInputRegisters(inputRegisters, numInputRegisters);

  Serial.println("Connecting to: " + String(SECRET_SSID));
  WiFi.begin(SECRET_SSID, SECRET_PASS);
  while (WiFi.status() != WL_CONNECTED);
  Serial.println("Connected to: " + String(WiFi.SSID()));
  Serial.println("IP address: " + WiFi.localIP().toString());

  server.begin();
}

void loop() {
  WiFiClient client = server.available();
  if (client) {
    while (client.connected()) {
      inputRegisters[0] = map(analogRead(knobPins[0]), 0, 1023, 0, 255);
      inputRegisters[1] = map(analogRead(knobPins[1]), 0, 1023, 0, 255);
      discreteInputs[0] = !digitalRead(buttonPins[0]);
      discreteInputs[1] = !digitalRead(buttonPins[1]);

      modbus.poll(client);

      analogWrite(ledPins[0], holdingRegisters[0]);
      analogWrite(ledPins[1], holdingRegisters[1]);
      digitalWrite(ledPins[2], coils[0]);
      digitalWrite(ledPins[3], coils[1]);
    }
  }
}
