/*
  ModbusTCPSlaveEthernetExample

  This example demonstrates how to setup and use the ModbusTCPSlave library (https://github.com/CMB27/ModbusTCPSlave).

  The following hardware was used to test this example program:
  - Arduino Make Your UNO Kit (just the UNO) (https://store.arduino.cc/pages/make-your-uno-kit)
  - Adruino Ethernet Shield 2 (https://store.arduino.cc/collections/shields-carriers/products/arduino-ethernet-shield-2)
  - ModbusRTU Test Shield (https://github.com/CMB27/ModbusRTU-Test-Shield)

  On the ModbusRTU Test Shield, the UART switch was set to HW to avid interference with the SPI interface.

  This example also requires a Modbus client on the same network as the slave/server device.
  This client will need to be configured to read two discrete inputs and two input registers.
  It will also need to be configured to write to two coils and two holding registers.
  
  Created: 2024-12-23
  By: C. M. Bulliner
  Last Modified: 2025-02-08
  By: C. M. Bulliner
  
*/

#include <Ethernet.h>
#include <ModbusTCPSlave.h>

// Your MAC address may be different, see your ethernet shield documentation.
// If using an Adruino Ethernet Shield 2, there should be a sticker on the back telling you what this should be.
uint8_t macAddress[] = {0xA8, 0x61, 0x0A, 0xAE, 0x13, 0xA0};

const int8_t buttonPins[2] = {2, 3};
const int8_t ledPins[4] = {5, 6, 7, 8};
const int8_t knobPins[2] = {A0, A1};

EthernetServer server(MODBUS_TCP_SLAVE_DEFAULT_PORT);
ModbusTCPSlave modbus;

const uint16_t numCoils = 2;
const uint16_t numDiscreteInputs = 2;
const uint16_t numHoldingRegisters = 2;
const uint16_t numInputRegisters = 2;

bool coils[numCoils];
bool discreteInputs[numDiscreteInputs];
uint16_t holdingRegisters[numHoldingRegisters];
uint16_t inputRegisters[numInputRegisters];

void setup() {
  Serial.begin(115200);
  while (!Serial && millis() < 2000);

  pinMode(knobPins[0], INPUT);
  pinMode(knobPins[1], INPUT);
  pinMode(buttonPins[0], INPUT_PULLUP);
  pinMode(buttonPins[1], INPUT_PULLUP);
  pinMode(ledPins[0], OUTPUT);
  pinMode(ledPins[1], OUTPUT);
  pinMode(ledPins[2], OUTPUT);
  pinMode(ledPins[3], OUTPUT);

  modbus.configureCoils(coils, numCoils);
  modbus.configureDiscreteInputs(discreteInputs, numDiscreteInputs);
  modbus.configureHoldingRegisters(holdingRegisters, numHoldingRegisters);
  modbus.configureInputRegisters(inputRegisters, numInputRegisters);

  Serial.println("Connecting");
  if (!Ethernet.begin(macAddress)) {
    Serial.println("Failed to configure Ethernet using DHCP");
    if (Ethernet.hardwareStatus() == EthernetNoHardware) Serial.println("Ethernet shield not found");
    else if (Ethernet.linkStatus() == LinkOFF) Serial.println("Ethernet cable not connected");
    while (true);
  }
  Serial.print("IP address: ");
  Serial.println(Ethernet.localIP());

  server.begin();
}

void loop() {
  EthernetClient client = server.available();
  if (client) {
    while (client.connected()) {
      inputRegisters[0] = map(analogRead(knobPins[0]), 0, 1023, 0, 255);
      inputRegisters[1] = map(analogRead(knobPins[1]), 0, 1023, 0, 255);
      discreteInputs[0] = !digitalRead(buttonPins[0]);
      discreteInputs[1] = !digitalRead(buttonPins[1]);

      modbus.poll(client);

      analogWrite(ledPins[0], holdingRegisters[0]);
      analogWrite(ledPins[1], holdingRegisters[1]);
      digitalWrite(ledPins[2], coils[0]);
      digitalWrite(ledPins[3], coils[1]);
    }
  }
}
