/* The MIT License (MIT)
 *
 * Copyright (c) 2014 - 2025 Andreas Merkle
 * http://www.blue-andi.de
 * vscp@blue-andi.de
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*******************************************************************************
    DESCRIPTION
*******************************************************************************/
/**
@brief  VSCP Protocol Functionality
@file   vscp_evt_protocol.h
@author Andreas Merkle, http://www.blue-andi.de

@section desc Description
This class defines some types that must be implemented by every node that implements the VSCP
protocol. The types in this class must be handled by all level I and Level II nodes. Note also that
this class is repeated as Level II class=512 with the only difference that GUID's are used instead
of nicknames. This means that for all Level II class=512 events the data described here is preceded
by the 16-bit destination GUID (address of receiver) stored with MSB first followed by the data
specified here. Nickname is present also if documented below but have no meaning and should be
discarded.

This file is automatically generated. Don't change it manually.

*******************************************************************************/

#ifndef __VSCP_EVT_PROTOCOL_H__
#define __VSCP_EVT_PROTOCOL_H__

/*******************************************************************************
    INCLUDES
*******************************************************************************/
#include <stdint.h>
#include "../user/vscp_platform.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*******************************************************************************
    COMPILER SWITCHES
*******************************************************************************/

/*******************************************************************************
    CONSTANTS
*******************************************************************************/

/*******************************************************************************
    MACROS
*******************************************************************************/

/*******************************************************************************
    TYPES AND STRUCTURES
*******************************************************************************/

/*******************************************************************************
    VARIABLES
*******************************************************************************/

/*******************************************************************************
    FUNCTIONS
*******************************************************************************/

/**
 * General event.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendGeneralEvent(void);

/**
 * Segment Controller Heartbeat.
 * 
 * @param[in] crc 8-bit CRC of the segment controller GUID
 * @param[in] time Time since epoch. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendSegmentControllerHeartbeat(uint8_t crc, uint32_t const * const time);

/**
 * New node on line / Probe.
 * 
 * @param[in] targetAddress This is the probe nickname that the new node is using to test if this is a
 * valid target node. If there is a node with this nickname address it should answer with probe ACK. A
 * probe always has 0xff as it's own temporary nickname while a new node on line use a non 0xff
 * nickname.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendNewNodeOnLineProbe(uint8_t targetAddress);

/**
 * Probe ACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendProbeAck(void);

/* "Reserved for future use." not supported. No frame defined. */

/* "Reserved for future use." not supported. No frame defined. */

/**
 * Set nickname-ID for node.
 * 
 * @param[in] oldNickname Old nickname for node.
 * @param[in] newNickname The new nickname for the node.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendSetNicknameIdForNode(uint8_t oldNickname, uint8_t newNickname);

/**
 * Nickname-ID accepted.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendNicknameIdAccepted(void);

/**
 * Drop nickname-ID / Reset Device.
 * 
 * @param[in] nickname The current nickname for the node.
 * @param[in] flags Flags. (optional)
 * @param[in] time Time the node should wait before it starts a nickname discovery or starts the
 * device. The time is in seconds. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendDropNicknameIdResetDevice(uint8_t nickname, uint8_t const * const flags, uint8_t const * const time);

/**
 * Read register.
 * 
 * @param[in] nodeAddress Node address.
 * @param[in] registerAddress Register to read.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendReadRegister(uint8_t nodeAddress, uint8_t registerAddress);

/**
 * Read/Write response.
 * 
 * @param[in] registerAddress Register address read/written.
 * @param[in] registerValue Value of register.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendReadWriteResponse(uint8_t registerAddress, uint8_t registerValue);

/**
 * Write register.
 * 
 * @param[in] nodeAddress Node address.
 * @param[in] registerAddress Register to write.
 * @param[in] registerValue Value which to write to register.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendWriteRegister(uint8_t nodeAddress, uint8_t registerAddress, uint8_t registerValue);

/**
 * Enter boot loader mode.
 * 
 * @param[in] nodeAddress The nickname for the node.
 * @param[in] bootloaderAlgorithm Code that select boot loader algorithm to use.
 * @param[in] guidByte0 GUID Byte 0 (MSB).
 * @param[in] guidByte3 GUID Byte 3 (MSB + 3).
 * @param[in] guidByte5 GUID Byte 5 (MSB + 5).
 * @param[in] guidByte7 GUID Byte 7 (MSB + 7).
 * @param[in] pageSelect Page select, content of register 0x92 and 0x93.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendEnterBootLoaderMode(uint8_t nodeAddress, uint8_t bootloaderAlgorithm, uint8_t guidByte0, uint8_t guidByte3, uint8_t guidByte5, uint8_t guidByte7, uint16_t pageSelect);

/**
 * ACK boot loader mode.
 * 
 * @param[in] flashBlockSize Flash block size in byte.
 * @param[in] numberOfBlocks Number of blocks available.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendAckBootLoaderMode(uint32_t flashBlockSize, uint32_t numberOfBlocks);

/**
 * NACK boot loader mode.
 * 
 * @param[in] errorCode Optional user defined error code. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendNackBootLoaderMode(uint32_t const * const errorCode);

/**
 * Start block data transfer.
 * 
 * @param[in] blockNumber Block number.
 * @param[in] memoryType Type of Memory we want to write.. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendStartBlockDataTransfer(uint32_t blockNumber, uint8_t const * const memoryType);

/**
 * Block data.
 * 
 * @param[in] data Data. (array[8])
 * @param[in] datasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBlockData(uint8_t const * const data, uint8_t dataSize);

/**
 * ACK data block.
 * 
 * @param[in] blockCrc The CRC is calculated over the block data only.
 * @param[in] blockToWrite The block to write is the block that was sent in the last block data event.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendAckDataBlock(uint16_t blockCrc, uint32_t blockToWrite);

/**
 * NACK data block.
 * 
 * @param[in] errorCode User defined error code.
 * @param[in] blockToWrite The block to write is the block that was sent in the last block data event.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendNackDataBlock(uint8_t errorCode, uint32_t blockToWrite);

/**
 * Program data block.
 * 
 * @param[in] blockNumber Block number.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendProgramDataBlock(uint32_t blockNumber);

/**
 * ACK program data block.
 * 
 * @param[in] blockNumber Block number.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendAckProgramDataBlock(uint32_t blockNumber);

/**
 * NACK program data block.
 * 
 * @param[in] errorCode User defined error code.
 * @param[in] blockNumber Block number.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendNackProgramDataBlock(uint8_t errorCode, uint32_t blockNumber);

/**
 * Activate new image.
 * 
 * @param[in] crc Sum of all CRC of blocks that was transferred to the node up to this point (all
 * memory types).
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendActivateNewImage(uint16_t crc);

/* "GUID drop nickname-ID / reset device." not supported, because of multi frame. */

/**
 * Page read.
 * 
 * @param[in] nodeId Node-ID which registers should be read.
 * @param[in] index Index into page.
 * @param[in] numberOfBytes Number of bytes to read.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendPageRead(uint8_t nodeId, uint8_t index, uint8_t numberOfBytes);

/**
 * Page write.
 * 
 * @param[in] nodeId Node-ID which registers should be read.
 * @param[in] registerStart Register start address.
 * @param[in] data Data to write. (array[6])
 * @param[in] datasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendPageWrite(uint8_t nodeId, uint8_t registerStart, uint8_t const * const data, uint8_t dataSize);

/**
 * Read/Write page response.
 * 
 * @param[in] sequenceNumber Sequence number.
 * @param[in] data Data. (array[7])
 * @param[in] datasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendReadWritePageResponse(uint8_t sequenceNumber, uint8_t const * const data, uint8_t dataSize);

/**
 * High end server/service probe.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendHighEndServerServiceProbe(void);

/**
 * High end server/service response.
 * 
 * @param[in] capabilityCode VSCP server capability code.
 * @param[in] serverIp Server IP address. (array[4])
 * @param[in] serverIpsize Size in byte.
 * @param[in] serverPort Server port number.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendHighEndServerServiceResponse(uint16_t capabilityCode, uint8_t const * const serverIp, uint8_t serverIpSize, uint16_t serverPort);

/**
 * Increment register.
 * 
 * @param[in] nodeId Node-ID.
 * @param[in] registerAddress Register to increment. (array[7])
 * @param[in] registerAddresssize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendIncrementRegister(uint8_t nodeId, uint8_t const * const registerAddress, uint8_t registerAddressSize);

/**
 * Decrement register.
 * 
 * @param[in] nodeId Node-ID.
 * @param[in] registerAddress Register to decrement. (array[7])
 * @param[in] registerAddresssize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendDecrementRegister(uint8_t nodeId, uint8_t const * const registerAddress, uint8_t registerAddressSize);

/**
 * Who is there?
 * 
 * @param[in] nodeId Node-ID or 0xFF for all nodes.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendWhoIsThere(uint8_t nodeId);

/* "Who is there response." not supported, because of multi frame. */

/**
 * Get decision matrix info.
 * 
 * @param[in] nodeAddress Node address.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendGetDecisionMatrixInfo(uint8_t nodeAddress);

/**
 * Decision matrix info response.
 * 
 * @param[in] matrixSize Matrix size (number of rows). Zero for a device with no decision matrix.
 * @param[in] offset Offset in register space.
 * @param[in] pageStart Page start. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendDecisionMatrixInfoResponse(uint8_t matrixSize, uint8_t offset, uint16_t const * const pageStart);

/**
 * Get embedded MDF.
 * 
 * @param[in] nodeAddress Node address.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendGetEmbeddedMdf(uint8_t nodeAddress);

/**
 * Embedded MDF response.
 * 
 * @param[in] mdfDescriptionIndex MDF description index.
 * @param[in] mdfData MDF data.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendEmbeddedMdfResponse(uint16_t mdfDescriptionIndex, uint8_t mdfData);

/**
 * Extended page read register.
 * 
 * @param[in] nodeAddress Node address.
 * @param[in] page Page where the register is located.
 * @param[in] offset Register to read (offset into page).
 * @param[in] number Number of registers to read. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendExtendedPageReadRegister(uint8_t nodeAddress, uint16_t page, uint8_t offset, uint8_t const * const number);

/**
 * Extended page write register.
 * 
 * @param[in] nodeAddress Node address.
 * @param[in] page Page where the register is located.
 * @param[in] offset Register to write (offset into page).
 * @param[in] value Content for register. (array[4])
 * @param[in] valuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendExtendedPageWriteRegister(uint8_t nodeAddress, uint16_t page, uint8_t offset, uint8_t const * const value, uint8_t valueSize);

/**
 * Extended page read/write response.
 * 
 * @param[in] index Index (starts at zero).
 * @param[in] page Page where the register is located.
 * @param[in] offset Register read/written.
 * @param[in] value Content of register. (array[4])
 * @param[in] valuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendExtendedPageReadWriteResponse(uint8_t index, uint16_t page, uint8_t offset, uint8_t const * const value, uint8_t valueSize);

/**
 * Get event interest.
 * 
 * @param[in] nodeAddress Node address.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendGetEventInterest(uint8_t nodeAddress);

/**
 * Get event interest response.
 * 
 * @param[in] index Index.
 * @param[in] class Class.
 * @param[in] type Type.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendGetEventInterestResponse(uint8_t index, uint16_t class, uint16_t type);

/**
 * Activate new image ACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendActivateNewImageAck(void);

/**
 * Activate new image NACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendActivateNewImageNack(void);

/**
 * Start block ACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendStartBlockAck(void);

/**
 * Start block NACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendStartBlockNack(void);

/**
 * Block Data Chunk ACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBlockDataChunkAck(void);

/**
 * Block Data Chunk NACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBlockDataChunkNack(void);

/**
 * Bootloader CHECK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBootloaderCheck(void);

/**
 * Bootloader Abort.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBootloaderAbort(void);

/**
 * Bootloader Abort ACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBootloaderAbortAck(void);

/**
 * Bootloader Abort NACK.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_protocol_sendBootloaderAbortNack(void);

#ifdef __cplusplus
}
#endif

#endif  /* __VSCP_EVT_PROTOCOL_H__ */
