/* The MIT License (MIT)
 *
 * Copyright (c) 2014 - 2025 Andreas Merkle
 * http://www.blue-andi.de
 * vscp@blue-andi.de
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*******************************************************************************
    DESCRIPTION
*******************************************************************************/
/**
@brief  Information
@file   vscp_evt_information.c
@author Andreas Merkle, http://www.blue-andi.de

@section desc Description
@see vscp_evt_information.h

This file is automatically generated. Don't change it manually.

*******************************************************************************/

/*******************************************************************************
    INCLUDES
*******************************************************************************/
#include "vscp_evt_information.h"
#include "../core/vscp_core.h"
#include "../core/vscp_class_l1.h"
#include "../core/vscp_type_information.h"

/*******************************************************************************
    COMPILER SWITCHES
*******************************************************************************/

/*******************************************************************************
    CONSTANTS
*******************************************************************************/

/*******************************************************************************
    MACROS
*******************************************************************************/

/*******************************************************************************
    TYPES AND STRUCTURES
*******************************************************************************/

/*******************************************************************************
    VARIABLES
*******************************************************************************/

/*******************************************************************************
    FUNCTIONS
*******************************************************************************/

/**
 * General event
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGeneralEvent(void)
{
    vscp_TxMessage  txMsg;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GENERAL, VSCP_PRIORITY_3_NORMAL);

    txMsg.dataSize = 0;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Button
 * 
 * @param[in] buttonState Button state.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] buttonCode Button code.
 * @param[in] codePage Code page. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendButton(uint8_t buttonState, uint8_t zone, uint8_t subZone, uint16_t buttonCode, uint16_t const * const codePage)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_BUTTON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = buttonState;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = (uint8_t)((buttonCode >> 8) & 0xff);
    txMsg.data[4] = (uint8_t)((buttonCode >> 0) & 0xff);
    size += 2;

    if (NULL != codePage)
    {
        txMsg.data[5] = (uint8_t)((*codePage >> 8) & 0xff);
        txMsg.data[6] = (uint8_t)((*codePage >> 0) & 0xff);
        size += 2;
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Mouse
 * 
 * @param[in] kindOfCoordinates If zero absolute coordinates follow. If equal to one relative
 * coordinates follow.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] normalizedXCoordinate Normalized x-coordinate.
 * @param[in] normalizedYCoordinate Normalized y-coordinate.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendMouse(uint8_t kindOfCoordinates, uint8_t zone, uint8_t subZone, uint16_t normalizedXCoordinate, uint16_t normalizedYCoordinate)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_MOUSE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = kindOfCoordinates;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = (uint8_t)((normalizedXCoordinate >> 8) & 0xff);
    txMsg.data[4] = (uint8_t)((normalizedXCoordinate >> 0) & 0xff);
    size += 2;

    txMsg.data[5] = (uint8_t)((normalizedYCoordinate >> 8) & 0xff);
    txMsg.data[6] = (uint8_t)((normalizedYCoordinate >> 0) & 0xff);
    size += 2;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * On
 * 
 * @param[in] index Index, often used as an index for channels within a module.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendOn(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Off
 * 
 * @param[in] index Index, often used as an index for channels within a module.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendOff(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_OFF, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Alive
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendAlive(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ALIVE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Terminating
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendTerminating(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_TERMINATING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Opened
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendOpened(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_OPENED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Closed
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendClosed(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_CLOSED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Node Heartbeat
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] userData Optional user provided data. (optional) (array[5])
 * @param[in] userDatasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendNodeHeartbeat(uint8_t userSpecific, uint8_t zone, uint8_t subZone, uint8_t const * const userData, uint8_t userDataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_NODE_HEARTBEAT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    if (NULL != userData)
    {
        for(byteIndex = 0; byteIndex < userDataSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = userData[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Below limit
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendBelowLimit(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_BELOW_LIMIT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Above limit
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendAboveLimit(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ABOVE_LIMIT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Pulse
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendPulse(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_PULSE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Error
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendError(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ERROR, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Resumed
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendResumed(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_RESUMED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Paused
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendPaused(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_PAUSED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Sleeping
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendSleeping(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SLEEP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Good morning
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodMorning(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOOD_MORNING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Good day
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodDay(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOOD_DAY, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Good afternoon
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodAfternoon(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOOD_AFTERNOON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Good evening
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodEvening(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOOD_EVENING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Good night
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodNight(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOOD_NIGHT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * See you soon
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendSeeYouSoon(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SEE_YOU_SOON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Goodbye
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendGoodbye(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_GOODBYE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Stop
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendStop(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_STOP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Start
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendStart(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * ResetCompleted
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendResetcompleted(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_RESET_COMPLETED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Interrupted
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendInterrupted(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_INTERRUPTED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * PreparingToSleep
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendPreparingtosleep(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_PREPARING_TO_SLEEP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * WokenUp
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendWokenup(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_WOKEN_UP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Dusk
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDusk(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DUSK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Dawn
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDawn(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DAWN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Active
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendActive(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ACTIVE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Inactive
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendInactive(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_INACTIVE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Busy
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendBusy(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_BUSY, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Idle
 * 
 * @param[in] reserved Reserved for future use.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendIdle(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_IDLE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Stream Data
 * 
 * @param[in] sequenceNumber Sequence number which is increased by one for every new event with stream
 * data. The sequence number is increased for every frame sent going from 0 - 255 and back to 0 and so
 * on if needed. A new stream starts with a sequence number of 0.
 * @param[in] streamData Stream data. (array[7])
 * @param[in] streamDatasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendStreamData(uint8_t sequenceNumber, uint8_t const * const streamData, uint8_t streamDataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == streamData) || (0 == streamDataSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_STREAM_DATA, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = sequenceNumber;
    size += 1;

    for(byteIndex = 0; byteIndex < streamDataSize; ++byteIndex)
    {
        txMsg.data[1 + byteIndex] = streamData[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Token Activity
 * 
 * @param[in] tokenType Bit 0,1 - Event code. Bit 2-7 - Token device type code.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] frameIndex Frame index (Increase by one for every frame sent out for one token activity.
 * Start with zero ).
 * @param[in] tokenData Token data. (array[4])
 * @param[in] tokenDatasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendTokenActivity(uint8_t tokenType, uint8_t zone, uint8_t subZone, uint8_t frameIndex, uint8_t const * const tokenData, uint8_t tokenDataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == tokenData) || (0 == tokenDataSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_TOKEN_ACTIVITY, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = tokenType;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = frameIndex;
    size += 1;

    for(byteIndex = 0; byteIndex < tokenDataSize; ++byteIndex)
    {
        txMsg.data[4 + byteIndex] = tokenData[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Stream Data with zone
 * 
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] sequenceNumber Sequence number which is increased by one for every new event with stream
 * data. The sequence number is increased for every frame sent going from 0 - 255 and back to 0 and so
 * on if needed. A new stream starts with a sequence number of 0.
 * @param[in] streamData Stream data. (array[4])
 * @param[in] streamDatasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendStreamDataWithZone(uint8_t zone, uint8_t subZone, uint8_t sequenceNumber, uint8_t const * const streamData, uint8_t streamDataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == streamData) || (0 == streamDataSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_STREAM_DATA_WITH_ZONE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = zone;
    size += 1;

    txMsg.data[1] = subZone;
    size += 1;

    txMsg.data[2] = sequenceNumber;
    size += 1;

    for(byteIndex = 0; byteIndex < streamDataSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = streamData[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Confirm
 * 
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] sequenceNumber Sequence number.
 * @param[in] vscpClass VSCP class.
 * @param[in] vscpType VSCP type.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendConfirm(uint8_t zone, uint8_t subZone, uint8_t sequenceNumber, uint16_t vscpClass, uint16_t vscpType)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_CONFIRM, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = zone;
    size += 1;

    txMsg.data[1] = subZone;
    size += 1;

    txMsg.data[2] = sequenceNumber;
    size += 1;

    txMsg.data[3] = (uint8_t)((vscpClass >> 8) & 0xff);
    txMsg.data[4] = (uint8_t)((vscpClass >> 0) & 0xff);
    size += 2;

    txMsg.data[5] = (uint8_t)((vscpType >> 8) & 0xff);
    txMsg.data[6] = (uint8_t)((vscpType >> 0) & 0xff);
    size += 2;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Level Changed
 * 
 * @param[in] level Relative or absolute level.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendLevelChanged(uint8_t level, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_LEVEL_CHANGED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = level;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Warning
 * 
 * @param[in] level Relative or absolute level.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendWarning(uint8_t level, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_WARNING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = level;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * State
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] from State changed from (old state).
 * @param[in] to State changed to (new state).
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendState(uint8_t userSpecific, uint8_t zone, uint8_t subZone, uint8_t from, uint8_t to)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_STATE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = from;
    size += 1;

    txMsg.data[4] = to;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Action Trigger
 * 
 * @param[in] actionId Action id.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendActionTrigger(uint8_t actionId, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ACTION_TRIGGER, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = actionId;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Sunrise
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendSunrise(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SUNRISE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Sunset
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendSunset(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SUNSET, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Start of record
 * 
 * @param[in] index Index for record.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] remainingFrames Number of frames to follow or zero for not used.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendStartOfRecord(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t remainingFrames)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_START_OF_RECORD, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = remainingFrames;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * End of record
 * 
 * @param[in] index Index for record.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendEndOfRecord(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_END_OF_RECORD, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Pre-set active
 * 
 * @param[in] user User specified.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] preSetCode Code for pre-set that has been set.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendPreSetActive(uint8_t user, uint8_t zone, uint8_t subZone, uint8_t preSetCode)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_PRESET_ACTIVE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = user;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[2] = preSetCode;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Detect
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDetect(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DETECT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Overflow
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendOverflow(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_OVERFLOW, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Big level changed
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] level Level as signed Integer. The range can be adjusted by the user by sending the
 * needed number of bytes 1-5. (array[5])
 * @param[in] levelsize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendBigLevelChanged(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t const * const level, uint8_t levelSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == level) || (0 == levelSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_BIG_LEVEL_CHANGED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < levelSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = level[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Civil sunrise twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendCivilSunriseTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SUNRISE_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Civil sunset twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendCivilSunsetTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SUNSET_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Nautical sunrise twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendNauticalSunriseTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_NAUTICAL_SUNRISE_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Nautical sunset twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendNauticalSunsetTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_NAUTICAL_SUNSET_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Astronomical sunrise twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendAstronomicalSunriseTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ASTRONOMICAL_SUNRISE_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Astronomical sunset twilight time
 * 
 * @param[in] userSpecific User specific.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendAstronomicalSunsetTwilightTime(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ASTRONOMICAL_SUNSET_TWILIGHT_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Calculated Noon
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendCalculatedNoon(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_CALCULATED_NOON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter up
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterUp(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_UP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter down
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterDown(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_DOWN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter left
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterLeft(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_LEFT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter right
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterRight(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_RIGHT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached top end
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedTopEnd(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_TOP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached bottom end
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedBottomEnd(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_BOTTOM, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached middle end
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedMiddleEnd(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_MIDDLE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached preset end
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedPresetEnd(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_PRESET, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached preset left
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedPresetLeft(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_LEFT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Shutter reached preset right
 * 
 * @param[in] reserved Reserved.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendShutterReachedPresetRight(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SHUTTER_END_RIGHT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Long click
 * 
 * @param[in] index Index for device generating long click. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendLongClick(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_LONG_CLICK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Single click
 * 
 * @param[in] index Index for device generating single click. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendSingleClick(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_SINGLE_CLICK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Double click
 * 
 * @param[in] index Index for device generating double click. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDoubleClick(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DOUBLE_CLICK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Date
 * 
 * @param[in] index Index for device generating date. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] yearCentury Year century.
 * @param[in] year Year.
 * @param[in] month Month (1-12).
 * @param[in] day Day.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDate(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t yearCentury, uint8_t year, uint8_t month, uint8_t day)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DATE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = yearCentury;
    size += 1;

    txMsg.data[4] = year;
    size += 1;

    txMsg.data[5] = month;
    size += 1;

    txMsg.data[6] = day;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Time
 * 
 * @param[in] index Index for device generating time. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] hour Hour (0-23).
 * @param[in] minutes Minutes (0-59).
 * @param[in] seconds Seconds (0-59).
 * @param[in] milliseconds Milliseconds (0-999).
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendTime(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t hour, uint8_t minutes, uint8_t seconds, uint16_t milliseconds)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_TIME, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = hour;
    size += 1;

    txMsg.data[4] = minutes;
    size += 1;

    txMsg.data[5] = seconds;
    size += 1;

    txMsg.data[6] = (uint8_t)((milliseconds >> 8) & 0xff);
    txMsg.data[7] = (uint8_t)((milliseconds >> 0) & 0xff);
    size += 2;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Weekday
 * 
 * @param[in] index Index for device generating weekday info. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] weekday Weekday ( 0 (Monday) - 6 (Sunday)).
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendWeekday(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t weekday)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_WEEKDAY, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = weekday;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Lock
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendLock(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_LOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Unlock
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendUnlock(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_UNLOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * DateTime
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * @param[in] dateTime Date/time in five byte bit array. (array[5])
 * @param[in] dateTimesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDatetime(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t const * const dateTime, uint8_t dateTimeSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == dateTime) || (0 == dateTimeSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DATETIME, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < dateTimeSize; ++byteIndex)
    {
        txMsg.data[2 + byteIndex] = dateTime[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Rising
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendRising(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_RISING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Falling
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendFalling(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_FALLING, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Updated
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendUpdated(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_UPDATED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Connect
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendConnect(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_CONNECT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Disconnect
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDisconnect(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DISCONNECT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Reconnect
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendReconnect(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_RECONNECT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Enter
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendEnter(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_ENTER, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Exit
 * 
 * @param[in] index Index for device. Set to zero if not used.
 * @param[in] zone Zone for which event applies to (0-254). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-254). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendExit(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_EXIT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Incremented
 * 
 * @param[in] userSpecific Optional byte that have a meaning given by the issuer of the event.
 * Prefered use is _index_ for compability.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones
 * @param[in] subzone Sub-zone for which event applies to (0-255). 255 is all sub-zones
 * @param[in] incrementedValue Incremented value. The range can be adjusted by the user by sending
 * just the needed number of bytes 1-5 (optional) (array[5])
 * @param[in] incrementedValuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendIncremented(uint8_t userSpecific, uint8_t zone, uint8_t subzone, uint8_t const * const incrementedValue, uint8_t incrementedValueSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_INCREMENTED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subzone;
    size += 1;

    if (NULL != incrementedValue)
    {
        for(byteIndex = 0; byteIndex < incrementedValueSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = incrementedValue[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Decremented
 * 
 * @param[in] userSpecific Optional byte that have a meaning given by the issuer of the event.
 * Prefered use is _index_ for compability.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones
 * @param[in] subzone Sub-zone for which event applies to (0-255). 255 is all sub-zones
 * @param[in] decrementedValue Decremented value. The range can be adjusted by the user by sending
 * just the needed number of bytes 1-5 (optional) (array[5])
 * @param[in] decrementedValuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendDecremented(uint8_t userSpecific, uint8_t zone, uint8_t subzone, uint8_t const * const decrementedValue, uint8_t decrementedValueSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_DECREMENTED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subzone;
    size += 1;

    if (NULL != decrementedValue)
    {
        for(byteIndex = 0; byteIndex < decrementedValueSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = decrementedValue[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Proximity detected
 * 
 * @param[in] userSpecific Optional byte that have a meaning given by the issuer of the event.
 * Prefered use is _index_ for compability.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones
 * @param[in] subzone Sub-zone for which event applies to (0-255). 255 is all sub-zones
 * @param[in] proximityLevel Optional uint16 that sets proximity level if present. (optional)
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_information_sendProximityDetected(uint8_t userSpecific, uint8_t zone, uint8_t subzone, uint16_t const * const proximityLevel)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_INFORMATION, VSCP_TYPE_INFORMATION_PROXIMITY_DETECTED, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subzone;
    size += 1;

    if (NULL != proximityLevel)
    {
        txMsg.data[3] = (uint8_t)((*proximityLevel >> 8) & 0xff);
        txMsg.data[4] = (uint8_t)((*proximityLevel >> 0) & 0xff);
        size += 2;
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

