/* The MIT License (MIT)
 *
 * Copyright (c) 2014 - 2025 Andreas Merkle
 * http://www.blue-andi.de
 * vscp@blue-andi.de
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*******************************************************************************
    DESCRIPTION
*******************************************************************************/
/**
@brief  VSCP Protocol Functionality
@file   vscp_type_protocol.h
@author Andreas Merkle, http://www.blue-andi.de

@section desc Description
This class defines some types that must be implemented by every node that implements the VSCP
protocol. The types in this class must be handled by all level I and Level II nodes. Note also that
this class is repeated as Level II class=512 with the only difference that GUID's are used instead
of nicknames. This means that for all Level II class=512 events the data described here is preceded
by the 16-bit destination GUID (address of receiver) stored with MSB first followed by the data
specified here. Nickname is present also if documented below but have no meaning and should be
discarded.

This file is automatically generated. Don't change it manually.

*******************************************************************************/

#ifndef __VSCP_TYPE_PROTOCOL_H__
#define __VSCP_TYPE_PROTOCOL_H__

/** @defgroup vscp_type_protocol VSCP Protocol Functionality
 * Level 1 protocol class types
 * @{
 * @ingroup vscp_l1
 */

/*******************************************************************************
    INCLUDES
*******************************************************************************/

#ifdef __cplusplus
extern "C"
{
#endif

/*******************************************************************************
    COMPILER SWITCHES
*******************************************************************************/

/*******************************************************************************
    CONSTANTS
*******************************************************************************/

/**
 * General Event.
 */
#define VSCP_TYPE_PROTOCOL_GENERAL                     0

/**
 * Not mandatory. Implement in device if needed by application.
 * A segment controller sends this event once a second on the segment that it controls. The data field
 * contains the 8-bit CRC of the segment controller GUID and the time since the epoch (00:00:00 UTC,
 * January 1, 1970) as a 32-bit value. A node that receive (and recognize) this event could respond
 * with a CLASS1.INFORMATION, Type=9 event (HEARTBEAT) and should do so if it does not send out a
 * regular heartbeat event.
 * Other nodes can originate this event on the segment. For these nodes the data part, as specified
 * below, should be omitted. A better choice for periodic heartbeat events from a node may be
 * CLASS1.INFORMATION, Type=9 (HEARTBEAT)
 * All nodes that recognize this event should save the 8-bit CRC in non-volatile storage and use it on
 * power up. When a node starts up on a segment it should begin to listen for the Segment controller
 * heartbeat. When/if it is received the node compares it with the stored value and if equal and the
 * node is assigned a nickname-ID it continues to its working mode. If different, the node has
 * detected that it has been moved to a new segment and therefore must drop its nickname-ID and enters
 * the configuration mode to obtain a new nickname-ID from the segment controller.
 * If the node is in working mode and its nickname-ID changes, the node should do a complete restart
 * after first setting all controls to their default state.
 * As a segment can be without a segment controller this event is not available on all segments and is
 * not mandatory.
 */
#define VSCP_TYPE_PROTOCOL_SEGCTRL_HEARTBEAT           1

/**
 * Mandatory. Must be implemented by all devices.
 * This is intended for nodes that have been initiated, is part of the segment and is powered up. All
 * nodes that have a nickname-ID that is not set to 0xFF should send this event before they go on line
 * to do their “day to day” work.
 * Normally all nodes should save their assigned nickname-ID in non-volatile memory and use this
 * assigned ID when powered up. A segment controller can however keep track of nodes that it controls
 * and reassign the ID to a node that it did not get a new node on-line event from. This is the method
 * a segment controller uses to detect nodes that have been removed from the segment.
 * For the nickname discovery procedure this event is used as the probe. The difference between a
 * probe and a new node on line is that the later has the same originating nickname as value in byte
 * 0.
 * If a node send this event with the unassigned ID 0xFF and byte 0 set to 0xFF it has given up the
 * search for a free ID.
 * It is recommended that also level II nodes send this event when they come alive. In this case the
 * target address is the 16-byte data GUID of the node with MSB in the first byte.
 */
#define VSCP_TYPE_PROTOCOL_NEW_NODE_ONLINE             2

/**
 * Mandatory. Must be implemented by all devices.
 * This event is sent from a node as a response to a probe. There are no arguments.
 */
#define VSCP_TYPE_PROTOCOL_PROBE_ACK                   3

/**
 * Reserved for future use.
 */
#define VSCP_TYPE_PROTOCOL_RESERVED4                   4

/**
 * Reserved for future use.
 */
#define VSCP_TYPE_PROTOCOL_RESERVED5                   5

/**
 * Mandatory. Must be implemented by all devices.
 * This event can be used to change the nickname for a node. The node just uses the new nickname and
 * don't start nickname discovery or similar.
 */
#define VSCP_TYPE_PROTOCOL_SET_NICKNAME                6

/**
 * Mandatory. Must be implemented by all devices.
 * A node sends this event to confirm that it accepts its assigned nickname-ID. When sending this
 * event the node uses its newly assigned nickname address.
 */
#define VSCP_TYPE_PROTOCOL_NICKNAME_ACCEPTED           7

/**
 * Mandatory. Must be implemented by all devices.
 * Request a node to drop its nickname. The node should drop its nickname and then behave in the same
 * manner as when it was first powered up on the segment.
 */
#define VSCP_TYPE_PROTOCOL_DROP_NICKNAME               8

/**
 * Mandatory. Must be implemented by all devices.
 * Read a register from a node.
 * If a node have several pages with user defined registers Extended Register Read is a better choice
 * to choose for reading as the page also is set when reading register using that type. The standard
 * registers can always be read without setting a page though as they are always mapped into the upper
 * 128 bytes.
 */
#define VSCP_TYPE_PROTOCOL_READ_REGISTER               9

/**
 * Mandatory. Must be implemented by all devices.
 * Response for a read/write event. . Note that the data is returned for both a read and a write and
 * can and probably should be checked for validity.
 */
#define VSCP_TYPE_PROTOCOL_RW_RESPONSE                 10

/**
 * Mandatory. Must be implemented by all devices.
 * Write register content to a node.
 * If a node have several pages with user defined registers Extended Register Write is a better choice
 * to choose for writing as the page is also set when writing a register using that type. The standard
 * registers can always be read without setting a page though as they are always mapped into the upper
 * 128 bytes.
 */
#define VSCP_TYPE_PROTOCOL_WRITE_REGISTER              11

/**
 * Mandatory. Must be implemented by all devices.
 * Send NACK (Class=0,Type=14 if no boot-loader implemented)
 * This is the first event in the boot loader sequence. The node should stop all other activities when
 * in boot loader mode. This also means that the node should not react on other events (commands) then
 * the boot loader related.
 */
#define VSCP_TYPE_PROTOCOL_ENTER_BOOT_LOADER           12

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * This event has no meaning for any node that is not in boot mode and should be disregarded.
 * The node confirms that it has entered boot loader mode. This is only sent for the VSCP boot loader
 * algorithm.
 */
#define VSCP_TYPE_PROTOCOL_ACK_BOOT_LOADER             13

/**
 * Mandatory. Should be implemented by all devices.
 * The node was unable to enter boot loader mode. The reason is given by a user specified error code
 * byte. This event has no meaning for any node that is not in boot mode and should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_NACK_BOOT_LOADER            14

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Begin transfer of data for a block of memory. This event has no meaning for any node that is not in
 * boot mode and should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_START_BLOCK                 15

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Data for a block of memory. This event has no meaning for any node that is not in boot mode and
 * should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_BLOCK_DATA                  16

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Confirm the reception of a complete data block. This event has no meaning for any node that is not
 * in boot mode and should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_BLOCK_DATA_ACK              17

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * NACK the reception of data block. This event has no meaning for any node that is not in boot mode
 * and should be disregarded.
 * The state machine of the node that is loaded with firmware should accept new start block data
 * events after this event. Other memory types can be programmed.
 */
#define VSCP_TYPE_PROTOCOL_BLOCK_DATA_NACK             18

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Request from a node to program a data block that has been uploaded and confirmed. This event has no
 * meaning for any node that is not in boot mode and should be disregarded.
 * Sent only if the block was successfully received and confirmed by checking the crc for the full
 * block. The block number is the number of the block that was sent in the last block data event.
 */
#define VSCP_TYPE_PROTOCOL_PROGRAM_BLOCK_DATA          19

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * A node confirms the successful programming of a block. This event has no meaning for any node that
 * is not in boot mode and should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_PROGRAM_BLOCK_DATA_ACK      20

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * A node failed to program a data block. This event has no meaning for any node that is not in boot
 * mode and should be disregarded.
 */
#define VSCP_TYPE_PROTOCOL_PROGRAM_BLOCK_DATA_NACK     21

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * This command is sent as the last command during the boot-loader sequence. It resets the device and
 * starts it up using the newly loaded code. The 16-bit sum of all CRC blocks that was transferred to
 * the node (all memory types) is sent as an argument. This sum should be checked and be correct for
 * the reset/activation to be performed. Activate new image NACK will be sent if the CRC is not
 * correct and the node will not leave boot loader mode.
 * If just one memory type is programmed, the CRC sum is the same as the CRC for the programmed block.
 * This can be used as an alternative way to program different memory types, that is enter boot loader
 * mode, program an area, and then activate the new image, and then enter boot loader mode again and
 * program another area, and so on.
 */
#define VSCP_TYPE_PROTOCOL_ACTIVATE_NEW_IMAGE          22

/**
 * Mandatory. Should be implemented by all devices.
 * Added in version 1.4.0
 * This is a variant of Class=0, Type=8 but here the full GUID is used instead of the nickname to
 * identify the node that should drop its current nickname and enter the node-name discovery
 * procedure.
 * As the GUID is 16 bytes this is a multi-frame event. To ease the storage requirements on the nodes
 * only four GUID bytes are send in each frame. The frames must be sent out within one second
 * interval.
 */
#define VSCP_TYPE_PROTOCOL_RESET_DEVICE                23

/**
 * Mandatory. Should be implemented by all devices.
 * The page read is implemented to make it possible to read/write larger blocks of data. Two register
 * positions are reserved to select a base into this storage. This is a 16-bit number pointing to a
 * 256-byte page. This means that a total of 65535 * 256 bytes are accessible with this method (page 0
 * is the standard registers).
 * To read a block of data from the storage, first write the base registers then issue this event and
 * n events will be sent out from the node containing the data from the specified area. If the count
 * pass the border it of the page ( > 0xFF) the transfer will end there.
 * Note that the page select registers only selects a virtual page that can be accessed with page
 * read/write and not with the ordinary read/write.
 */
#define VSCP_TYPE_PROTOCOL_PAGE_READ                   24

/**
 * Mandatory. Should be implemented by all devices.
 * The write page is implemented to make it possible to write larger blocks of data. One data-space
 * positions is reserved to select a base into this storage. See Page read for a full description.
 * It is only possible to write one 6-byte chunk at a time in contrast to reading several. This is
 * because VSCP at Level I is aimed at low end devices with limited resources meaning little room for
 * buffers.
 */
#define VSCP_TYPE_PROTOCOL_PAGE_WRITE                  25

/**
 * Mandatory. Should be implemented by all devices.
 * This is a response frame for the read/write page command. The Sequence number goes from 0 up to the
 * last sent frame for a read page request.
 */
#define VSCP_TYPE_PROTOCOL_RW_PAGE_RESPONSE            26

/**
 * Should be implemented by all devices that work over 802.15.4/Ethernet/Internet or other high end
 * protocols.This event can be broadcasted on a segment by a node to get information about available
 * servers.
 * The VSCP daemon documentation have a description on how server/service discovery works.
 */
#define VSCP_TYPE_PROTOCOL_HIGH_END_SERVER_PROBE       27

/**
 * Should be implemented by all devices that work on 802.15.4/Ethernet/Internet and have a Level I
 * link. This is because a Level II device can be present on a Level I bus. A typical example is a
 * Bluetooth gateway. A user find the bud/segment by the Bluetooth device and can then discover other
 * parts of the system through it.
 * A Level II node respond with CLASS2.PROTOCOL, Type=32 Level II who is response to this event. It is
 * also possible to listen for  CLASS2.PROTOCOL, Type=20 (0x14) High end server capabilities to
 * discover Level II nodes.
 */
#define VSCP_TYPE_PROTOCOL_HIGH_END_SERVER_RESPONSE    28

/**
 * Mandatory. Should be implemented by all devices.
 * Increment a register content by one with no risk of it changing in between
 */
#define VSCP_TYPE_PROTOCOL_INCREMENT_REGISTER          29

/**
 * Mandatory. Should be implemented by all devices.
 * Decrement a register content by one with no risk of it changing in between
 */
#define VSCP_TYPE_PROTOCOL_DECREMENT_REGISTER          30

/**
 * Mandatory. Must be implemented by all devices.
 * This event can be used as a fast way to find out which nodes there is on a segment. All nodes
 * receiving it should respond.
 */
#define VSCP_TYPE_PROTOCOL_WHO_IS_THERE                31

/**
 * Mandatory. Must be implemented by all devices.
 * Response from node(s) looks like this
 */
#define VSCP_TYPE_PROTOCOL_WHO_IS_THERE_RESPONSE       32

/**
 * Mandatory
 * Request a node to report size and offset for its decision matrix.
 */
#define VSCP_TYPE_PROTOCOL_GET_MATRIX_INFO             33

/**
 * Mandatory for nodes with a decision matrix
 * Report the size for the decision matrix and the offset to its storage. The reported size is the
 * number of decision matrix lines. The offset is the offset in the register address counter from 0x00
 * (See the register model in this document). If the size returned is zero the node does not have a
 * decision matrix. A node without a decision matrix can also skip to implement this event but it's
 * better if it returns a decision matrix size of zero.
 */
#define VSCP_TYPE_PROTOCOL_GET_MATRIX_INFO_RESPONSE    34

/**
 * Not mandatory.
 * A node that get this event and has an embedded MDF description in flash or similar respond with the
 * description .
 */
#define VSCP_TYPE_PROTOCOL_GET_EMBEDDED_MDF            35

/**
 * Not mandatory.
 * This is the response from a Get embedded MDF. The response consist of several frames where an index
 * in byte0/1 is incremented for each frame and MDF data is in byte 2-7.
 * If an embedded MDF is not available a response on the form
 *  byte 0 = 0 
 *  byte 1 = 0 
 *  byte 2 = 0
 * should be sent.
 */
#define VSCP_TYPE_PROTOCOL_GET_EMBEDDED_MDF_RESPONSE   36

/**
 * Mandatory. Must be implemented by all devices.
 * Read a register from a node with page information.
 * Implementation must take care so all page register change done by these routines must restore the
 * content of the page registers to there original content when they are done.
 */
#define VSCP_TYPE_PROTOCOL_EXTENDED_PAGE_READ          37

/**
 * Mandatory. Must be implemented by all devices.
 * Write register content to a node.
 * Implementation must take care so all page register change done by these routines must restore the
 * content of the page registers to there original content when they are done.
 */
#define VSCP_TYPE_PROTOCOL_EXTENDED_PAGE_WRITE         38

/**
 * Mandatory. Must be implemented by all devices.
 * This is the replay sent for events CLASS1.PROTOCOL, Type=40,41.
 */
#define VSCP_TYPE_PROTOCOL_EXTENDED_PAGE_RESPONSE      39

/**
 * Not Mandatory. Implemented if needed.
 * It is possible to ask a node which event(s) it is interested in with this event. If not implemented
 * the node is supposed to be interested in all events.
 * All nodes are by default interested in CLASS1.PROTOCOL.
 * The event is intended for very low bandwidth nodes like low power wireless nodes where it saves a
 * lot of bandwidth if only events that really concerns the node is sent to them.
 */
#define VSCP_TYPE_PROTOCOL_GET_EVENT_INTEREST          40

/**
 * Not mandatory. Implemented if needed.
 * Response for event CLASS1.PROTOCOL, Type=40 (Get event interest). The node report all events it is
 * interested in.
 */
#define VSCP_TYPE_PROTOCOL_GET_EVENT_INTEREST_RESPONSE 41

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the positive response after a node received a
 * CLASS1.PROTOCOL, Type=22 (Activate new image). It is sent by the node before the new firmware is
 * booted into.
 */
#define VSCP_TYPE_PROTOCOL_ACTIVATE_NEW_IMAGE_ACK      48

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the negative response after a node received a
 * CLASS1.PROTOCOL, Type=22 (Activate new image). It is sent by the node to inform it that it will (or
 * can not) switch to the new firmware image.
 */
#define VSCP_TYPE_PROTOCOL_ACTIVATE_NEW_IMAGE_NACK     49

/**
 * Not mandatory Only needed if a VSCP boot loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the positive response after a node received a
 * CLASS1.PROTOCOL, Type=15 (Block data) event (a part of a block). It is sent by the node as a
 * validation that it can handle the block data transfer.
 */
#define VSCP_TYPE_PROTOCOL_START_BLOCK_ACK             50

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the negative response after a node received a
 * CLASS1.PROTOCOL, Type=15 (Block data) event (a part of a block). It is sent by the node as an
 * indication that it can NOT handle the block data transfer.
 */
#define VSCP_TYPE_PROTOCOL_START_BLOCK_NACK            51

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the positive response after a node received
 * CLASS1.PROTOCOL, Type=16 (Block data) event (a part of a block). It is sent by the node as a
 * validation that it can handle the block data transfer.
 */
#define VSCP_TYPE_PROTOCOL_BLOCK_CHUNK_ACK             52

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This is the negative response after a node received
 * CLASS1.PROTOCOL, Type=16 (Block data) event (a part of a block). It is sent by the node as an
 * indication that it can NOT handle the block data transfer.
 */
#define VSCP_TYPE_PROTOCOL_BLOCK_CHUNK_NACK            53

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This event is a way to check if a device is already in
 * bootloader mode. A device that is in bootloader mode should respond with a CLASS1.PROTOCOL, Type=13
 * (ACK boot loader mode) event regardless of the internal state it is in.
 */
#define VSCP_TYPE_PROTOCOL_BOOT_LOADER_CHECK           54

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This event provide a way to leave the bootloader in a
 * secure fashion if there is problems loading firmware to a device. It is only available if the
 * device has a bootloader that supports this functionality. Typically this is a device that has
 * several firmware slots and can switch between them, and by that switch back to the last known
 * working firmware.
 * The event can be sent in all states of the bootloading procedure.
 * CLASS1.PROTOCOL, Type=56 (Bootloader abort ACK), should be sent as a positive response to a
 * bootloader exit or rollback.
 * CLASS1.PROTOCOL, Type=57 (Bootloader abort NACK), should be sent as a negative response to a
 * bootloader exit or rollback.
 */
#define VSCP_TYPE_PROTOCOL_BOOT_LOADER_ABORT           55

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This event is a positive response to a bootloader abort
 * or rollback CLASS1.PROTOCOL, Type=55 (Bootloader abort).
 * Event is sent before execution of the new firmware starts. It is used to tell the controlling
 * device that the device is ready to leave the bootloader mode and start the new firmware.
 */
#define VSCP_TYPE_PROTOCOL_BOOT_LOADER_ABORT_ACK       56

/**
 * Not mandatory. Only needed if a VSCP boot-loader algorithm is used.
 * Part of the VSCP boot-loader functionality. This event is a negative response to a bootloader abort
 * or rollback CLASS1.PROTOCOL, Type=55 (Bootloader abort).
 * On a single slot firmware device where a firmware update has failed and there is no working
 * firmware to switch back to this event should be sent to tell the controlling device that we can't
 * leave the bootloader mode until a new full firmware has been loaded.
 */
#define VSCP_TYPE_PROTOCOL_BOOT_LOADER_ABORT_NACK      57

/*******************************************************************************
    MACROS
*******************************************************************************/

/*******************************************************************************
    TYPES AND STRUCTURES
*******************************************************************************/

/*******************************************************************************
    VARIABLES
*******************************************************************************/

/*******************************************************************************
    FUNCTIONS
*******************************************************************************/

#ifdef __cplusplus
}
#endif

#endif  /* __VSCP_TYPE_PROTOCOL_H__ */

/** @} */
