/* The MIT License (MIT)
 *
 * Copyright (c) 2014 - 2025 Andreas Merkle
 * http://www.blue-andi.de
 * vscp@blue-andi.de
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*******************************************************************************
    DESCRIPTION
*******************************************************************************/
/**
@brief  Control
@file   vscp_evt_control.c
@author Andreas Merkle, http://www.blue-andi.de

@section desc Description
@see vscp_evt_control.h

This file is automatically generated. Don't change it manually.

*******************************************************************************/

/*******************************************************************************
    INCLUDES
*******************************************************************************/
#include "vscp_evt_control.h"
#include "../core/vscp_core.h"
#include "../core/vscp_class_l1.h"
#include "../core/vscp_type_control.h"

/*******************************************************************************
    COMPILER SWITCHES
*******************************************************************************/

/*******************************************************************************
    CONSTANTS
*******************************************************************************/

/*******************************************************************************
    MACROS
*******************************************************************************/

/*******************************************************************************
    TYPES AND STRUCTURES
*******************************************************************************/

/*******************************************************************************
    VARIABLES
*******************************************************************************/

/*******************************************************************************
    FUNCTIONS
*******************************************************************************/

/**
 * General event
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendGeneralEvent(void)
{
    vscp_TxMessage  txMsg;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_GENERAL, VSCP_PRIORITY_3_NORMAL);

    txMsg.dataSize = 0;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Mute on/off
 * 
 * @param[in] command If equal to zero no mute else mute.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMuteOnOff(uint8_t command, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_MUTE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = command;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * (All) Lamp(s) on/off
 * 
 * @param[in] state If equal to zero off else on.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendAllLampSOnOff(uint8_t state, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_ALL_LAMPS, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = state;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Open
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendOpen(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_OPEN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Close
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendClose(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_CLOSE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * TurnOn
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendTurnon(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TURNON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * TurnOff
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendTurnoff(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TURNOFF, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Start
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendStart(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_START, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Stop
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendStop(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_STOP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Reset
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendReset(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_RESET, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Interrupt
 * 
 * @param[in] interruptLevel Interrupt level. (0 – 255 , zero is lowest interrupt level. ).
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendInterrupt(uint8_t interruptLevel, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_INTERRUPT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = interruptLevel;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Sleep
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSleep(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SLEEP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Wakeup
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendWakeup(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_WAKEUP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Resume
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendResume(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_RESUME, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Pause
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendPause(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_PAUSE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Activate
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendActivate(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_ACTIVATE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Deactivate
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendDeactivate(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_DEACTIVATE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/* "Reserved for future use" not supported. No frame defined. */

/* "Reserved for future use" not supported. No frame defined. */

/* "Reserved for future use" not supported. No frame defined. */

/**
 * Dim lamp(s)
 * 
 * @param[in] value Value (0 – 100) . 0 = off, 100 = full on. 254 dim down one step. 255 dim up one
 * step.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendDimLampS(uint8_t value, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_DIM_LAMPS, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = value;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Change Channel
 * 
 * @param[in] channel A value between 0 and 127 indicates the channel number. A value between 128 to
 * 157 is change down by the specified number of channels. A value between 160 to 191 is change up by
 * the specified number of channels. A value of 255 means that this is an extended change channel
 * event and that the channel number is sent in byte 3 and after if needed.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendChangeChannel(uint8_t channel, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_CHANGE_CHANNEL, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = channel;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Change Level
 * 
 * @param[in] level Absolute level.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendChangeLevel(uint8_t level, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_CHANGE_LEVEL, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = level;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Relative Change Level
 * 
 * @param[in] level Relative level.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendRelativeChangeLevel(uint8_t level, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_RELATIVE_CHANGE_LEVEL, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = level;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Measurement Request
 * 
 * @param[in] index Zero indicates all measurements supported by node should be sent (as separate
 * events). Non-zero indicates a node specific index specifying which measurement to send.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMeasurementRequest(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_MEASUREMENT_REQUEST, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Stream Data
 * 
 * @param[in] index Sequence number which is increase by one for each stream data event sent.
 * @param[in] data  Stream data. (optional) (array[7])
 * @param[in] datasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendStreamData(uint8_t index, uint8_t const * const data, uint8_t dataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_STREAM_DATA, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    if (NULL != data)
    {
        for(byteIndex = 0; byteIndex < dataSize; ++byteIndex)
        {
            txMsg.data[1 + byteIndex] = data[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Sync
 * 
 * @param[in] index Sensor index for a sensor within a module (see data coding). 255 is all sensors.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSync(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SYNC, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Zoned Stream Data
 * 
 * @param[in] index Sequence number which is increase by one for each stream data event sent.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] data  Stream data. (optional) (array[5])
 * @param[in] datasize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendZonedStreamData(uint8_t index, uint8_t zone, uint8_t subZone, uint8_t const * const data, uint8_t dataSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_ZONED_STREAM_DATA, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    if (NULL != data)
    {
        for(byteIndex = 0; byteIndex < dataSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = data[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set Pre-set
 * 
 * @param[in] presetCode Code for pre-set to set.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetPreSet(uint8_t presetCode, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_PRESET, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = presetCode;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Toggle state
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendToggleState(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TOGGLE_STATE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Timed pulse on
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] control Control byte.
 * @param[in] time Set time as a long with MSB in the first byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendTimedPulseOn(uint8_t userSpecific, uint8_t zone, uint8_t subZone, uint8_t control, uint32_t time)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TIMED_PULSE_ON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = control;
    size += 1;

    txMsg.data[4] = (uint8_t)((time >> 24) & 0xff);
    txMsg.data[5] = (uint8_t)((time >> 16) & 0xff);
    txMsg.data[6] = (uint8_t)((time >> 8) & 0xff);
    txMsg.data[7] = (uint8_t)((time >> 0) & 0xff);
    size += 4;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Timed pulse off
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] control Control byte.
 * @param[in] time Set time as a long with MSB in the first byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendTimedPulseOff(uint8_t userSpecific, uint8_t zone, uint8_t subZone, uint8_t control, uint32_t time)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TIMED_PULSE_OFF, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = control;
    size += 1;

    txMsg.data[4] = (uint8_t)((time >> 24) & 0xff);
    txMsg.data[5] = (uint8_t)((time >> 16) & 0xff);
    txMsg.data[6] = (uint8_t)((time >> 8) & 0xff);
    txMsg.data[7] = (uint8_t)((time >> 0) & 0xff);
    size += 4;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set country/language
 * 
 * @param[in] code Country/Language code.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] codeSpecific Country/Language code specific (array[4])
 * @param[in] codeSpecificsize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetCountryLanguage(uint8_t code, uint8_t zone, uint8_t subZone, uint8_t const * const codeSpecific, uint8_t codeSpecificSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == codeSpecific) || (0 == codeSpecificSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_COUNTRY_LANGUAGE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = code;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < codeSpecificSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = codeSpecific[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Big Change level
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] level Level as signed Integer. The range can be adjusted by the user by sending the
 * needed number of bytes 1-5.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendBigChangeLevel(uint8_t index, uint8_t zone, uint8_t subZone, int32_t level)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_BIG_CHANGE_LEVEL, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    /* level not supported! */

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter up
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterUp(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_UP, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter down
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterDown(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_DOWN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter left
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterLeft(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_LEFT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter right
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterRight(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_RIGHT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter to middle position
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterToMiddlePosition(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_MIDDLE, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Move shutter to preset position
 * 
 * @param[in] index Index.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendMoveShutterToPresetPosition(uint8_t index, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SHUTTER_PRESET, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = index;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * (All) Lamp(s) on
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendAllLampSOn(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_ALL_LAMPS_ON, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * (All) Lamp(s) off
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendAllLampSOff(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_ALL_LAMPS_OFF, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Lock
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendLock(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_LOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Unlock
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-Zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendUnlock(uint8_t userSpecific, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_UNLOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * PWM set
 * 
 * @param[in] repeats Repeat/counter: 0=repeat forever, >0 number of repeats
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] control Control byte.
 * @param[in] timeOn Time-On
 * @param[in] timeOff Time-Off
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendPwmSet(uint8_t repeats, uint8_t zone, uint8_t subZone, uint8_t control, uint16_t timeOn, uint16_t timeOff)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_PWM, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = repeats;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.data[3] = control;
    size += 1;

    txMsg.data[4] = (uint8_t)((timeOn >> 8) & 0xff);
    txMsg.data[5] = (uint8_t)((timeOn >> 0) & 0xff);
    size += 2;

    txMsg.data[6] = (uint8_t)((timeOff >> 8) & 0xff);
    txMsg.data[7] = (uint8_t)((timeOff >> 0) & 0xff);
    size += 2;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Lock with token
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] token Token. This token can be 1-5 bytes and length of event is set accordingly. It
 * should be interpreted as an unsigned integer in the range 0-1099511627775. MSB byte is stored in
 * first byte. (array[5])
 * @param[in] tokensize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendLockWithToken(uint8_t reserved, uint8_t zone, uint8_t subZone, uint8_t const * const token, uint8_t tokenSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == token) || (0 == tokenSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TOKEN_LOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < tokenSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = token[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Unlock with token
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] token Token. This token can be 1-5 bytes and length of event is set accordingly. It
 * should be interpreted as an unsigned integer in the range 0-1099511627775. MSB byte is stored in
 * first byte. (array[5])
 * @param[in] tokensize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendUnlockWithToken(uint8_t reserved, uint8_t zone, uint8_t subZone, uint8_t const * const token, uint8_t tokenSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == token) || (0 == tokenSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_TOKEN_UNLOCK, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < tokenSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = token[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set security level
 * 
 * @param[in] securityLevel Security level to set. 0-255 (Higher value is higher security level).
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetSecurityLevel(uint8_t securityLevel, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_SECURITY_LEVEL, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = securityLevel;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set security pin
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] securityPin Security pin. This pin can be 1-5 bytes and length of event is set
 * accordingly. It should be interpreted as an unsigned integer in the range 0-1099511627775. MSB byte
 * is stored in first byte. (array[5])
 * @param[in] securityPinsize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetSecurityPin(uint8_t reserved, uint8_t zone, uint8_t subZone, uint8_t const * const securityPin, uint8_t securityPinSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == securityPin) || (0 == securityPinSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_SECURITY_PIN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < securityPinSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = securityPin[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set security password
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] securityPassword Security password. This password can be 1-5 bytes and length of event
 * is set accordingly. It should be interpreted as a UTF-8 string of length equal to the event data
 * length minus 3 bytes (array[5])
 * @param[in] securityPasswordsize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetSecurityPassword(uint8_t reserved, uint8_t zone, uint8_t subZone, uint8_t const * const securityPassword, uint8_t securityPasswordSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == securityPassword) || (0 == securityPasswordSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_SECURITY_PASSWORD, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < securityPasswordSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = securityPassword[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Set security token
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * @param[in] token Token. This token can be 1-5 bytes and length of event is set accordingly. It
 * should be interpreted as an unsigned integer in the range 0-1099511627775. MSB byte is stored in
 * first byte. (array[5])
 * @param[in] tokensize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendSetSecurityToken(uint8_t reserved, uint8_t zone, uint8_t subZone, uint8_t const * const token, uint8_t tokenSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    if ((NULL == token) || (0 == tokenSize))
    {
        return FALSE;
    }

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_SET_SECURITY_TOKEN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    for(byteIndex = 0; byteIndex < tokenSize; ++byteIndex)
    {
        txMsg.data[3 + byteIndex] = token[byteIndex];
        size += 1;

        if (VSCP_L1_DATA_SIZE <= size)
        {
            break;
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Request new security token
 * 
 * @param[in] reserved Not used.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones.
 * @param[in] subZone Sub-zone for which event applies to (0-255). 255 is all sub-zones.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendRequestNewSecurityToken(uint8_t reserved, uint8_t zone, uint8_t subZone)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_REQUEST_SECURITY_TOKEN, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = reserved;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subZone;
    size += 1;

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Increment
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones
 * @param[in] subzone Sub-zone for which event applies to (0-255). 255 is all sub-zones
 * @param[in] incrementValue Increment as unsigned integer. The range can be adjusted by the user by
 * sending just the needed number of bytes (1-5) which form the unsigned integer (MSB first). If
 * omitted (or 0) 1 is assumed as default increment value (optional) (array[5])
 * @param[in] incrementValuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendIncrement(uint8_t userSpecific, uint8_t zone, uint8_t subzone, uint8_t const * const incrementValue, uint8_t incrementValueSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_INCREMENT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subzone;
    size += 1;

    if (NULL != incrementValue)
    {
        for(byteIndex = 0; byteIndex < incrementValueSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = incrementValue[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

/**
 * Decrement
 * 
 * @param[in] userSpecific User specific value.
 * @param[in] zone Zone for which event applies to (0-255). 255 is all zones
 * @param[in] subzone Sub-zone for which event applies to (0-255). 255 is all sub-zones
 * @param[in] decrementValue Decrement as unsigned integer. The range can be adjusted by the user by
 * sending just the needed number of bytes (1-5) which form the unsigned integer (MSB first). If
 * omitted (or 0) 1 is assumed as default decrement value (optional) (array[5])
 * @param[in] decrementValuesize Size in byte.
 * 
 * @return If event is sent, it will return TRUE otherwise FALSE.
 */
extern BOOL vscp_evt_control_sendDecrement(uint8_t userSpecific, uint8_t zone, uint8_t subzone, uint8_t const * const decrementValue, uint8_t decrementValueSize)
{
    vscp_TxMessage  txMsg;
    uint8_t         size    = 0;
    uint8_t         byteIndex   = 0;

    vscp_core_prepareTxMessage(&txMsg, VSCP_CLASS_L1_CONTROL, VSCP_TYPE_CONTROL_DECREMENT, VSCP_PRIORITY_3_NORMAL);

    txMsg.data[0] = userSpecific;
    size += 1;

    txMsg.data[1] = zone;
    size += 1;

    txMsg.data[2] = subzone;
    size += 1;

    if (NULL != decrementValue)
    {
        for(byteIndex = 0; byteIndex < decrementValueSize; ++byteIndex)
        {
            txMsg.data[3 + byteIndex] = decrementValue[byteIndex];
            size += 1;

            if (VSCP_L1_DATA_SIZE <= size)
            {
                break;
            }
        }
    }

    txMsg.dataSize = size;

    return vscp_core_sendEvent(&txMsg);
}

