#define CATCH_CONFIG_MAIN
#include "catch2/catch.hpp"

// Include painlessMesh configuration to ensure macros are defined before testing them
#include "painlessTaskOptions.h"

/**
 * Unit tests for v1.7.6 fix
 * 
 * This test verifies that the problematic scheduler_queue files have been
 * removed and are not causing compilation errors. The thread-safe scheduler
 * queue feature was removed because:
 * 
 * 1. It required _TASK_THREAD_SAFE macro to be defined
 * 2. _TASK_THREAD_SAFE conflicts with _TASK_STD_FUNCTION in TaskScheduler v4.0.x
 * 3. painlessMesh requires _TASK_STD_FUNCTION for lambda callbacks
 * 4. Result: scheduler_queue.cpp tried to compile but _task_request_t type was undefined
 * 
 * The fix maintains ~85% FreeRTOS crash reduction via semaphore timeout increase.
 * 
 * @see docs/releases/RELEASE_PLAN_v1.7.6.md
 * @see docs/troubleshooting/PAINLESSMESH_V1.7.4_COMPILATION_ISSUES.md
 */

TEST_CASE("scheduler_queue files should not cause compilation errors", "[v1.7.6][compilation][fix]") {
    // This test verifies that the problematic scheduler_queue files
    // have been removed and are not breaking the build
    
    SECTION("scheduler_queue.hpp should not be included") {
        // If scheduler_queue.hpp was included, it would define a macro
        // We verify it's not present by checking the macro is undefined
        #ifdef PAINLESSMESH_SCHEDULER_QUEUE_HPP_INCLUDED
        FAIL("scheduler_queue.hpp should not be included - file should be deleted");
        #endif
        
        REQUIRE(true); // If we get here, the header is not included
    }
    
    SECTION("Implementation compiles successfully") {
        // If this test compiles and runs, scheduler_queue.cpp is not breaking the build
        // The problematic _task_request_t type references have been removed
        REQUIRE(true);
    }
}

TEST_CASE("TaskScheduler configuration is correct for painlessMesh", "[v1.7.6][taskscheduler][config]") {
    // Verify that _TASK_STD_FUNCTION is defined
    // This is REQUIRED for painlessMesh lambda callbacks to work
    
    SECTION("_TASK_STD_FUNCTION must be defined") {
        #ifdef _TASK_STD_FUNCTION
        REQUIRE(true); // Good - we have std::function support
        #else
        FAIL("_TASK_STD_FUNCTION must be defined for painlessMesh lambdas");
        #endif
    }
    
    SECTION("_TASK_THREAD_SAFE should be disabled") {
        // _TASK_THREAD_SAFE is incompatible with _TASK_STD_FUNCTION in TaskScheduler v4.0.x
        // It should remain disabled until TaskScheduler v4.1+ fixes this
        #ifdef _TASK_THREAD_SAFE
        FAIL("_TASK_THREAD_SAFE should be disabled (incompatible with _TASK_STD_FUNCTION)");
        #else
        REQUIRE(true); // Good - thread-safe mode is disabled
        #endif
    }
    
    SECTION("_TASK_PRIORITY should be defined") {
        // painlessMesh requires priority support
        #ifdef _TASK_PRIORITY
        REQUIRE(true);
        #else
        FAIL("_TASK_PRIORITY must be defined for painlessMesh");
        #endif
    }
}

#ifdef ESP32
#include "painlessmesh/mesh.hpp"

TEST_CASE("mesh.hpp compiles without scheduler_queue on ESP32", "[v1.7.6][esp32][mesh][compilation]") {
    // Verify painlessMesh can be instantiated without thread-safe queue
    
    SECTION("Can create painlessMesh instance") {
        // If this compiles and runs, mesh.hpp doesn't require scheduler_queue
        painlessMesh mesh;
        REQUIRE(true);
    }
    
    SECTION("Mesh initialization doesn't require queue") {
        // Verify that init() method compiles without scheduler::initQueue()
        painlessMesh mesh;
        
        // Note: We can't actually call init() here because it requires WiFi,
        // but if this compiles, the queue references have been removed
        REQUIRE(true);
    }
}

TEST_CASE("FreeRTOS semaphore timeout fix is conceptually active", "[v1.7.6][esp32][freertos][fix]") {
    // This test documents that we still have the FreeRTOS crash fix
    // even though the thread-safe queue has been removed
    
    SECTION("Semaphore timeout value is documented") {
        // Expected: mesh.hpp line 555 uses timeout of 100 ticks (not 10)
        // This provides ~85% crash reduction on ESP32
        
        constexpr int EXPECTED_TIMEOUT_MS = 100;
        constexpr int OLD_TIMEOUT_MS = 10;
        constexpr float EXPECTED_CRASH_REDUCTION = 0.85f; // ~85%
        
        REQUIRE(EXPECTED_TIMEOUT_MS == 100);
        REQUIRE(OLD_TIMEOUT_MS == 10);
        REQUIRE(EXPECTED_CRASH_REDUCTION >= 0.80f);
        REQUIRE(EXPECTED_CRASH_REDUCTION <= 0.90f);
        
        // Note: The actual timeout value is in mesh.hpp takeSemaphore()
        // We can't easily test the runtime value without mocking FreeRTOS,
        // but this test documents the expected configuration
    }
    
    SECTION("Expected crash reduction is acceptable") {
        // v1.7.4 goal: 95-98% crash reduction (dual approach)
        // v1.7.6 actual: ~85% crash reduction (semaphore timeout only)
        // Trade-off: Acceptable for maintaining library functionality
        
        constexpr float TARGET_CRASH_REDUCTION = 0.85f;
        constexpr float ORIGINAL_GOAL = 0.95f;
        constexpr float ACCEPTABLE_THRESHOLD = 0.80f;
        
        REQUIRE(TARGET_CRASH_REDUCTION >= ACCEPTABLE_THRESHOLD);
        
        // Document that we're accepting a trade-off
        INFO("Original goal: " << (ORIGINAL_GOAL * 100) << "% crash reduction");
        INFO("Current: " << (TARGET_CRASH_REDUCTION * 100) << "% crash reduction");
        INFO("Trade-off: Maintain library functionality over maximum protection");
    }
}
#endif // ESP32

TEST_CASE("v1.7.6 fix resolves compilation issues", "[v1.7.6][regression][fix]") {
    // This test suite verifies that v1.7.6 resolves the issues from v1.7.4/v1.7.5
    
    SECTION("v1.7.4 issue: _task_request_t was not declared") {
        // This was the primary compilation error in v1.7.4 and v1.7.5
        // By removing scheduler_queue.cpp, this error no longer occurs
        
        // If this test compiles, the error is resolved
        REQUIRE(true);
    }
    
    SECTION("v1.7.5 issue: Dead code not removed after disabling macro") {
        // v1.7.5 disabled _TASK_THREAD_SAFE but left scheduler_queue files
        // This test verifies the files have been properly removed
        
        #ifdef PAINLESSMESH_SCHEDULER_QUEUE_HPP_INCLUDED
        FAIL("scheduler_queue.hpp should have been removed in v1.7.6");
        #endif
        
        REQUIRE(true);
    }
    
    SECTION("No breaking changes introduced") {
        // v1.7.6 should not break any existing functionality
        // The only changes are file removals and cleanup
        
        // Verify essential features are still available:
        #ifdef _TASK_STD_FUNCTION
        // Lambda support: Available
        #endif
        
        #ifdef _TASK_PRIORITY
        // Priority support: Available
        #endif
        
        REQUIRE(true);
    }
}

TEST_CASE("Documentation for v1.7.6 fix", "[v1.7.6][documentation]") {
    // This test documents the fix for future reference
    
    SECTION("What was broken") {
        INFO("v1.7.4 and v1.7.5 failed to compile with error:");
        INFO("  '_task_request_t' was not declared in this scope");
        INFO("Affected file: src/painlessmesh/scheduler_queue.cpp");
    }
    
    SECTION("Why it was broken") {
        INFO("Root cause (3 layers):");
        INFO("1. _task_request_t type only defined when _TASK_THREAD_SAFE enabled");
        INFO("2. _TASK_THREAD_SAFE disabled due to std::function conflict");
        INFO("3. scheduler_queue.cpp not removed/guarded after disabling macro");
    }
    
    SECTION("How v1.7.6 fixes it") {
        INFO("Solution: Remove dead code (Option A)");
        INFO("- Deleted: scheduler_queue.hpp");
        INFO("- Deleted: scheduler_queue.cpp");
        INFO("- Updated: mesh.hpp (removed queue includes and init)");
        INFO("- Maintained: Semaphore timeout fix (~85% crash reduction)");
    }
    
    SECTION("Future roadmap") {
        INFO("Thread-safe queue may return in v1.8.0 when:");
        INFO("- TaskScheduler v4.1+ released with fixed compatibility");
        INFO("- OR painlessMesh refactored to use raw function pointers");
        INFO("- OR TaskScheduler forked and fixed");
    }
    
    REQUIRE(true);
}
