/*
  QUAD7SHIFT.h - Header of the library for driving 4 digits seven segment displays for modules that use 74HC595 shiftregisters.
  For ARDUINO UNO and NANO and ATtiny85 (or other ATtinys that use the same USI (Universal Serial Interface) module and same pins as the ATtiny85)
  Created by Alex Rosito, July 24, 2023.
  Released into the public domain.
  © Alex Rosito
*/

#ifndef QUAD7SHIFT_h

#define QUAD7SHIFT_h
#include <avr/pgmspace.h>
#include "QUAD7SHIFTData.h"

#define setBitValue(pos, val, var) \
	((val ? (var) |= _BV(pos) : (var) &= ~_BV(pos)))
#define getBit(var, pos) \
	(var >> pos) & 0b1
#define setMode(pin, val) \
	setBitValue(pin, val, DDRB)
#define writePin(pin, val) \
	setBitValue(pin, val, PORTB)
#define readPin(pin) \
	((PINB & _BV(pin)) >> (pin))
#define togglePin(pin) \
	(PINB = bit(pin))

#if defined(__AVR_ATtiny25__) || defined(__AVR_ATtiny45__) || defined(__AVR_ATtiny85__) || defined(__AVR_ATtiny261__) || defined(__AVR_ATtiny461__) || defined(__AVR_ATtiny861__)
#define LATCHPIN DDB0 // ATITINY PIN PB0 / PIN 5
#define DATAPIN DDB1  // ATITINY PIN PB1 / PIN 6
#define CLOCKPIN DDB2 // ATITINY PIN PB2 / PIN 7
#endif
#if defined(__AVR_ATmega328P__) || (__AVR_ATmega168__)
// Arduino
#define LATCHPIN DDB2 // ARDUINO UNO/NANO PIN 10
#define DATAPIN DDB3  // ARDUINO UNO/NANO PIN 11
#define CLOCKPIN DDB5 // ARDUINO UNO/NANO PIN 13
#endif

// Optional user overrides for bit-banged pins (Arduino digital pin numbers).
// Define these before including the library in your sketch if you want different pins.
// Default for Arduino UNO/NANO
#if !defined(QUAD7SHIFT_BB_LATCH_PIN) && (defined(__AVR_ATmega328P__) || defined(__AVR_ATmega168__))
#define QUAD7SHIFT_BB_LATCH_PIN 10
#define QUAD7SHIFT_BB_DATA_PIN 11
#define QUAD7SHIFT_BB_CLOCK_PIN 13
#endif
// Default for ATtiny cores using digital pin mapping 0..4 = PB0..PB4
#if !defined(QUAD7SHIFT_BB_LATCH_PIN) && (defined(__AVR_ATtiny25__) || defined(__AVR_ATtiny45__) || defined(__AVR_ATtiny85__) || defined(__AVR_ATtiny261__) || defined(__AVR_ATtiny461__) || defined(__AVR_ATtiny861__))
#define QUAD7SHIFT_BB_LATCH_PIN 0
#define QUAD7SHIFT_BB_DATA_PIN 1
#define QUAD7SHIFT_BB_CLOCK_PIN 2
#endif
// Fallback (leave undefined for now if unknown)
#ifndef QUAD7SHIFT_BB_LATCH_PIN
#define QUAD7SHIFT_BB_LATCH_PIN 10
#endif
#ifndef QUAD7SHIFT_BB_DATA_PIN
#define QUAD7SHIFT_BB_DATA_PIN 11
#endif
#ifndef QUAD7SHIFT_BB_CLOCK_PIN
#define QUAD7SHIFT_BB_CLOCK_PIN 13
#endif

// Display types
#define COMMON_ANODE 0
#define COMMON_CATHODE 1

// Other self explanatory macros
#define DEFAULT_REFRESH_RATE 10000
#define DIGITS_PER_MODULE 4

// QUAD7SHIFT class definition, its properties and methods prototypes
class QUAD7SHIFT
{
private:
	// Class Attributes
	unsigned long _refreshRate;
	bool _type;

	// Class private methods
	void transferDigit(uint16_t);
	unsigned int fixedPointRepresentation(float, uint8_t);
	void printNumber(uint16_t, uint8_t);
	uint8_t getTableIndex(char);
	void printDigit(uint8_t, uint8_t, bool);

	// Bit-bang runtime config and helpers
	bool _useBitbang;
	bool _bbPinsSet;
	uint8_t _bbLatchPin;
	uint8_t _bbDataPin;
	uint8_t _bbClockPin;
	volatile uint8_t *_bbOutLatch;
	volatile uint8_t *_bbOutData;
	volatile uint8_t *_bbOutClock;
	uint8_t _bbMaskLatch;
	uint8_t _bbMaskData;
	uint8_t _bbMaskClock;

public:
	// Instance constructor
	QUAD7SHIFT(bool displayType = COMMON_ANODE);

	// Class public methods
	void begin(unsigned long freshRate = DEFAULT_REFRESH_RATE);
	void setBitbangPins(uint8_t latchPin, uint8_t dataPin, uint8_t clockPin);
	void setBitbang(bool enable);
	bool getBitbang();
	unsigned long getRefreshRate(void);
	void setRefreshRate(unsigned long);
	void print(const float, uint8_t);
	void print(const uint16_t);
	void print(const char *);
};
#endif
