# QUAD7SHIFT Library

This library provides functions for interfacing with 4-digit seven-segment displays in Arduino projects.

The QUAD7SHIFT library is a useful and well-organized library for interfacing with 4-digit seven-segment displays in Arduino projects. It provides functions to initialize the display, print numbers (including floating-point numbers with decimal point positioning), and print strings on the display. The library also supports both common anode and common cathode configurations, making it versatile for different types of seven-segment displays.

The library is written in a modular and object-oriented manner, with well-commented code that enhances readability and understanding. The comments provide clear explanations of the code's functionality, which is helpful for documentation purposes and for users to understand how the library works.

One notable feature is the use of PROGMEM to store data in program memory, which is a good optimization to conserve RAM. Additionally, the library provides support for both ATmega-based Arduino boards, ATtiny85, ATtiny25, ATtiny45, ATtiny261, ATtiny461 and ATtiny861, making it applicable to a broader range of Arduino platforms.


## Table of Contents

- [Installation](#installation)
- [Features](#features)
- [Usage](#usage)
- [Methods](#methods)
- [Examples](#examples)
- [Dependencies](#dependencies)
- [Contributing](#contributing)
- [License](#license)

## Installation

### Arduino IDE Library Manager (Recommended)

1. Open the Arduino IDE.
2. Go to "Sketch" -> "Include Library" -> "Manage Libraries...".
3. In the Library Manager, search for "QUAD7SHIFT".
4. Click the "Install" button for the "QUAD7SHIFT" library.

### Manual Installation

1. Download the library from [GitHub](https://github.com/AlexRosito67/QUAD7SHIFT).
2. Extract the downloaded ZIP file.
3. Rename the extracted folder to "QUAD7SHIFT".
4. Move the "QUAD7SHIFT" folder to your Arduino libraries folder. By default, it is located in "Documents/Arduino/libraries" on macOS and Linux, "My Documents\Arduino\libraries" on Windows.

## Features

- Support for both common anode and common cathode 4-digit seven-segment displays.
- Print numbers (including floating-point numbers with decimal point positioning) and strings on the display.
- Configurable refresh rate for display updating.
- Works with ATmega-based Arduino UNO and NANO boards and ATtiny85 (or other ATtinys that use the same USI (Universal Serial Interface) module and same pins as the ATtiny85).
   
## Examples

Check out the "examples" folder in this library for more usage examples and test sketches.

## Dependencies

- For ATmega-based Arduino boards: SPI library. 
    - LATCHPIN: ARDUINO UNO/NANO PIN 10
    - DATAPIN:  ARDUINO UNO/NANO PIN 11 
    - CLOCKPIN: ARDUINO UNO/NANO PIN 13
- For ATtiny85, ATtiny25, ATtiny45, ATtiny261, ATtiny461 and ATtiny861: Standard `SPI.h` (via the core's USI wrapper).
    - LATCHPIN: ATITINY PIN PB0 / PIN 5
    - DATAPIN:  ATITINY PIN PB1 / PIN 6
    - CLOCKPIN: ATITINY PIN PB2 / PIN 7

Notes:
- `SPI.transfer16()` is used when the core supports it; otherwise the library sends two `SPI.transfer()` calls (LSB then MSB) as a fallback.
- If you prefer or require a software (bit-banged) SPI implementation, define `BITBANG_SPI` in your sketch or use a build flag (PlatformIO example: `build_flags = -DBITBANG_SPI` to enable or `-UBITBANG_SPI` to disable).
- The library sets `LSBFIRST` bit order automatically in `begin()`.
 
Bit-banged SPI (optional):

If you enable software SPI (`BITBANG_SPI`), the library toggles pins directly for higher speed. See the Usage section for compile-time and runtime examples.

Notes:
- `QUAD7SHIFT_BB_*` macro values are Arduino digital pin numbers (e.g., 10, 11, 13 on UNO).
- `BITBANG_SPI` is useful on MCUs where hardware SPI is unavailable or undesirable. If you use arbitrary digital pins on other ports, the library computes the proper port and DDR registers using Arduino macros so bit-banging works on other ports, too.

## Contributing

Please read CONTRIBUTING.md for information about contributing.

## License

Please read LICENSE.md.

## Methods

    - begin();        / Sets the refresh rate of the display to a default of 1000 milliseconds.
    - begin(freshRate); / Sets the refresh rate to the desired refresh rate.
    - getRefreshRate(); / Gets the refresh rate in an interactive manner.
    - setRefreshRate(refreshRate); / Sets the desired refresh rate in an interactive manner. 
    - print(integer); / MAX 9999 
    - print(float);   / MAX 999.9 
    - print(string);  / MAX LENGTH 4 or in the case of including points MAX LENGTH 8
  - setBitbangPins(latchPin, dataPin, clockPin); / Configure bit-banged pins at runtime (digital pin numbers)
  - setBitbang(bool); / Enable or disable bit-banged SPI at runtime
  - getBitbang(); / Query if bit-bang is currently enabled
    
## Usage

### Hardware SPI (default)

```cpp
#include <QUAD7SHIFT.h>

// Create an instance of the QUAD7SHIFT class
QUAD7SHIFT display(COMMON_ANODE);

void setup() {
  // Initialize the QUAD7SHIFT library
  display.begin(); // Uses a default refresh rate of 1000
}

void loop() {
  display.print(1234);  // Print an integer number on the display
  display.print(12.34); // Print a float number on the display
  display.print("GOOD");      // Print a char string on the display    
  display.print("G.O.O.D.");  // Print a char string with dots on the display 
}
```

### Compile-time bit-bang (override macros)

```cpp
// Override pins and enable bit-bang at compile-time (UNO example)
#define BITBANG_SPI
#define QUAD7SHIFT_BB_LATCH_PIN 10
#define QUAD7SHIFT_BB_DATA_PIN  11
#define QUAD7SHIFT_BB_CLOCK_PIN 13
#include <QUAD7SHIFT.h>

QUAD7SHIFT display(COMMON_ANODE);

void setup() {
  display.begin();
}

void loop() {
  display.print(1234);
}
```

### Runtime bit-bang (any MCU)

```cpp
#include <QUAD7SHIFT.h>

QUAD7SHIFT display(COMMON_ANODE);

void setup() {
  display.begin();
  // Configure bit-bang pins and enable runtime bit-bang mode
  display.setBitbangPins(10, 11, 13); // UNO example
  display.setBitbang(true);
}

void loop() {
  display.print(1234);
}
```

### Runtime bit-bang (ATtiny85 — PB1, PB3, PB4 mapping)

```cpp
#include <QUAD7SHIFT.h>

QUAD7SHIFT display(COMMON_ANODE);

void setup() {
  display.begin();
  // Configure runtime bit-bang pins for ATtiny85 (PB1, PB3, PB4 = Arduino digital pins 1,3,4)
  display.setBitbangPins(1, 3, 4);
  display.setBitbang(true);
}

void loop() {
  display.print(1234);
}
```

   
