#include "OpenLogManager.h"

OpenLogManager::OpenLogManager(Stream &serial) : _serial(serial) {}

// Send (CTRL+Z)
void OpenLogManager::sendEscape()
{
    _serial.write((uint8_t)26);
}

// Wait for prompt '>'
bool OpenLogManager::waitForPrompt(unsigned long timeout)
{
    unsigned long start = millis();
    while (millis() - start < timeout)
    {
        if (_serial.available())
        {
            char c = _serial.read();
            if (c == '>')
                return true;
        }
    }
    return false;
}

// Read all characters until timeout
String OpenLogManager::readAll(unsigned long timeout)
{
    String out;
    unsigned long start = millis();
    while (millis() - start < timeout)
    {
        while (_serial.available())
        {
            out += (char)_serial.read();
            start = millis(); // Reset timeout if something is received
        }
    }
    return out;
}

// Enter command mode
bool OpenLogManager::enterCommandMode()
{
    for (int i = 0; i < 3; i++)
    {
        sendEscape();
        delay(50);
    }
    return waitForPrompt(2000);
}

// List all files
bool OpenLogManager::listFiles(String prefix)
{
    _serial.print("ls\r");
    String out = readAll(1000);
    if (out.length() == 0)
        return false;
    
    // NOTE: Direct printing to Serial is generally avoided in libraries
    // to allow usage on chips without USB Serial or for cleaner output.
    // You can uncomment the lines below for debugging:
    
    // Serial.println(F("[OpenLog] File list:"));
    // Serial.println(out);
    
    return true;
}

// Find last file of type prefixN.txt
int OpenLogManager::getLastFileNumber(String prefix)
{
    _serial.print("ls\r");
    String out = readAll(1500);
    int lastNum = -1;

    int idx = 0;
    while ((idx = out.indexOf(prefix, idx)) >= 0)
    {
        idx += prefix.length();
        int start = idx;
        // Read number
        while (idx < out.length() && isDigit(out[idx]))
            idx++;
        String numStr = out.substring(start, idx);
        if (numStr.length() > 0)
        {
            int n = numStr.toInt();
            if (n > lastNum)
                lastNum = n;
        }
    }

    return lastNum;
}

// Create file "prefix(num).txt"
bool OpenLogManager::createNewFile(String prefix, int newNum)
{
    String fname = prefix + String(newNum) + ".txt";
    String cmd = "new " + fname + "\r";
    _serial.print(cmd);
    return waitForPrompt(500);
}

// Write data to file using "append"
bool OpenLogManager::appendToFile(String filename)
{
    String cmd = "append " + filename + "\r";
    _serial.print(cmd);
    return waitForPrompt(500);
}

void OpenLogManager::writeOnFile(String msg)
{
    _serial.print(msg);
}