# AlashMotorControlLite

> 🌐 [English](README.md) | [Қазақша](README_kz.md) | **Русский**

[![Arduino](https://img.shields.io/badge/Arduino-Compatible-blue.svg)](https://www.arduino.cc/)
[![ESP32](https://img.shields.io/badge/ESP32-Supported-green.svg)](https://www.espressif.com/)
[![Version](https://img.shields.io/badge/version-1.1.0-orange.svg)](https://github.com/Alash-electronics/AlashMotorControlLite)
[![License](https://img.shields.io/badge/license-MIT-blue.svg)](LICENSE)

**Простая и мощная библиотека для управления DC моторами на Arduino и ESP32**

Библиотека AlashMotorControlLite предоставляет универсальный интерфейс для управления DC моторами через различные драйверы (L298N, TB6612, ZK-5AD, DRV8833, BTS7960 и другие). Поддерживает Arduino, ESP32 и совместимые платформы.

## 🚀 Основные возможности

- ✅ **Простой API** - управление скоростью от -100 до 100
- ✅ **4 режима драйверов** - поддержка всех популярных H-Bridge микросхем
- ✅ **Кроссплатформенность** - Arduino, ESP32, совместимые платы
- ✅ **Защита от ошибок** - автоматическое ограничение скорости
- ✅ **ESP32 3.x готова** - поддержка нового PWM API
- ✅ **17 примеров** - от базовых до продвинутых (меканум колеса)
- ✅ **Трёхъязычная документация** - English, Қазақша и Русский

## 📦 Установка

### Через Arduino Library Manager (рекомендуется)

1. Откройте Arduino IDE
2. **Скетч** → **Подключить библиотеку** → **Управление библиотеками**
3. Найдите `AlashMotorControlLite`
4. Нажмите **Установить**
5. Перезапустите Arduino IDE

### Ручная установка

1. Скачайте [последнюю версию](https://github.com/Alash-electronics/AlashMotorControlLite/releases)
2. Распакуйте в папку `Arduino/libraries/`
3. Перезапустите Arduino IDE

## 🎯 Быстрый старт

### Простейший пример (Arduino + L298N)

```cpp
#include "AlashMotorControlLite.h"

// DIR_DIR_PWM режим: IN1=4, IN2=5, EN=6
AlashMotorControlLite motor(DIR_DIR_PWM, 4, 5, 6);

void setup() {
  // Инициализация не требуется
}

void loop() {
  motor.setSpeed(75);    // Вперед на 75%
  delay(2000);

  motor.setSpeed(-50);   // Назад на 50%
  delay(2000);

  motor.stop();          // Остановка
  delay(1000);
}
```

### ESP32 + ZK-5AD (PWM_PWM режим)

```cpp
#include "AlashMotorControlLite.h"

// PWM_PWM режим: IN1=GPIO32, IN2=GPIO33
AlashMotorControlLite motor(PWM_PWM, 32, 33);

void setup() {
  Serial.begin(115200);
}

void loop() {
  motor.setSpeed(100);   // Полная скорость вперед
  delay(2000);

  motor.brake();         // Активное торможение
  delay(500);

  motor.setSpeed(-100);  // Полная скорость назад
  delay(2000);

  motor.stop();          // Свободная остановка
  delay(1000);
}
```

## 🎛️ Режимы драйверов (MODE)

Библиотека поддерживает 4 режима для различных типов драйверов:

| Режим | Пины | PWM | Драйверы | Описание |
|-------|------|-----|----------|----------|
| **DIR_PWM** | 2 | 1 | TB6612FNG, DRV8833 | 1 направление + 1 скорость |
| **PWM_PWM** | 2 | 2 | ZK-5AD, L298N, BTS7960 | 2 PWM для двунаправленного управления |
| **DIR_DIR_PWM** | 3 | 1 | L298N (с EN), L293D | 2 направления + 1 скорость |
| **DIR_DIR** | 2 | 0 | Реле, простые драйверы | Только вкл/выкл без ШИМ |

### Примеры подключения

#### L298N драйвер (DIR_DIR_PWM)
```cpp
AlashMotorControlLite motor(DIR_DIR_PWM, 4, 5, 6);
// PIN 4 → IN1 (направление 1)
// PIN 5 → IN2 (направление 2)
// PIN 6 → EN  (скорость PWM)
```

#### TB6612FNG драйвер (DIR_PWM)
```cpp
AlashMotorControlLite motor(DIR_PWM, 4, 3);
// PIN 4 → IN1 (направление)
// PIN 3 → PWM (скорость)
```

#### ZK-5AD драйвер для ESP32 (PWM_PWM)
```cpp
AlashMotorControlLite motor(PWM_PWM, 32, 33);
// GPIO 32 → IN1 (PWM вперед)
// GPIO 33 → IN2 (PWM назад)
```

## 📚 API Документация

### Конструкторы

```cpp
// Для режимов: DIR_PWM, PWM_PWM, DIR_DIR (2 пина)
AlashMotorControlLite(MODE mode, uint8_t pin1, uint8_t pin2);

// Для режима: DIR_DIR_PWM (3 пина)
AlashMotorControlLite(MODE mode, uint8_t pin1, uint8_t pin2, uint8_t pin_pwm);
```

### Основные методы

#### `void setSpeed(int16_t speed)`
Установка скорости и направления мотора

**Параметры:**
- `speed` - скорость от **-100 до 100**
  - `-100` до `-1` = движение назад
  - `0` = остановка
  - `1` до `100` = движение вперед

**Пример:**
```cpp
motor.setSpeed(75);    // Вперед на 75%
motor.setSpeed(-50);   // Назад на 50%
motor.setSpeed(0);     // Остановка
```

#### `void stop()`
Свободная остановка мотора (coast stop)

Мотор отключается и останавливается по инерции. Используется для:
- Плавной остановки без нагрузки на механику
- Экономии энергии
- Обычной остановки в конце движения

**Пример:**
```cpp
motor.setSpeed(100);
delay(2000);
motor.stop();  // Плавная остановка
```

#### `void brake()`
Активное торможение мотора

Мотор активно тормозит (короткое замыкание обмоток). Используется для:
- Быстрой остановки
- Удержания позиции под нагрузкой
- Точной остановки

**Пример:**
```cpp
motor.setSpeed(100);
delay(2000);
motor.brake();  // Резкое торможение
```

#### `int16_t getSpeed()`
Получение текущей скорости мотора

**Возвращает:** Текущую скорость от -100 до 100

**Пример:**
```cpp
motor.setSpeed(75);
int speed = motor.getSpeed();  // Вернет 75
Serial.println(speed);
```

#### `MODE getMode()`
Получение текущего режима драйвера

**Возвращает:** Значение MODE enum

**Пример:**
```cpp
MODE mode = motor.getMode();
if (mode == PWM_PWM) {
  Serial.println("Режим PWM_PWM");
}
```

## 🔌 Схемы подключения

### Arduino Uno + L298N

```
Arduino Uno          L298N Driver
─────────────────────────────────
PIN 4      →  IN1
PIN 5      →  IN2
PIN 6      →  ENA (PWM)
5V         →  5V
GND        →  GND

            Motor Output
            ─────────────
            OUT1 → Motor +
            OUT2 → Motor -
            12V  → Power Supply +
            GND  → Power Supply -
```

### ESP32 + ZK-5AD (2 мотора)

```
ESP32              ZK-5AD Driver
──────────────────────────────────
GPIO 32   →  Motor1 IN1
GPIO 33   →  Motor1 IN2
GPIO 25   →  Motor2 IN1
GPIO 26   →  Motor2 IN2
5V        →  VCC
GND       →  GND

            Motor Outputs
            ──────────────
            M1+/M1- → Motor 1
            M2+/M2- → Motor 2
            VM      → 6-12V Power
            GND     → Power GND
```

## 💡 Примеры использования

### Плавный разгон и торможение

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motor(DIR_PWM, 4, 3);

void smoothSpeed(int target, int delayMs) {
  int current = motor.getSpeed();
  int step = (target > current) ? 1 : -1;

  for (int speed = current; speed != target; speed += step) {
    motor.setSpeed(speed);
    delay(delayMs);
  }
}

void setup() {}

void loop() {
  smoothSpeed(100, 20);   // Плавный разгон за 2 секунды
  delay(2000);
  smoothSpeed(0, 20);     // Плавное торможение
  delay(1000);
}
```

### Управление двумя моторами (робот)

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motorLeft(PWM_PWM, 32, 33);
AlashMotorControlLite motorRight(PWM_PWM, 25, 26);

void moveForward(int speed) {
  motorLeft.setSpeed(speed);
  motorRight.setSpeed(speed);
}

void turnRight(int speed) {
  motorLeft.setSpeed(speed);
  motorRight.setSpeed(-speed);
}

void stopAll() {
  motorLeft.stop();
  motorRight.stop();
}

void setup() {}

void loop() {
  moveForward(70);    // Вперед
  delay(2000);

  turnRight(50);      // Поворот направо
  delay(1000);

  stopAll();          // Остановка
  delay(1000);
}
```

### Меканум колеса (4WD)

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motorFL(PWM_PWM, 32, 33);  // Front Left
AlashMotorControlLite motorFR(PWM_PWM, 25, 26);  // Front Right
AlashMotorControlLite motorRL(PWM_PWM, 19, 18);  // Rear Left
AlashMotorControlLite motorRR(PWM_PWM, 17, 16);  // Rear Right

void mecanumDrive(int vx, int vy, int rotation) {
  int speedFL = vx + vy + rotation;
  int speedFR = vx - vy - rotation;
  int speedRL = vx - vy + rotation;
  int speedRR = vx + vy - rotation;

  motorFL.setSpeed(constrain(speedFL, -100, 100));
  motorFR.setSpeed(constrain(speedFR, -100, 100));
  motorRL.setSpeed(constrain(speedRL, -100, 100));
  motorRR.setSpeed(constrain(speedRR, -100, 100));
}

void setup() {}

void loop() {
  mecanumDrive(50, 0, 0);     // Вперед
  delay(2000);

  mecanumDrive(0, 50, 0);     // Страфинг влево
  delay(2000);

  mecanumDrive(0, 0, 30);     // Поворот на месте
  delay(2000);
}
```

## 📖 Встроенные примеры

После установки библиотеки примеры доступны в Arduino IDE:
**Файл → Примеры → AlashMotorControlLite**

> 💡 **Примечание:** Все примеры доступны на трёх языках (EN/KZ/RU). В Arduino IDE выберите папку с нужным языком: **EN** (English), **KZ** (Қазақша), или **RU** (Русский).

### Базовые примеры (изучение API)

| Пример | Описание |
|--------|----------|
| **StopVsBrake** | Разница между stop() и brake() |
| **GettersDemo** | Использование getSpeed() и getMode() |
| **SmoothSpeed** | Плавное изменение скорости (ramping) |
| **AllModesDemo** | Сравнение всех 4 режимов драйверов |
| **SpeedLimits** | Автоматическое ограничение скорости |
| **DirectionChange** | Безопасная смена направления |
| **MultiMotorSync** | Синхронизация нескольких моторов |

### Примеры по драйверам

| Пример | Режим | Драйвер |
|--------|-------|---------|
| **DIR_PWM** | DIR_PWM | TB6612FNG, DRV8833 |
| **DIR_PWM_DualMotor** | DIR_PWM | 2 мотора TB6612FNG |
| **PWM_PWM_DualMotor** | PWM_PWM | 2 мотора общий |
| **DIR_DIR_PWM** | DIR_DIR_PWM | L298N |
| **DIR_DIR_PWM_DualMotor** | DIR_DIR_PWM | 2 мотора L298N |
| **DIR_DIR_PWM_pot** | DIR_DIR_PWM | Управление потенциометром |

### ESP32 специализированные

| Пример | Описание |
|--------|----------|
| **ESP32_ZK5AD_SingleMotor** | 1 мотор на ESP32 + ZK-5AD |
| **ESP32_ZK5AD_DualMotor** | 2 мотора на ESP32 + ZK-5AD |
| **ESP32_Mecanum_4WD_Demo** | Меканум робот (автодемо) |
| **ESP32_Mecanum_Serial** | Меканум робот (управление) |

## ❓ FAQ (Часто задаваемые вопросы)

### Какой режим выбрать для моего драйвера?

- **L298N** → `DIR_DIR_PWM` (если используете EN пин) или `PWM_PWM`
- **TB6612FNG** → `DIR_PWM`
- **ZK-5AD** → `PWM_PWM`
- **DRV8833** → `DIR_PWM`
- **BTS7960** → `PWM_PWM`
- **Реле** → `DIR_DIR`

### Почему мотор не вращается?

1. Проверьте питание драйвера (должно быть достаточным для мотора)
2. Убедитесь, что выбран правильный MODE
3. Проверьте подключение пинов
4. На ESP32: убедитесь, что используете не только INPUT пины
5. Попробуйте setSpeed(100) для максимальной скорости

### Мотор вращается в неправильную сторону

Поменяйте местами провода мотора (M+ и M-) или поменяйте pin1 и pin2 в конструкторе:

```cpp
// Было
AlashMotorControlLite motor(PWM_PWM, 32, 33);

// Стало (инвертировано)
AlashMotorControlLite motor(PWM_PWM, 33, 32);
```

### Как управлять более чем 2 моторами?

Создайте несколько объектов:

```cpp
AlashMotorControlLite motor1(PWM_PWM, 32, 33);
AlashMotorControlLite motor2(PWM_PWM, 25, 26);
AlashMotorControlLite motor3(PWM_PWM, 19, 18);
AlashMotorControlLite motor4(PWM_PWM, 17, 16);
```

ESP32 поддерживает до 16 PWM каналов.

### В чем разница между stop() и brake()?

- **stop()** - свободная остановка (выбег по инерции), меньше нагрузка
- **brake()** - активное торможение (быстрая остановка), больше нагрузка

Используйте `stop()` для обычной остановки, `brake()` когда нужна быстрая остановка.

### Поддерживается ли ESP32-S3 / ESP32-C3?

Да, библиотека работает на всех вариантах ESP32 (ESP32, ESP32-S2, ESP32-S3, ESP32-C3, ESP32-C6).

### Можно ли использовать с Servo библиотекой?

Да, но на Arduino будьте осторожны с PWM пинами - Servo и ШИМ могут конфликтовать на одних таймерах. На ESP32 проблем нет.

## 🔧 Расширенные возможности

### Плавный реверс (безопасная смена направления)

```cpp
void safeReverse(int newSpeed) {
  // Текущая скорость
  int current = motor.getSpeed();

  // Если разные знаки - нужен реверс
  if ((current > 0 && newSpeed < 0) || (current < 0 && newSpeed > 0)) {
    // Плавно до нуля
    while (abs(motor.getSpeed()) > 0) {
      int speed = motor.getSpeed();
      motor.setSpeed(speed > 0 ? speed - 1 : speed + 1);
      delay(10);
    }

    // Активное торможение
    motor.brake();
    delay(300);

    // Пауза
    motor.stop();
    delay(200);
  }

  // Плавно до новой скорости
  while (motor.getSpeed() != newSpeed) {
    int speed = motor.getSpeed();
    int step = (newSpeed > speed) ? 1 : -1;
    motor.setSpeed(speed + step);
    delay(10);
  }
}
```

### Автоматическая калибровка (поиск нуля)

```cpp
void findMinSpeed() {
  Serial.println("Поиск минимальной скорости...");

  for (int speed = 0; speed <= 100; speed++) {
    motor.setSpeed(speed);
    delay(100);

    // Здесь можно добавить проверку энкодера
    // если мотор начал вращаться - это минимум
    Serial.print("Тест: ");
    Serial.println(speed);
  }

  motor.stop();
}
```

## 🛠️ Совместимость

### Поддерживаемые платформы

- ✅ Arduino Uno, Nano, Mega
- ✅ Arduino Leonardo, Pro Mini
- ✅ ESP32 (все варианты)
- ✅ ESP8266
- ✅ STM32 (с Arduino Core)
- ✅ Teensy
- ✅ Другие Arduino-совместимые платы

### Поддерживаемые драйверы

- ✅ L298N, L293D
- ✅ TB6612FNG
- ✅ DRV8833, DRV8835
- ✅ ZK-5AD, ZK-5AD-V2
- ✅ BTS7960
- ✅ MX1508, MX1919
- ✅ VNH2SP30, VNH5019
- ✅ Любые H-Bridge драйверы

## 📝 Changelog

### Version 1.1.0
- ✅ Трёхъязычная поддержка (English/Қазақша/Русский)
- ✅ Все примеры доступны на трёх языках
- ✅ Исходный код на английском
- ✅ Улучшена документация

### Version 1.0.9
- ✅ Двуязычная поддержка (English/Русский)
- ✅ Обслуживание библиотеки

### Version 1.0.7
- ✅ Поддержка ESP32 Arduino Core 3.x (новый PWM API)
- ✅ Добавлены методы getSpeed() и getMode()
- ✅ Добавлен метод stop() (отдельно от brake)
- ✅ Защита от копирования класса
- ✅ Исправлен порядок параметров конструктора
- ✅ 17 примеров (включая меканум колеса)
- ✅ Улучшенная документация

### Version 1.0.6
- Базовая функциональность
- Поддержка 4 режимов
- ESP32 2.x поддержка

## 🤝 Поддержка и контакты

- 🌐 **Сайт:** [alash-electronics.kz](https://alash-electronics.kz/)
- 💬 **GitHub:** [Issues](https://github.com/Alash-electronics/AlashMotorControlLite/issues)
- 📧 **Email:** support@alash-electronics.kz
- 📖 **Документация:** [Wiki](https://github.com/Alash-electronics/AlashMotorControlLite/wiki)
- 🎓 **Обучение:** [Проекты и туториалы](https://alash-electronics.kz/blogs/wiki)

## 📄 Лицензия

MIT License - см. файл [LICENSE](LICENSE)

## 🌟 Благодарности

Спасибо всем, кто использует и улучшает эту библиотеку!

---

**Сделано с ❤️ в Казахстане | Alash Engineering**

## Ключевые слова для поиска

Arduino motor control, ESP32 motor driver, DC motor library, L298N Arduino, TB6612FNG library, PWM motor control, H-bridge Arduino, motor driver library, Arduino robot, ESP32 robot control, mecanum wheels, dual motor control, Arduino motor shield, ZK-5AD driver, DRV8833 Arduino, BTS7960 library, motor control library, Arduino DC motor, ESP32 PWM, motor speed control
