# AlashMotorControlLite

> 🌐 [English](README.md) | **Қазақша** | [Русский](README_ru.md)

[![Arduino](https://img.shields.io/badge/Arduino-Compatible-blue.svg)](https://www.arduino.cc/)
[![ESP32](https://img.shields.io/badge/ESP32-Supported-green.svg)](https://www.espressif.com/)
[![Version](https://img.shields.io/badge/version-1.1.0-orange.svg)](https://github.com/Alash-electronics/AlashMotorControlLite)
[![License](https://img.shields.io/badge/license-MIT-blue.svg)](LICENSE)

**Arduino және ESP32 платформаларында DC қозғалтқыштарды басқаруға арналған қарапайым және күшті кітапхана**

AlashMotorControlLite кітапханасы әртүрлі драйверлер (L298N, TB6612, ZK-5AD, DRV8833, BTS7960 және т.б.) арқылы DC қозғалтқыштарды басқаруға арналған әмбебап интерфейс ұсынады. Arduino, ESP32 және үйлесімді платформаларды қолдайды.

## 🚀 Негізгі мүмкіндіктер

- ✅ **Қарапайым API** - жылдамдықты -100-ден 100-ге дейін басқару
- ✅ **4 драйвер режимі** - барлық танымал H-Bridge микросхемаларын қолдау
- ✅ **Кроссплатформалылық** - Arduino, ESP32, үйлесімді тақталар
- ✅ **Қателерден қорғау** - жылдамдықты автоматты түрде шектеу
- ✅ **ESP32 3.x дайын** - жаңа PWM API қолдауы
- ✅ **17 мысал** - базалық деңгейден кәсіби деңгейге дейін (меканум дөңгелектер)
- ✅ **Үш тілді құжаттама** - English, Русский және Қазақша

## 📦 Орнату

### Arduino Library Manager арқылы (ұсынылады)

1. Arduino IDE ашыңыз
2. **Скетч** → **Библиотеканы қосу** → **Библиотекаларды басқару**
3. `AlashMotorControlLite` іздеңіз
4. **Орнату** батырмасын басыңыз
5. Arduino IDE қайта іске қосыңыз

### Қолмен орнату

1. [Соңғы нұсқаны](https://github.com/Alash-electronics/AlashMotorControlLite/releases) жүктеп алыңыз
2. `Arduino/libraries/` қалтасына шығарыңыз
3. Arduino IDE қайта іске қосыңыз

## 🎯 Жылдам бастау

### Қарапайым мысал (Arduino + L298N)

```cpp
#include "AlashMotorControlLite.h"

// DIR_DIR_PWM режимі: IN1=4, IN2=5, EN=6
AlashMotorControlLite motor(DIR_DIR_PWM, 4, 5, 6);

void setup() {
  // Инициализация қажет емес
}

void loop() {
  motor.setSpeed(75);    // Алға 75%
  delay(2000);

  motor.setSpeed(-50);   // Артқа 50%
  delay(2000);

  motor.stop();          // Тоқтату
  delay(1000);
}
```

### ESP32 + ZK-5AD (PWM_PWM режимі)

```cpp
#include "AlashMotorControlLite.h"

// PWM_PWM режимі: IN1=GPIO32, IN2=GPIO33
AlashMotorControlLite motor(PWM_PWM, 32, 33);

void setup() {
  Serial.begin(115200);
}

void loop() {
  motor.setSpeed(100);   // Толық жылдамдық алға
  delay(2000);

  motor.brake();         // Белсенді тежеу
  delay(500);

  motor.setSpeed(-100);  // Толық жылдамдық артқа
  delay(2000);

  motor.stop();          // Еркін тоқтату
  delay(1000);
}
```

## 🎛️ Драйвер режимдері (MODE)

Кітапхана әртүрлі драйвер түрлері үшін 4 режимді қолдайды:

| Режим | Пиндер | PWM | Драйверлер | Сипаттама |
|-------|--------|-----|------------|-----------|
| **DIR_PWM** | 2 | 1 | TB6612FNG, DRV8833 | 1 бағыт + 1 жылдамдық |
| **PWM_PWM** | 2 | 2 | ZK-5AD, L298N, BTS7960 | Екі бағытты басқаруға арналған 2 PWM |
| **DIR_DIR_PWM** | 3 | 1 | L298N (EN бар), L293D | 2 бағыт + 1 жылдамдық |
| **DIR_DIR** | 2 | 0 | Реле, қарапайым драйверлер | ШИМ-сіз тек қосу/өшіру |

### Қосылым мысалдары

#### L298N драйвері (DIR_DIR_PWM)
```cpp
AlashMotorControlLite motor(DIR_DIR_PWM, 4, 5, 6);
// PIN 4 → IN1 (бағыт 1)
// PIN 5 → IN2 (бағыт 2)
// PIN 6 → EN  (жылдамдық PWM)
```

#### TB6612FNG драйвері (DIR_PWM)
```cpp
AlashMotorControlLite motor(DIR_PWM, 4, 3);
// PIN 4 → IN1 (бағыт)
// PIN 3 → PWM (жылдамдық)
```

#### ESP32 үшін ZK-5AD драйвері (PWM_PWM)
```cpp
AlashMotorControlLite motor(PWM_PWM, 32, 33);
// GPIO 32 → IN1 (PWM алға)
// GPIO 33 → IN2 (PWM артқа)
```

## 📚 API құжаттамасы

### Конструкторлар

```cpp
// Режимдер үшін: DIR_PWM, PWM_PWM, DIR_DIR (2 пин)
AlashMotorControlLite(MODE mode, uint8_t pin1, uint8_t pin2);

// Режим үшін: DIR_DIR_PWM (3 пин)
AlashMotorControlLite(MODE mode, uint8_t pin1, uint8_t pin2, uint8_t pin_pwm);
```

### Негізгі әдістер

#### `void setSpeed(int16_t speed)`
Қозғалтқыш жылдамдығы мен бағытын орнату

**Параметрлер:**
- `speed` - жылдамдық **-100-ден 100-ге дейін**
  - `-100`-ден `-1`-ге дейін = артқа қозғалыс
  - `0` = тоқтату
  - `1`-ден `100`-ге дейін = алға қозғалыс

**Мысал:**
```cpp
motor.setSpeed(75);    // Алға 75%
motor.setSpeed(-50);   // Артқа 50%
motor.setSpeed(0);     // Тоқтату
```

#### `void stop()`
Қозғалтқыштың еркін тоқтауы (coast stop)

Қозғалтқыш ажыратылады және инерция бойынша тоқтайды. Мыналар үшін қолданылады:
- Механикаға жүктемесіз тегіс тоқтау
- Энергияны үнемдеу
- Қозғалыс соңындағы қалыпты тоқтау

**Мысал:**
```cpp
motor.setSpeed(100);
delay(2000);
motor.stop();  // Тегіс тоқтау
```

#### `void brake()`
Қозғалтқыштың белсенді тежелуі

Қозғалтқыш белсенді тежеледі (орамаларды қысқа тұйықтау). Мыналар үшін қолданылады:
- Жылдам тоқтау
- Жүктеме астында позицияны ұстау
- Нақты тоқтау

**Мысал:**
```cpp
motor.setSpeed(100);
delay(2000);
motor.brake();  // Күрт тежеу
```

#### `int16_t getSpeed()`
Қозғалтқыштың ағымдағы жылдамдығын алу

**Қайтарады:** -100-ден 100-ге дейінгі ағымдағы жылдамдық

**Мысал:**
```cpp
motor.setSpeed(75);
int speed = motor.getSpeed();  // 75 қайтарады
Serial.println(speed);
```

#### `MODE getMode()`
Драйвердің ағымдағы режимін алу

**Қайтарады:** MODE enum мәні

**Мысал:**
```cpp
MODE mode = motor.getMode();
if (mode == PWM_PWM) {
  Serial.println("PWM_PWM режимі");
}
```

## 🔌 Қосылым сызбалары

### Arduino Uno + L298N

```
Arduino Uno          L298N драйвері
─────────────────────────────────
PIN 4      →  IN1
PIN 5      →  IN2
PIN 6      →  ENA (PWM)
5V         →  5V
GND        →  GND

            Қозғалтқыш шығысы
            ─────────────────
            OUT1 → Motor +
            OUT2 → Motor -
            12V  → Қуат көзі +
            GND  → Қуат көзі -
```

### ESP32 + ZK-5AD (2 қозғалтқыш)

```
ESP32              ZK-5AD драйвері
──────────────────────────────────
GPIO 32   →  Motor1 IN1
GPIO 33   →  Motor1 IN2
GPIO 25   →  Motor2 IN1
GPIO 26   →  Motor2 IN2
5V        →  VCC
GND       →  GND

            Қозғалтқыш шығыстары
            ────────────────────
            M1+/M1- → Қозғалтқыш 1
            M2+/M2- → Қозғалтқыш 2
            VM      → 6-12V қуат
            GND     → Қуат GND
```

## 💡 Пайдалану мысалдары

### Тегіс үдету және тежеу

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motor(DIR_PWM, 4, 3);

void smoothSpeed(int target, int delayMs) {
  int current = motor.getSpeed();
  int step = (target > current) ? 1 : -1;

  for (int speed = current; speed != target; speed += step) {
    motor.setSpeed(speed);
    delay(delayMs);
  }
}

void setup() {}

void loop() {
  smoothSpeed(100, 20);   // 2 секундта тегіс үдету
  delay(2000);
  smoothSpeed(0, 20);     // Тегіс тежеу
  delay(1000);
}
```

### Екі қозғалтқышты басқару (робот)

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motorLeft(PWM_PWM, 32, 33);
AlashMotorControlLite motorRight(PWM_PWM, 25, 26);

void moveForward(int speed) {
  motorLeft.setSpeed(speed);
  motorRight.setSpeed(speed);
}

void turnRight(int speed) {
  motorLeft.setSpeed(speed);
  motorRight.setSpeed(-speed);
}

void stopAll() {
  motorLeft.stop();
  motorRight.stop();
}

void setup() {}

void loop() {
  moveForward(70);    // Алға
  delay(2000);

  turnRight(50);      // Оңға бұрылу
  delay(1000);

  stopAll();          // Тоқтату
  delay(1000);
}
```

### Меканум дөңгелектер (4WD)

```cpp
#include "AlashMotorControlLite.h"

AlashMotorControlLite motorFL(PWM_PWM, 32, 33);  // Алдыңғы сол
AlashMotorControlLite motorFR(PWM_PWM, 25, 26);  // Алдыңғы оң
AlashMotorControlLite motorRL(PWM_PWM, 19, 18);  // Артқы сол
AlashMotorControlLite motorRR(PWM_PWM, 17, 16);  // Артқы оң

void mecanumDrive(int vx, int vy, int rotation) {
  int speedFL = vx + vy + rotation;
  int speedFR = vx - vy - rotation;
  int speedRL = vx - vy + rotation;
  int speedRR = vx + vy - rotation;

  motorFL.setSpeed(constrain(speedFL, -100, 100));
  motorFR.setSpeed(constrain(speedFR, -100, 100));
  motorRL.setSpeed(constrain(speedRL, -100, 100));
  motorRR.setSpeed(constrain(speedRR, -100, 100));
}

void setup() {}

void loop() {
  mecanumDrive(50, 0, 0);     // Алға
  delay(2000);

  mecanumDrive(0, 50, 0);     // Солға страфинг
  delay(2000);

  mecanumDrive(0, 0, 30);     // Орнында айналу
  delay(2000);
}
```

## 📖 Кіріктірілген мысалдар

Кітапхананы орнатқаннан кейін мысалдар Arduino IDE-де қолжетімді:
**Файл → Мысалдар → AlashMotorControlLite**

> 💡 **Ескерту:** Барлық мысалдар ағылшын, орыс және қазақ тілдерінде қолжетімді. Орыс нұсқасын іздеу үшін `_ru` жұрнақты файлдарды, қазақ нұсқасын іздеу үшін `_kz` жұрнақты файлдарды қараңыз.

### Базалық мысалдар (API үйрену)

| Мысал | Сипаттама |
|-------|-----------|
| **StopVsBrake** | stop() және brake() арасындағы айырмашылық |
| **GettersDemo** | getSpeed() және getMode() пайдалану |
| **SmoothSpeed** | Жылдамдықты тегіс өзгерту (ramping) |
| **AllModesDemo** | Барлық 4 драйвер режимін салыстыру |
| **SpeedLimits** | Жылдамдықты автоматты түрде шектеу |
| **DirectionChange** | Бағытты қауіпсіз өзгерту |
| **MultiMotorSync** | Бірнеше қозғалтқышты синхрондау |

### Драйвер бойынша мысалдар

| Мысал | Режим | Драйвер |
|-------|-------|---------|
| **DIR_PWM** | DIR_PWM | TB6612FNG, DRV8833 |
| **DIR_PWM_DualMotor** | DIR_PWM | TB6612FNG 2 қозғалтқыш |
| **PWM_PWM_DualMotor** | PWM_PWM | Жалпы 2 қозғалтқыш |
| **DIR_DIR_PWM** | DIR_DIR_PWM | L298N |
| **DIR_DIR_PWM_DualMotor** | DIR_DIR_PWM | L298N 2 қозғалтқыш |
| **DIR_DIR_PWM_pot** | DIR_DIR_PWM | Потенциометрмен басқару |

### ESP32 мамандандырылған

| Мысал | Сипаттама |
|-------|-----------|
| **ESP32_ZK5AD_SingleMotor** | ESP32 + ZK-5AD 1 қозғалтқыш |
| **ESP32_ZK5AD_DualMotor** | ESP32 + ZK-5AD 2 қозғалтқыш |
| **ESP32_Mecanum_4WD_Demo** | Меканум робот (автодемо) |
| **ESP32_Mecanum_Serial** | Меканум робот (басқару) |

## ❓ Жиі қойылатын сұрақтар

### Менің драйверім үшін қандай режимді таңдау керек?

- **L298N** → `DIR_DIR_PWM` (EN пинін пайдаланған кезде) немесе `PWM_PWM`
- **TB6612FNG** → `DIR_PWM`
- **ZK-5AD** → `PWM_PWM`
- **DRV8833** → `DIR_PWM`
- **BTS7960** → `PWM_PWM`
- **Реле** → `DIR_DIR`

### Неліктен қозғалтқыш айналмайды?

1. Драйвердің қуат көзін тексеріңіз (қозғалтқышқа жеткілікті болуы керек)
2. Дұрыс MODE таңдалғанына көз жеткізіңіз
3. Пиндердің қосылымын тексеріңіз
4. ESP32-де: тек INPUT пиндерін қолданбағаныңызға көз жеткізіңіз
5. Максималды жылдамдық үшін setSpeed(100) қолданып көріңіз

### Қозғалтқыш дұрыс емес бағытта айналады

Қозғалтқыш сымдарын (M+ және M-) ауыстырыңыз немесе конструктордағы pin1 және pin2 ауыстырыңыз:

```cpp
// Болды
AlashMotorControlLite motor(PWM_PWM, 32, 33);

// Болды (инверттелген)
AlashMotorControlLite motor(PWM_PWM, 33, 32);
```

### 2-ден артық қозғалтқышты қалай басқаруға болады?

Бірнеше объект жасаңыз:

```cpp
AlashMotorControlLite motor1(PWM_PWM, 32, 33);
AlashMotorControlLite motor2(PWM_PWM, 25, 26);
AlashMotorControlLite motor3(PWM_PWM, 19, 18);
AlashMotorControlLite motor4(PWM_PWM, 17, 16);
```

ESP32 16 PWM арнаға дейін қолдайды.

### stop() және brake() арасындағы айырмашылық неде?

- **stop()** - еркін тоқтату (инерция бойынша шығу), жүктеме аз
- **brake()** - белсенді тежеу (жылдам тоқтау), жүктеме көп

Қалыпты тоқтату үшін `stop()`, жылдам тоқтау қажет болғанда `brake()` қолданыңыз.

### ESP32-S3 / ESP32-C3 қолдау көрсетілеме?

Иә, кітапхана барлық ESP32 нұсқаларында жұмыс істейді (ESP32, ESP32-S2, ESP32-S3, ESP32-C3, ESP32-C6).

### Servo кітапханасымен бірге қолдануға бола ма?

Иә, бірақ Arduino-да PWM пиндерімен сақ болыңыз - Servo және ШИМ бір таймерлерде қақтығыс тудыруы мүмкін. ESP32-де мәселе жоқ.

## 🔧 Кеңейтілген мүмкіндіктер

### Тегіс реверс (бағытты қауіпсіз өзгерту)

```cpp
void safeReverse(int newSpeed) {
  // Ағымдағы жылдамдық
  int current = motor.getSpeed();

  // Егер әртүрлі таңбалар болса - реверс қажет
  if ((current > 0 && newSpeed < 0) || (current < 0 && newSpeed > 0)) {
    // Нөлге дейін тегіс
    while (abs(motor.getSpeed()) > 0) {
      int speed = motor.getSpeed();
      motor.setSpeed(speed > 0 ? speed - 1 : speed + 1);
      delay(10);
    }

    // Белсенді тежеу
    motor.brake();
    delay(300);

    // Үзіліс
    motor.stop();
    delay(200);
  }

  // Жаңа жылдамдыққа дейін тегіс
  while (motor.getSpeed() != newSpeed) {
    int speed = motor.getSpeed();
    int step = (newSpeed > speed) ? 1 : -1;
    motor.setSpeed(speed + step);
    delay(10);
  }
}
```

### Автоматты калибрлеу (нөлді іздеу)

```cpp
void findMinSpeed() {
  Serial.println("Минималды жылдамдықты іздеу...");

  for (int speed = 0; speed <= 100; speed++) {
    motor.setSpeed(speed);
    delay(100);

    // Мұнда энкодер тексеруін қосуға болады
    // егер қозғалтқыш айнала бастаса - бұл минимум
    Serial.print("Тест: ");
    Serial.println(speed);
  }

  motor.stop();
}
```

## 🛠️ Үйлесімділік

### Қолдау көрсетілетін платформалар

- ✅ Arduino Uno, Nano, Mega
- ✅ Arduino Leonardo, Pro Mini
- ✅ ESP32 (барлық нұсқалар)
- ✅ ESP8266
- ✅ STM32 (Arduino Core-мен)
- ✅ Teensy
- ✅ Басқа Arduino-үйлесімді тақталар

### Қолдау көрсетілетін драйверлер

- ✅ L298N, L293D
- ✅ TB6612FNG
- ✅ DRV8833, DRV8835
- ✅ ZK-5AD, ZK-5AD-V2
- ✅ BTS7960
- ✅ MX1508, MX1919
- ✅ VNH2SP30, VNH5019
- ✅ Кез келген H-Bridge драйверлер

## 📝 Өзгерістер тізімі

### Нұсқа 1.1.0
- ✅ Қазақ тіліне аударма қосылды (README_kz.md)
- ✅ Үш тілді қолдау: English/Русский/Қазақша
- ✅ Құжаттама жақсартылды

### Нұсқа 1.0.9
- ✅ Кітапхана екі тілді қолдайды (English/Русский)
- ✅ Барлық мысалдар екі тілде қолжетімді
- ✅ Құжаттама жақсартылды

### Нұсқа 1.0.7
- ✅ ESP32 Arduino Core 3.x қолдауы (жаңа PWM API)
- ✅ getSpeed() және getMode() әдістері қосылды
- ✅ stop() әдісі қосылды (brake-тен бөлек)
- ✅ Класты көшіруден қорғау
- ✅ Конструктор параметрлерінің реті түзетілді
- ✅ 17 мысал (меканум дөңгелектерді қоса)
- ✅ Жақсартылған құжаттама

### Нұсқа 1.0.6
- Базалық функционалдық
- 4 режим қолдауы
- ESP32 2.x қолдауы

## 🤝 Қолдау және байланыс

- 🌐 **Сайт:** [alash-electronics.kz](https://alash-electronics.kz/)
- 💬 **GitHub:** [Issues](https://github.com/Alash-electronics/AlashMotorControlLite/issues)
- 📧 **Email:** support@alash-electronics.kz
- 📖 **Құжаттама:** [Wiki](https://github.com/Alash-electronics/AlashMotorControlLite/wiki)
- 🎓 **Оқыту:** [Жобалар мен туториалдар](https://alash-electronics.kz/blogs/wiki)

## 📄 Лицензия

MIT License - [LICENSE](LICENSE) файлын қараңыз

## 🌟 Алғыс айту

Бұл кітапхананы пайдаланатын және жақсартатын барлығыңызға рахмет!

---

**Қазақстанда ❤️ жасалған | Alash Engineering**

## Іздеуге арналған кілт сөздер

Arduino motor control, ESP32 motor driver, DC motor library, L298N Arduino, TB6612FNG library, PWM motor control, H-bridge Arduino, motor driver library, Arduino robot, ESP32 robot control, mecanum wheels, dual motor control, Arduino motor shield, ZK-5AD driver, DRV8833 Arduino, BTS7960 library, motor control library, Arduino DC motor, ESP32 PWM, motor speed control, қозғалтқыш басқару, Arduino қозғалтқыш, ESP32 робот
