/*
 * Пример демонстрирует автоматическое ограничение скорости
 *
 * Библиотека автоматически ограничивает значения скорости в диапазоне -100 до 100:
 * - Значения > 100 автоматически устанавливаются как 100
 * - Значения < -100 автоматически устанавливаются как -100
 * - Это защищает от ошибок программирования
 *
 * Этот пример показывает, как работает защита и как правильно
 * использовать диапазон скоростей.
 *
 * Подключение:
 * Arduino: Motor DIR -> Pin 4, PWM -> Pin 3 (режим DIR_PWM)
 * ESP32:   Motor IN1 -> GPIO 32, IN2 -> GPIO 33 (режим PWM_PWM)
 *
 * Код написан для Alash engineering
 */

#include "AlashMotorControlLite.h"

// Выберите режим для вашей платформы:
#if defined(ARDUINO_ARCH_ESP32)
  AlashMotorControlLite motor(PWM_PWM, 32, 33);
#else
  AlashMotorControlLite motor(DIR_PWM, 4, 3);
#endif

void testSpeed(int requestedSpeed) {
  motor.setSpeed(requestedSpeed);
  int actualSpeed = motor.getSpeed();

  Serial.print(F("Запрошено: "));
  if (requestedSpeed >= 0) Serial.print(F(" "));
  Serial.print(requestedSpeed);
  Serial.print(F("  →  Установлено: "));
  if (actualSpeed >= 0) Serial.print(F(" "));
  Serial.print(actualSpeed);

  if (requestedSpeed != actualSpeed) {
    Serial.print(F("  ⚠ ОГРАНИЧЕНО!"));
  } else {
    Serial.print(F("  ✓"));
  }

  Serial.println();
  delay(500);
}

void setup() {
  Serial.begin(115200);
  delay(1000);

  Serial.println(F("╔═══════════════════════════════════════════════╗"));
  Serial.println(F("║  АВТОМАТИЧЕСКОЕ ОГРАНИЧЕНИЕ СКОРОСТИ          ║"));
  Serial.println(F("╚═══════════════════════════════════════════════╝\n"));

  delay(2000);
}

void loop() {
  // ========== ТЕСТИРОВАНИЕ ДОПУСТИМЫХ ЗНАЧЕНИЙ ==========
  Serial.println(F("═══════════════════════════════════════════════"));
  Serial.println(F("ДОПУСТИМЫЕ ЗНАЧЕНИЯ (от -100 до 100)"));
  Serial.println(F("═══════════════════════════════════════════════\n"));

  Serial.println(F("Тестирование положительных скоростей:"));
  testSpeed(0);
  testSpeed(25);
  testSpeed(50);
  testSpeed(75);
  testSpeed(100);

  Serial.println(F("\nТестирование отрицательных скоростей:"));
  testSpeed(0);
  testSpeed(-25);
  testSpeed(-50);
  testSpeed(-75);
  testSpeed(-100);

  motor.stop();
  delay(2000);

  // ========== ТЕСТИРОВАНИЕ ВЫХОДА ЗА ПРЕДЕЛЫ ==========
  Serial.println(F("\n═══════════════════════════════════════════════"));
  Serial.println(F("ВЫХОД ЗА ВЕРХНИЙ ПРЕДЕЛ (+100)"));
  Serial.println(F("═══════════════════════════════════════════════\n"));

  Serial.println(F("Библиотека автоматически ограничит значения:\n"));

  testSpeed(100);   // Норма
  testSpeed(101);   // Будет ограничено до 100
  testSpeed(120);   // Будет ограничено до 100
  testSpeed(150);   // Будет ограничено до 100
  testSpeed(200);   // Будет ограничено до 100
  testSpeed(255);   // Будет ограничено до 100
  testSpeed(500);   // Будет ограничено до 100
  testSpeed(1000);  // Будет ограничено до 100
  testSpeed(32767); // Будет ограничено до 100

  motor.stop();
  delay(2000);

  Serial.println(F("\n═══════════════════════════════════════════════"));
  Serial.println(F("ВЫХОД ЗА НИЖНИЙ ПРЕДЕЛ (-100)"));
  Serial.println(F("═══════════════════════════════════════════════\n"));

  Serial.println(F("Библиотека автоматически ограничит значения:\n"));

  testSpeed(-100);   // Норма
  testSpeed(-101);   // Будет ограничено до -100
  testSpeed(-120);   // Будет ограничено до -100
  testSpeed(-150);   // Будет ограничено до -100
  testSpeed(-200);   // Будет ограничено до -100
  testSpeed(-255);   // Будет ограничено до -100
  testSpeed(-500);   // Будет ограничено до -100
  testSpeed(-1000);  // Будет ограничено до -100
  testSpeed(-32768); // Будет ограничено до -100

  motor.stop();
  delay(2000);

  // ========== ПРАКТИЧЕСКИЕ ПРИМЕРЫ ==========
  Serial.println(F("\n═══════════════════════════════════════════════"));
  Serial.println(F("ПРАКТИЧЕСКИЕ ПРИМЕРЫ"));
  Serial.println(F("═══════════════════════════════════════════════\n"));

  Serial.println(F("Пример 1: Безопасное суммирование скоростей\n"));

  Serial.println(F("int baseSpeed = 80;"));
  Serial.println(F("int boost = 30;"));
  Serial.println(F("motor.setSpeed(baseSpeed + boost);  // 80 + 30 = 110\n"));

  int baseSpeed = 80;
  int boost = 30;

  Serial.print(F("Результат: "));
  testSpeed(baseSpeed + boost);  // Автоматически ограничится до 100

  Serial.println(F("\nБиблиотека защищает от превышения!\n"));
  delay(2000);

  Serial.println(F("Пример 2: Использование процентов\n"));

  Serial.println(F("int userInput = 150;  // Пользователь ввел 150%"));
  Serial.println(F("motor.setSpeed(userInput);\n"));

  int userInput = 150;
  Serial.print(F("Результат: "));
  testSpeed(userInput);  // Автоматически ограничится до 100

  Serial.println(F("\nДаже неправильный ввод безопасен!\n"));
  delay(2000);

  Serial.println(F("Пример 3: Математические вычисления\n"));

  Serial.println(F("int sensorValue = analogRead(A0);  // 0-1023"));
  Serial.println(F("int speed = map(sensorValue, 0, 1023, -200, 200);"));
  Serial.println(F("motor.setSpeed(speed);\n"));

  // Симулируем разные значения датчика
  Serial.println(F("При sensorValue = 0:"));
  testSpeed(-200);  // Ограничится до -100

  Serial.println(F("При sensorValue = 512:"));
  testSpeed(0);     // Середина

  Serial.println(F("При sensorValue = 1023:"));
  testSpeed(200);   // Ограничится до 100

  Serial.println(F("\nДаже при неправильном map() защита работает!\n"));

  motor.stop();
  delay(2000);

  // ========== ГРАНИЧНЫЕ ЗНАЧЕНИЯ ==========
  Serial.println(F("\n═══════════════════════════════════════════════"));
  Serial.println(F("ГРАНИЧНЫЕ ЗНАЧЕНИЯ"));
  Serial.println(F("═══════════════════════════════════════════════\n"));

  Serial.println(F("Точные границы диапазона:\n"));

  testSpeed(99);    // Последнее допустимое перед максимумом
  testSpeed(100);   // Максимум
  testSpeed(101);   // Первое ограничиваемое

  Serial.println();

  testSpeed(-99);   // Последнее допустимое перед минимумом
  testSpeed(-100);  // Минимум
  testSpeed(-101);  // Первое ограничиваемое

  motor.stop();
  delay(2000);

  // ========== РЕКОМЕНДАЦИИ ==========
  Serial.println(F("\n╔═══════════════════════════════════════════════╗"));
  Serial.println(F("║        РЕКОМЕНДАЦИИ ПО ИСПОЛЬЗОВАНИЮ          ║"));
  Serial.println(F("╠═══════════════════════════════════════════════╣"));
  Serial.println(F("║  Диапазон скорости: -100 до 100               ║"));
  Serial.println(F("║                                                ║"));
  Serial.println(F("║  Значения:                                     ║"));
  Serial.println(F("║    -100 = Полная скорость назад               ║"));
  Serial.println(F("║      -50 = 50% назад                          ║"));
  Serial.println(F("║        0 = Остановка                          ║"));
  Serial.println(F("║      +50 = 50% вперед                         ║"));
  Serial.println(F("║     +100 = Полная скорость вперед             ║"));
  Serial.println(F("║                                                ║"));
  Serial.println(F("║  Защита работает автоматически:               ║"));
  Serial.println(F("║    • Значения > 100 → устанавливается 100     ║"));
  Serial.println(F("║    • Значения < -100 → устанавливается -100   ║"));
  Serial.println(F("║                                                ║"));
  Serial.println(F("║  Хорошая практика:                            ║"));
  Serial.println(F("║    ✓ Используйте constrain() перед setSpeed() ║"));
  Serial.println(F("║    ✓ Проверяйте map() для корректности       ║"));
  Serial.println(F("║    ✓ Валидируйте пользовательский ввод       ║"));
  Serial.println(F("║                                                ║"));
  Serial.println(F("║  Но даже если забудете - защита сработает!   ║"));
  Serial.println(F("╚═══════════════════════════════════════════════╝\n"));

  // ========== КОД-ПРИМЕР ==========
  Serial.println(F("╔═══════════════════════════════════════════════╗"));
  Serial.println(F("║              ПРИМЕРЫ КОДА                     ║"));
  Serial.println(F("╚═══════════════════════════════════════════════╝\n"));

  Serial.println(F("// Правильно: с проверкой"));
  Serial.println(F("int speed = analogRead(A0);"));
  Serial.println(F("speed = map(speed, 0, 1023, -100, 100);"));
  Serial.println(F("motor.setSpeed(speed);\n"));

  Serial.println(F("// Тоже работает: библиотека защитит"));
  Serial.println(F("int speed = analogRead(A0);"));
  Serial.println(F("speed = map(speed, 0, 1023, -200, 200);"));
  Serial.println(F("motor.setSpeed(speed);  // Ограничится автоматически\n"));

  Serial.println(F("// С дополнительной защитой:"));
  Serial.println(F("int speed = getUserInput();"));
  Serial.println(F("speed = constrain(speed, -100, 100);"));
  Serial.println(F("motor.setSpeed(speed);  // Двойная защита\n"));

  Serial.println(F("Цикл завершен. Повтор через 5 секунд...\n\n"));
  delay(5000);
}
