/*
 * Пример демонстрирует разницу между методами stop() и brake()
 *
 * stop()  - Свободная остановка (coast)
 *           Мотор просто отключается и останавливается по инерции
 *           Используется для плавной остановки без нагрузки на механику
 *
 * brake() - Активное торможение
 *           Мотор активно тормозит, создавая сопротивление
 *           Используется для быстрой остановки или удержания позиции
 *
 * Подключение:
 * Arduino: Motor DIR -> Pin 4, PWM -> Pin 3 (режим DIR_PWM)
 * ESP32:   Motor IN1 -> GPIO 32, IN2 -> GPIO 33 (режим PWM_PWM)
 *
 * ВАЖНО: Разница наиболее заметна при высоких скоростях и большой инерции!
 *
 * Код написан для Alash engineering
 */

#include "AlashMotorControlLite.h"

// Выберите режим для вашей платформы:
#if defined(ARDUINO_ARCH_ESP32)
  // ESP32 - режим PWM_PWM
  AlashMotorControlLite motor(PWM_PWM, 32, 33);
#else
  // Arduino - режим DIR_PWM
  AlashMotorControlLite motor(DIR_PWM, 4, 3);
#endif

void setup() {
  Serial.begin(115200);
  delay(1000);

  Serial.println("╔═══════════════════════════════════════════════╗");
  Serial.println("║    СРАВНЕНИЕ stop() VS brake()               ║");
  Serial.println("╚═══════════════════════════════════════════════╝\n");

  Serial.println("Этот пример показывает разницу между двумя способами остановки мотора.\n");
  delay(2000);
}

void loop() {
  // ========== ТЕСТ 1: Свободная остановка stop() ==========
  Serial.println("═══════════════════════════════════════════════");
  Serial.println("ТЕСТ 1: Свободная остановка - stop()");
  Serial.println("═══════════════════════════════════════════════");
  Serial.println("Мотор разгоняется до 100%...");

  motor.setSpeed(100);
  delay(2000);  // Даем мотору разогнаться

  Serial.println("→ Вызов motor.stop()");
  Serial.println("  Мотор отключается и останавливается по инерции");
  Serial.println("  Наблюдайте: плавная остановка, долгий выбег\n");

  motor.stop();  // Свободная остановка
  delay(4000);   // Ждем полной остановки

  Serial.println("Пауза 2 секунды...\n");
  delay(2000);

  // ========== ТЕСТ 2: Активное торможение brake() ==========
  Serial.println("═══════════════════════════════════════════════");
  Serial.println("ТЕСТ 2: Активное торможение - brake()");
  Serial.println("═══════════════════════════════════════════════");
  Serial.println("Мотор разгоняется до 100%...");

  motor.setSpeed(100);
  delay(2000);  // Даем мотору разогнаться

  Serial.println("→ Вызов motor.brake()");
  Serial.println("  Мотор активно тормозит");
  Serial.println("  Наблюдайте: быстрая остановка, короткий выбег\n");

  motor.brake();  // Активное торможение
  delay(4000);    // Ждем полной остановки

  Serial.println("Пауза 2 секунды...\n");
  delay(2000);

  // ========== СРАВНЕНИЕ В ОБРАТНОМ НАПРАВЛЕНИИ ==========
  Serial.println("═══════════════════════════════════════════════");
  Serial.println("ТЕСТ 3: Сравнение при движении назад");
  Serial.println("═══════════════════════════════════════════════\n");

  // Движение назад с stop()
  Serial.println("Назад 100% → stop()");
  motor.setSpeed(-100);
  delay(2000);
  motor.stop();
  delay(3000);

  Serial.println("Пауза...\n");
  delay(2000);

  // Движение назад с brake()
  Serial.println("Назад 100% → brake()");
  motor.setSpeed(-100);
  delay(2000);
  motor.brake();
  delay(3000);

  // ========== РЕКОМЕНДАЦИИ ==========
  Serial.println("\n╔═══════════════════════════════════════════════╗");
  Serial.println("║           КОГДА ЧТО ИСПОЛЬЗОВАТЬ?             ║");
  Serial.println("╠═══════════════════════════════════════════════╣");
  Serial.println("║  stop() - Свободная остановка:                ║");
  Serial.println("║    ✓ Плавная остановка без рывков             ║");
  Serial.println("║    ✓ Меньше нагрузка на механику              ║");
  Serial.println("║    ✓ Экономия энергии                         ║");
  Serial.println("║    ✗ Медленная остановка                      ║");
  Serial.println("║                                                ║");
  Serial.println("║  brake() - Активное торможение:               ║");
  Serial.println("║    ✓ Быстрая остановка                        ║");
  Serial.println("║    ✓ Удержание позиции под нагрузкой          ║");
  Serial.println("║    ✓ Точная остановка                         ║");
  Serial.println("║    ✗ Больше нагрузка на механику              ║");
  Serial.println("║    ✗ Больше потребление энергии               ║");
  Serial.println("╚═══════════════════════════════════════════════╝\n");

  Serial.println("Цикл завершен. Повтор через 5 секунд...\n\n");
  delay(5000);
}
