"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const serial_service_impl_1 = require("./../../node/serial/serial-service-impl");
const typemoq_1 = require("typemoq");
const sinon_1 = require("sinon");
const sinonChai = require("sinon-chai");
const chai_1 = require("chai");
chai_1.use(sinonChai);
const protocol_1 = require("../../common/protocol");
describe('SerialServiceImpl', () => {
    let subject;
    let logger;
    let serialClientProvider;
    let webSocketService;
    beforeEach(() => {
        logger = typemoq_1.Mock.ofType();
        logger.setup((b) => b.info(typemoq_1.It.isAnyString()));
        logger.setup((b) => b.warn(typemoq_1.It.isAnyString()));
        logger.setup((b) => b.error(typemoq_1.It.isAnyString()));
        serialClientProvider = typemoq_1.Mock.ofType();
        webSocketService = typemoq_1.Mock.ofType();
        subject = new serial_service_impl_1.SerialServiceImpl(logger.object, serialClientProvider.object, webSocketService.object);
    });
    context('when a serial connection is requested', () => {
        const sandbox = sinon_1.createSandbox();
        beforeEach(() => {
            subject.uploadInProgress = false;
            sandbox.spy(subject, 'disconnect');
            sandbox.spy(subject, 'updateWsConfigParam');
        });
        afterEach(function () {
            sandbox.restore();
        });
        context('and an upload is in progress', () => {
            beforeEach(async () => {
                subject.uploadInProgress = true;
            });
            it('should not change the connection status', async () => {
                await subject.connectSerialIfRequired();
                chai_1.expect(subject.disconnect).to.have.callCount(0);
            });
        });
        context('and there is no upload in progress', () => {
            beforeEach(async () => {
                subject.uploadInProgress = false;
            });
            context('and there are 0 attached ws clients', () => {
                it('should disconnect', async () => {
                    await subject.connectSerialIfRequired();
                    chai_1.expect(subject.disconnect).to.have.been.calledOnce;
                });
            });
            context('and there are > 0 attached ws clients', () => {
                beforeEach(() => {
                    webSocketService
                        .setup((b) => b.getConnectedClientsNumber())
                        .returns(() => 1);
                });
                it('should not call the disconenct', async () => {
                    await subject.connectSerialIfRequired();
                    chai_1.expect(subject.disconnect).to.have.callCount(0);
                });
            });
        });
    });
    context('when a disconnection is requested', () => {
        const sandbox = sinon_1.createSandbox();
        beforeEach(() => { });
        afterEach(function () {
            sandbox.restore();
        });
        context('and a serialConnection is not set', () => {
            it('should return a NOT_CONNECTED status', async () => {
                const status = await subject.disconnect();
                chai_1.expect(status).to.be.equal(protocol_1.Status.NOT_CONNECTED);
            });
        });
        context('and a serialConnection is set', async () => {
            beforeEach(async () => {
                sandbox.spy(subject, 'updateWsConfigParam');
                await subject.disconnect();
            });
            it('should dispose the serialConnection', async () => {
                const serialConnectionOpen = await subject.isSerialPortOpen();
                chai_1.expect(serialConnectionOpen).to.be.false;
            });
            it('should call updateWsConfigParam with disconnected status', async () => {
                chai_1.expect(subject.updateWsConfigParam).to.be.calledWith({
                    connected: false,
                });
            });
        });
    });
    context('when a new config is passed in', () => {
        const sandbox = sinon_1.createSandbox();
        beforeEach(async () => {
            subject.uploadInProgress = false;
            webSocketService
                .setup((b) => b.getConnectedClientsNumber())
                .returns(() => 1);
            serialClientProvider
                .setup((b) => b.client())
                .returns(async () => {
                return {
                    streamingOpen: () => {
                        return {
                            on: (str, cb) => { },
                            write: (chunk, cb) => {
                                cb();
                            },
                            cancel: () => { },
                        };
                    },
                };
            });
            sandbox.spy(subject, 'disconnect');
            await subject.setSerialConfig({
                board: { name: 'test' },
                port: { id: 'test|test', address: 'test', addressLabel: 'test', protocol: 'test', protocolLabel: 'test' },
            });
        });
        afterEach(function () {
            sandbox.restore();
            subject.dispose();
        });
        it('should disconnect from previous connection', async () => {
            chai_1.expect(subject.disconnect).to.be.called;
        });
        it('should create the serialConnection', async () => {
            const serialConnectionOpen = await subject.isSerialPortOpen();
            chai_1.expect(serialConnectionOpen).to.be.true;
        });
    });
});
//# sourceMappingURL=serial-service-impl.test.js.map