"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const monitor_utils_1 = require("../../browser/serial/monitor/monitor-utils");
const mockdate_1 = require("mockdate");
const date = new Date();
const testLines = [
    {
        messages: ['Hello'],
        expected: { lines: [{ message: 'Hello', lineLen: 5 }], charCount: 5 },
    },
    {
        messages: ['Hello', 'Dog!'],
        expected: { lines: [{ message: 'HelloDog!', lineLen: 9 }], charCount: 9 },
    },
    {
        messages: ['Hello\n', 'Dog!'],
        expected: {
            lines: [
                { message: 'Hello\n', lineLen: 6 },
                { message: 'Dog!', lineLen: 4 },
            ],
            charCount: 10,
        },
    },
    {
        messages: ['Dog!'],
        prevLines: { lines: [{ message: 'Hello\n', lineLen: 6 }], charCount: 6 },
        expected: {
            lines: [
                { message: 'Hello\n', lineLen: 6 },
                { message: 'Dog!', lineLen: 4 },
            ],
            charCount: 10,
        },
    },
    {
        messages: [' Dog!\n', " Who's a good ", 'boy?\n', "You're a good boy!"],
        prevLines: { lines: [{ message: 'Hello', lineLen: 5 }], charCount: 5 },
        expected: {
            lines: [
                { message: 'Hello Dog!\n', lineLen: 11 },
                { message: " Who's a good boy?\n", lineLen: 19 },
                { message: "You're a good boy!", lineLen: 8 },
            ],
            charCount: 48,
        },
        expectedTruncated: {
            maxCharacters: 20,
            charCount: 20,
            lines: [
                { message: '?\n', lineLen: 2 },
                { message: "You're a good boy!", lineLen: 8 },
            ],
        },
    },
    {
        messages: ['boy?\n', "You're a good boy!"],
        prevLines: {
            lines: [
                { message: 'Hello Dog!\n', lineLen: 11 },
                { message: " Who's a good ", lineLen: 14 },
            ],
            charCount: 25,
        },
        expected: {
            lines: [
                { message: 'Hello Dog!\n', lineLen: 11 },
                { message: " Who's a good boy?\n", lineLen: 19 },
                { message: "You're a good boy!", lineLen: 8 },
            ],
            charCount: 48,
        },
        expectedTruncated: {
            maxCharacters: 20,
            charCount: 20,
            lines: [
                { message: '?\n', lineLen: 2 },
                { message: "You're a good boy!", lineLen: 8 },
            ],
        },
    },
    {
        messages: ["Who's a good boy?\n", 'Yo'],
        prevLines: {
            lines: [{ message: 'Hello Dog!\n', lineLen: 11 }],
            charCount: 11,
        },
        expected: {
            lines: [
                { message: 'Hello Dog!\n', lineLen: 11 },
                { message: "Who's a good boy?\n", lineLen: 18 },
                { message: 'Yo', lineLen: 2 },
            ],
            charCount: 31,
        },
        expectedTruncated: {
            maxCharacters: 20,
            charCount: 20,
            lines: [
                { message: "Who's a good boy?\n", lineLen: 18 },
                { message: 'Yo', lineLen: 2 },
            ],
        },
    },
];
testLines.forEach((t) => {
    var _a;
    return [...t.expected.lines, ...(((_a = t.prevLines) === null || _a === void 0 ? void 0 : _a.lines) || [])].forEach((l) => (l.timestamp = date));
});
describe('Monitor Utils', () => {
    beforeEach(() => {
        mockdate_1.set(date);
    });
    afterEach(() => {
        mockdate_1.reset();
    });
    testLines.forEach((testLine) => {
        context('when converting messages', () => {
            it('should give the right result', () => {
                var _a, _b, _c;
                const [newLines, addedCharCount] = monitor_utils_1.messagesToLines(testLine.messages, (_a = testLine.prevLines) === null || _a === void 0 ? void 0 : _a.lines, (_b = testLine.prevLines) === null || _b === void 0 ? void 0 : _b.charCount);
                newLines.forEach((line, index) => {
                    chai_1.expect(line.message).to.equal(testLine.expected.lines[index].message);
                    chai_1.expect(line.timestamp).to.deep.equal(testLine.expected.lines[index].timestamp);
                });
                chai_1.expect(addedCharCount).to.equal(testLine.expected.charCount);
                const [truncatedLines, totalCharCount] = monitor_utils_1.truncateLines(newLines, addedCharCount, (_c = testLine.expectedTruncated) === null || _c === void 0 ? void 0 : _c.maxCharacters);
                let charCount = 0;
                if (testLine.expectedTruncated) {
                    truncatedLines.forEach((line, index) => {
                        var _a;
                        chai_1.expect(line.message).to.equal((_a = testLine.expectedTruncated) === null || _a === void 0 ? void 0 : _a.lines[index].message);
                        charCount += line.message.length;
                    });
                    chai_1.expect(totalCharCount).to.equal(charCount);
                }
            });
        });
    });
});
//# sourceMappingURL=monitor-utils.test.js.map